﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <windows.h>
#include <dolphin/hio.h>
#include <nw4r/mcs/hioRingBuffer.h>
#include <nw4r/mcs/hioNegotiate.h>
#include <vcclr.h>
#include "PCIOCommDeviceException.h"
#include "pcio.h"

extern HANDLE hMutex;

/*
    #define HIOEnumDevicesStr   "HIOEnumDevices"
    #define HIOInitStr          "HIOInit"
    #define HIOReadMailboxStr   "HIOReadMailbox"
    #define HIOWriteMailboxStr  "HIOWriteMailbox"
    #define HIOReadStr          "HIORead"
    #define HIOWriteStr         "HIOWrite"
    #define HIOReadAsyncStr     "HIOReadAsync"
    #define HIOWriteAsyncStr    "HIOWriteAsync"
    #define HIOReadStatusStr    "HIOReadStatus"
    #define HIOInit2Str         "HIOInit2"
    #define HIOExitStr          "HIOExit"
*/

using namespace System;
using namespace System::Collections::Generic;
using namespace System::Runtime::InteropServices;
using namespace System::Diagnostics;
using namespace System::IO;

namespace Nintendo
{

namespace McsServer
{

    public ref class PCIOCommDevice : public ICommDevice
    {
    public:
#if _DEBUG
        literal String^ DllName = "hioD.dll";
#else
        literal String^ DllName = "hio.dll";
#endif

    private:
        // typedef BOOL (WINAPI*HIOEnumCallback)( s32 chan );
        delegate bool DelegateHIOEnumCallback(s32 chan);

        static List<int>^           sFindChannelList_;

        bool                        bTargetConnect_;

        // アンマネージリソースの関連
        bool                        bAttach_;
        nw4r::mcs::HioRingBuffer*   pReader_;
        nw4r::mcs::HioRingBuffer*   pWriter_;
        IntPtr                      hTempBuf_;

    private:
        static bool IsFindHIOModule()
        {
/*
            if (shHIOModule_)
            {
                return true;
            }

            // パス指定なしでアクセス可能か?
            {
                pin_ptr<const wchar_t> wchDllPath = PtrToStringChars(DllName);
                shHIOModule_ = LoadLibrary(wchDllPath);
            }
            if (! shHIOModule_)
            {
                // Dolphin SDKの環境変数が設定されていてそこから辿れるか?
                String^ dllPath = Path::Combine(Path::Combine(Environment::ExpandEnvironmentVariables("%DOLPHIN_ROOT%"), "X86\\bin"), gcnew String(DllName));
                pin_ptr<const wchar_t> wchDllPath = PtrToStringChars(dllPath);
                shHIOModule_ = LoadLibrary(wchDllPath);
            }

            if (! shHIOModule_)
            {
                return false;
            }

            if ( 0 == (spfHIOEnumDevices_  = reinterpret_cast<HIOEnumDevicesType >(GetProcAddress(shHIOModule_, HIOEnumDevicesStr )))
              || 0 == (spfHIOInit_         = reinterpret_cast<HIOInitType        >(GetProcAddress(shHIOModule_, HIOInitStr        )))
              || 0 == (spfHIOReadMailbox_  = reinterpret_cast<HIOReadMailboxType >(GetProcAddress(shHIOModule_, HIOReadMailboxStr )))
              || 0 == (spfHIOWriteMailbox_ = reinterpret_cast<HIOWriteMailboxType>(GetProcAddress(shHIOModule_, HIOWriteMailboxStr)))
              || 0 == (spfHIORead_         = reinterpret_cast<HIOReadType        >(GetProcAddress(shHIOModule_, HIOReadStr        )))
              || 0 == (spfHIOWrite_        = reinterpret_cast<HIOWriteType       >(GetProcAddress(shHIOModule_, HIOWriteStr       )))
              || 0 == (spfHIOReadAsync_    = reinterpret_cast<HIOReadAsyncType   >(GetProcAddress(shHIOModule_, HIOReadAsyncStr   )))
              || 0 == (spfHIOWriteAsync_   = reinterpret_cast<HIOWriteAsyncType  >(GetProcAddress(shHIOModule_, HIOWriteAsyncStr  )))
              || 0 == (spfHIOReadStatus_   = reinterpret_cast<HIOReadStatusType  >(GetProcAddress(shHIOModule_, HIOReadStatusStr  )))
              || 0 == (spfHIOInit2_        = reinterpret_cast<HIOInit2Type       >(GetProcAddress(shHIOModule_, HIOInit2Str       )))
              || 0 == (spfHIOExit_         = reinterpret_cast<HIOExitType        >(GetProcAddress(shHIOModule_, HIOExitStr        )))
            )
            {
                FreeLibrary(shHIOModule_);
                shHIOModule_ = 0;
                return false;
            }
*/

            nw4r::mcs::HioRingBuffer::SetHIOFuncAddress(spfHIORead_, spfHIOWrite_);
            return true;
        }

        /// <summary>
        /// HIOEnumDevicesに渡すコールバック関数
        /// </summary>
        /// <param name="chan"></param>
        /// <returns></returns>
        static bool EnumDeviceCallback(int chan)
        {
            sFindChannelList_->Add(chan);
            return true;    // USB I/Fの列挙を継続
        }

    public:
        static bool SearchDevice()
        {
//          usbNo = 0;

            if (! IsFindHIOModule())
            {
                return false;
            }
/*
            sFindChannelList_ = gcnew List<int>();

            DelegateHIOEnumCallback^ enumDeviceCallback = gcnew DelegateHIOEnumCallback(EnumDeviceCallback);
            if (! spfHIOEnumDevices_(HIOEnumCallback(Marshal::GetFunctionPointerForDelegate(enumDeviceCallback).ToPointer())))
            {
                throw gcnew PCIOCommDeviceException("HIOEnumDevices() failed.");
            }

            if (sFindChannelList_->Count == 0)   // デバイスが1つも見つからなかった場合
            {
                return false;
            }

            usbNo = sFindChannelList_[0];
            sFindChannelList_ = nullptr;
*/
            if (! spfHIOSearchDevices_() )  // PC_Viewer が起動していなかった場合
            {
                return false;
            }
            return true;
        }

    public:
        PCIOCommDevice()
            :   bTargetConnect_(false),
                bAttach_(false),
                pReader_(0),
                pWriter_(0),
                hTempBuf_(IntPtr::Zero)
        {
            using namespace nw4r::mcs;

            try
            {
                InitHIO();
                bAttach_ = true;

                if (0 != (ReadStatus() & HIO_STATUS_RX))   // 未読メールがあれば呼んでおく
                {
                    u32 mailWord = ReadMailbox();
                    Debug::WriteLine(String::Format("clear mail box. drop mail {0:X8}", mailWord));

                }

                WriteMailbox(detail::HioNegotiate::RebootCode);
                Debug::WriteLine("Write reboot mail.");

            }
            catch (HIOFunctionException^ ex)
            {
                // コンストラクタが失敗するので、元に戻す
                DeviceFinalize();
                throw gcnew PCIOCommDeviceException(ex->Message);
            }

            const u32        READ_TRANSBUF_SIZE  = 0x08000;
            const u32        WRITE_TRANSBUF_SIZE = 0x10000;

            hTempBuf_ = Marshal::AllocHGlobal(HioRingBuffer::CopyAlignment + READ_TRANSBUF_SIZE + WRITE_TRANSBUF_SIZE);
            IntPtr readBuffer = RoundUp(hTempBuf_, HioRingBuffer::CopyAlignment);
            IntPtr writeBuffer = IntPtr(readBuffer.ToInt64() + READ_TRANSBUF_SIZE);

            pReader_ = new HioRingBuffer(
                HioRingBuffer::CmnMemBaseAddress                     , READ_TRANSBUF_SIZE,  readBuffer.ToPointer());

            pWriter_ = new HioRingBuffer(
                HioRingBuffer::CmnMemBaseAddress + READ_TRANSBUF_SIZE, WRITE_TRANSBUF_SIZE, writeBuffer.ToPointer());
        }

        virtual void Negotiate()
        {
            using namespace nw4r::mcs;

            if (! bAttach_)
            {
                return;
            }

            try
            {
                if (0 != (ReadStatus() & HIO_STATUS_RX))   // 未読メールがある
                {
                    switch (ReadMailbox())
                    {
                    case detail::HioNegotiate::RebootCode:
                        if (bTargetConnect_)
                        {
                            Debug::WriteLine("client reset.");
                        }
                        bTargetConnect_ = false;    // 一旦ネゴシエーション状態を解除

                        // 共有メモリ上にリングバッファを構築
                        pReader_->EnablePort();
                        pWriter_->EnablePort();
                        pReader_->InitBuffer();
                        pWriter_->InitBuffer();

                        // 共有メモリを初期化したことを伝える
                        WriteMailbox(detail::HioNegotiate::InitBufferCode);
                        break;

                    case detail::HioNegotiate::AckCode:
                        Debug::WriteLine("Negotiation end.");
                        bTargetConnect_ = true;
                        break;
                    }
                }
            }
            catch (HIOFunctionException^ ex)
            {
                throw gcnew PCIOCommDeviceException(ex->Message);
            }
        }

        property String^ Name
        {
            virtual String^ get()
            {
                return "PC Viewer";
            }
        }

        ~PCIOCommDevice()
        {
            if (bTargetConnect_)
            {
                try
                {
                    WriteMailbox(nw4r::mcs::detail::HioNegotiate::Disconnect);
                }
                catch (HIOFunctionException^ e)
                {
                    Debug::WriteLine(e->Message);
                }
                bTargetConnect_ = false;
            }

            // アンマネージリソースの解放
            this->!PCIOCommDevice();
        }

        !PCIOCommDevice()
        {
            delete pWriter_;
            pWriter_ = 0;

            delete pReader_;
            pReader_ = 0;

            if (hTempBuf_ != IntPtr::Zero)
            {
                Marshal::FreeHGlobal(hTempBuf_);
                hTempBuf_ = IntPtr::Zero;
            }

            DeviceFinalize();
        }

    private:
        void DeviceFinalize()
        {
            if (bAttach_)
            {
                spfHIOExit_();
                bAttach_ = false;
            }
        }

    public:
        property bool IsAttach
        {
            virtual bool get()
            {
                return bAttach_;
            }
        }

        property bool IsTargetConnect
        {
            virtual bool get()
            {
                return bTargetConnect_;
            }
        }

        virtual bool Read()
        {
            bool bAvailable;
            if (! pReader_->Read(&bAvailable))
            {
                throw gcnew PCIOCommDeviceException("Read() failed.");
            }
            return bAvailable;
        }

        virtual MessageData^ GetMessage()
        {
            u32 wkChannel, wkSize;
            void *const msgPtr = pReader_->GetMessage(&wkChannel, &wkSize);

            if (msgPtr == NULL || wkSize == 0)
            {
                return nullptr;
            }

            Int32 channel = Int32(wkChannel);
            Int32 size = Int32(wkSize);

            cli::array<u8>^ buff = gcnew array<u8>(size);
            pin_ptr<u8> ptr = &buff[0];
            memcpy(ptr, msgPtr, size);

            return gcnew MessageData(channel, buff, 0, size);
        }

        virtual int GetWritableBytes(bool withUpdate)
        {
            u32 writableBytes;
            if (pWriter_->GetWritableBytes(&writableBytes, withUpdate))
            {
                return int(writableBytes);
            }
            else
            {
                throw gcnew PCIOCommDeviceException("GetWritableBytes() failed.");
            }
        }

        virtual void Write(int channel, array<u8>^ buf, int offset, int size)
        {
            if (buf == nullptr)
            {
                throw gcnew ArgumentNullException();
            }

            if (offset < 0 || size < 0 || offset + size > buf->Length)
            {
                throw gcnew ArgumentOutOfRangeException();
            }

            pin_ptr<Byte> pinBuf = &buf[offset];
            if (!pWriter_->Write(channel, pinBuf, size))
            {
                throw gcnew PCIOCommDeviceException("Write() failed.");
            }
        }

        virtual u32 GetWriteBufferSize()
        {
            return pWriter_->GetBufferSize();
        }

        property bool IsPort
        {
            bool get()
            {
                return pReader_->IsPort() && pWriter_->IsPort();
            }
        }

    private:
        static void InitHIO()
        {
            if (! spfHIOInit_())
            {
                throw gcnew HIOFunctionException(String::Format("PC HIOInit() failed."));
            }
        }

        static u32 ReadMailbox()
        {
            u32 mailWord;
            if (! spfHIOReadMailbox_(&mailWord))
            {
                throw gcnew HIOFunctionException("HIOReadMailbox() failed.");
            }

            return mailWord;
        }

        static void WriteMailbox(u32 mailWord)
        {
            if (! spfHIOWriteMailbox_(mailWord))
            {
                throw gcnew HIOFunctionException("HIOWriteMailbox() failed.");
            }
        }

        static u32 ReadStatus()
        {
            u32 hioStatus;
            if (! spfHIOReadStatus_(&hioStatus))
            {
                throw gcnew HIOFunctionException("HIOReadStatus() failed.");
            }
            return hioStatus;
        }

        static IntPtr RoundUp(
            IntPtr value,
            u32    alignment
        )
        {
            Int64 mask = alignment - 1;
            return IntPtr((value.ToInt64() + mask) & ~mask);
        }

    private:
        ref class HIOFunctionException : public MCSException
        {
        public:
            HIOFunctionException()
            {
            }

            HIOFunctionException(String^ message)
                : MCSException(message)
            {
            }
        };

    };

}   // namespace McsServer

}   // namespace Nintendo
