﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <autoTestAssistTool/HidInputerLibrary.h>

#include <thread>
#include "IOPortSender.h"
#include "MultiControllerManager.h"
#include "MultiTouchManager.h"


namespace {
    IoPortSender &sender = IoPortSender::GetInstance();
    MultiControllerManager &multiController = MultiControllerManager::GetInstance(&sender);
    MultiTouchManager &multiTouchManager = MultiTouchManager::GetInstance(&sender);

    bool            IsActiveFlushThread = false;
    bool            IsHidInputerInitialized = false;
    std::unique_ptr<std::thread> IoFlushThreadObject = NULL;

    // InputDirector フラッシュスレッド
    void IoFlushThread()
    {
        while (IsActiveFlushThread)
        {
            sender.Flush();
            Sleep(static_cast<unsigned int>(1000.0 / 120));
        }
    }

    // フラッシュスレッド終了
    void IoFlushEnd()
    {
        IsActiveFlushThread = false;

        if (IoFlushThreadObject != NULL)
        {
            IoFlushThreadObject->join();
            IoFlushThreadObject = NULL;
        }
        IsHidInputerInitialized = false;
    }

    // フラッシュスレッド開始
    HidInputerResult IoFlushBegin( const char *serialNumber )
    {
        IoFlushEnd();

        HidInputerResult result;
        if (serialNumber == nullptr)
        {
            result = sender.SetPortNumberForLocal();
        }
        else
        {
            result = sender.SetPortNumberForTarget(serialNumber);
        }

        // ターゲット設定に失敗した場合は、エラー値を返し未初期化とする。
        if (result != HidInputerResult::HidInputerResult_Success)
        {
            return result;
        }

        // コントローラー初期化
        multiController.InitializeControllerConnection();

        IsActiveFlushThread = true;
        IsHidInputerInitialized = true;

        if (IoFlushThreadObject == NULL)
        {
            IoFlushThreadObject.reset(new std::thread(IoFlushThread));
        }

        return HidInputerResult::HidInputerResult_Success;
    }

    bool IsInitialized()
    {
        if (!IsHidInputerInitialized)
        {
            ERRMSG("Initialization is not completed.");
            return false;
        }
        return true;
    }
}

// 初期化・終了処理
HidInputerResult __cdecl Initialize( const SerialNumberString& serialNumber )
{
    if (serialNumber.value == nullptr)
    {
        return HidInputerResult::HidInputerResult_TargetNotFound;
    }

    return IoFlushBegin( serialNumber.value );
}

HidInputerResult __cdecl InitializeForLocal()
{
    return IoFlushBegin( nullptr );
}

void __cdecl Finalize()
{
    IoFlushEnd();

    // hid 入力状態をクリア
    multiController.ClearAllControllerState();
    multiTouchManager.ResetTouchState();
    // クリア情報をフラッシュ
    sender.Flush();
}

/*---------------------------------------------------------------------
/ HID 入力情報の記録
/--------------------------------------------------------------------*/
HidInputerResult __cdecl BeginHidInputRecording()
{
    if (!IsInitialized())
    {
        return  HidInputerResult::HidInputerResult_NotInitialized;
    }

    // 記録済みの情報をリセットしておく
    HidInputerResult result = sender.ClearHidRecord();
    if (result != HidInputerResult::HidInputerResult_Success)
    {
        return result;
    }

    return sender.BeginHidInputRecording();
}
HidInputerResult __cdecl EndHidInputRecording()
{
    if (!IsInitialized())
    {
        return  HidInputerResult::HidInputerResult_NotInitialized;
    }
    return sender.EndHidInputRecording();
}

HidInputerResult __cdecl ExportHidInputRecordsWithPythonScript( const char* exportFilename )
{
    if (!IsInitialized())
    {
        return HidInputerResult::HidInputerResult_NotInitialized;
    }

    return sender.ExportHidInputRecords( exportFilename );
}

/*---------------------------------------------------------------------
/ コントローラー接続解除
/--------------------------------------------------------------------*/
ControllerResult __cdecl AddController(int *pOutControllerId )
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.AddController( pOutControllerId  );
}

ControllerResult AddControllerWithDeviceInfo(
    int *pOutControllerId,
    const ControllerDeviceInfo& deviceInfo)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.AddControllerWithDeviceInfo(pOutControllerId, deviceInfo);
}

ControllerResult __cdecl AddControllerForId(int controllerId)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.AddController(controllerId);
}

ControllerResult __cdecl ReleaseController(int controllerId)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.ReleaseController(controllerId);
}

ControllerResult __cdecl AddDebugController(int *pOutControllerId )
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.AddDebugController(pOutControllerId );
}

/*---------------------------------------------------------------------
/ コントローラー情報取得
/--------------------------------------------------------------------*/
ControllerResult __cdecl GetConnectedControllerCount(int *pOutCount)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }

    return multiController.GetConnectedControllerCount(pOutCount);
}

ControllerResult __cdecl GetConnectedControllerIds(int *pOutCount, int pOutIdsArray[], int arrayLength)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }

    return multiController.GetConnectedControllerIds(pOutCount, pOutIdsArray, arrayLength);
}

ControllerResult __cdecl GetConnectedControllerDeviceInfo( ControllerDeviceInfo* pOutDeviceInfo,
    int controllerId)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }

    return multiController.GetConnectedControllerDeviceInfo(pOutDeviceInfo, controllerId);
}

ControllerResult __cdecl GetConnectedDebugControllerCount(int *pOutCount)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.GetConnectedDebugControllerCount(pOutCount);
}

ControllerResult __cdecl GetConnectedDebugControllerIds(int *pOutCount, int pOutIdsArray[], int arrayLength)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.GetConnectedDebugControllerIds(pOutCount, pOutIdsArray, arrayLength);
}

/*---------------------------------------------------------------------
/ コントローラーボタン操作
/--------------------------------------------------------------------*/
ControllerResult __cdecl PressButton(int controllerId, Button button)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.PressButton(controllerId, button);
}

ControllerResult __cdecl ReleaseButton(int controllerId, Button button)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.ReleaseButton(controllerId, button);
}

ControllerResult __cdecl PushButton(int controllerId, Button button, int waitPress)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.PushButton(controllerId, button, waitPress);
}

ControllerResult __cdecl PressMultiButton(int controllerId, Button buttonList[], int buttonCount)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.PressMultiButton(controllerId, buttonList, buttonCount);
}

ControllerResult __cdecl ReleaseMultiButton(int controllerId, Button buttonList[], int buttonCount)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.ReleaseMultiButton(controllerId, buttonList, buttonCount);
}
ControllerResult __cdecl PushMultiButton(int controllerId, Button buttonList[], int buttonCount, int waitPress)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.PushMultiButton(controllerId, buttonList, buttonCount, waitPress);
}

ControllerResult __cdecl PushMultiSeq(int controllerId, Button buttonList[], int buttonCount, int waitPress, int waitSpan)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return  multiController.PushMultiSeq(controllerId, buttonList, buttonCount, waitPress, waitSpan);
}

ControllerResult __cdecl ResetControllerState(int controllerId)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.ResetControllerState(controllerId);
}

ControllerResult __cdecl SetControllerState(int controllerId, Button buttonList[], int buttonCount, Stick stickList[], float stickDegreeList[], float stickPowerList[], int stickCount)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.SetControllerState(controllerId, buttonList, buttonCount, stickList, stickDegreeList, stickPowerList, stickCount);
}


/*---------------------------------------------------------------------
/ コントローラーアナログスティック操作
/--------------------------------------------------------------------*/
ControllerResult __cdecl HoldAnalogStick(int controllerId, Stick stick, float degree, float power)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.HoldAnalogStick(controllerId, stick, degree, power);
}

ControllerResult __cdecl ReleaseAnalogStick(int controllerId, Stick stick)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.ReleaseAnalogStick(controllerId, stick);
}

ControllerResult __cdecl TiltAnalogStick(int controllerId, Stick stick, float degree, float power, int waitHold)
{
    if (!IsInitialized())
    {
        return ControllerResult::ControllerResult_NotInitialized;
    }
    return multiController.TiltAnalogStick(controllerId, stick, degree, power, waitHold);
}



/*---------------------------------------------------------------------
/ シングルタッチ操作
/--------------------------------------------------------------------*/
TouchResult __cdecl TouchPressDown(int touchId, int x, int y)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.TouchPressDown(touchId, x, y);
}

TouchResult __cdecl TouchPressDownWithFingerId(int touchId, int x, int y, int fingerId)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.TouchPressDown(touchId, x, y, fingerId);
}

TouchResult __cdecl TouchPressUp(int touchId)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.TouchPressUp(touchId);
}

TouchResult __cdecl TouchMove(int touchId, int x, int y)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.TouchMove(touchId, x, y);
}

TouchResult __cdecl Tap(int touchId, int x, int y, int waitPress)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.Tap(touchId, x, y, waitPress);
}

TouchResult __cdecl DragAngle(int touchId, int startX, int startY, float angle, float length, int actionDuration, int waitSpan, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.DragAngle(touchId, startX, startY, angle, length, actionDuration, waitSpan, releaseAfter);
}

TouchResult __cdecl DragPos(int touchId, int startX, int startY, int endX, int endY, int actionDuration, int waitSpan, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.DragPos(touchId, startX, startY, endX, endY, actionDuration, waitSpan, releaseAfter);
}


/*---------------------------------------------------------------------
/ マルチタッチ操作
/--------------------------------------------------------------------*/
TouchResult __cdecl SetTouchState(int touchIdList[], int fingerIdList[], float posXList[], float posYList[], int touchCount)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }

    return multiTouchManager.SetTouchState(touchIdList, fingerIdList, posXList, posYList, touchCount);
}

TouchResult __cdecl ResetTouchState()
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }

    return multiTouchManager.ResetTouchState();
}

TouchResult __cdecl MultiTap(int touchIdList[], int touchCount, float touchPosXList[], float touchPosYList[],
    int waitSpan)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.MultiTap(touchIdList, touchCount, touchPosXList, touchPosYList, waitSpan);
}

TouchResult __cdecl MultiTouchAnimator(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    float centerMoveDeg, float centerMoveLength,
    float touchRelativePosXList[], float touchRelativePosYList[],
    float startDeg, float rotateDeg,
    float startScale, float increaseScale,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.MultiTouchAnimator(touchIdList, touchCount,
        startCenterX, startCenterY,
        centerMoveDeg, centerMoveLength,
        touchRelativePosXList, touchRelativePosYList,
        startDeg, rotateDeg,
        startScale, increaseScale,
        actionDuration, waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}


TouchResult __cdecl MultiDragPos(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    int endCenterX, int endCenterY,
    float touchRelativePosXList[], float touchRelativePosYList[],
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.MultiDragPos(touchIdList, touchCount,
        startCenterX, startCenterY,
        endCenterX, endCenterY,
        touchRelativePosXList, touchRelativePosYList,
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult __cdecl MultiDragAngle(int touchIdList[], int touchCount,
    int startCenterX, int startCenterY,
    float centerMoveDeg, float centerMoveLength,
    float touchRelativePosXList[], float touchRelativePosYList[],
    int actionDuration,  int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.MultiDragAngle(touchIdList, touchCount,
        startCenterX, startCenterY,
        centerMoveDeg, centerMoveLength,
        touchRelativePosXList, touchRelativePosYList,
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult __cdecl Pinch(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[],
    float startRadius, float increaseRadius,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.Pinch(touchIdList, touchCount,
        centerX, centerY,
        touchDegList,
        startRadius, increaseRadius,
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult __cdecl CenterTouchPinch(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[],
    float startRadius, float increaseRadius,
    int actionDuration,  int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{
    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.CenterTouchPinch(touchIdList, touchCount,
        centerX, centerY,
        touchDegList,
        startRadius, increaseRadius,
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult __cdecl MultiRotate(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[], float radius,
    float startDeg, float rotateDeg,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.MultiRotate(touchIdList, touchCount,
        centerX, centerY,
        touchDegList, radius,
        startDeg, rotateDeg,
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);
}

TouchResult __cdecl CenterTouchMultiRotate(int touchIdList[], int touchCount,
    int centerX, int centerY,
    float touchDegList[], float radius,
    float startDeg, float rotateDeg,
    int actionDuration, int waitSpan,
    int waitTouchBegin, int waitTouchEnd, bool releaseAfter)
{

    if (!IsInitialized())
    {
        return TouchResult::TouchResult_NotInitialized;
    }
    return multiTouchManager.CenterTouchMultiRotate(touchIdList, touchCount,
        centerX, centerY,
        touchDegList, radius,
        startDeg, rotateDeg,
        actionDuration,  waitSpan,
        waitTouchBegin, waitTouchEnd, releaseAfter);

}



BOOL APIENTRY DllMain(HMODULE hModule,
    DWORD  ul_reason_for_call,
    LPVOID lpReserved
)
{
    switch (ul_reason_for_call)
    {
        case DLL_PROCESS_ATTACH:
            // TODO: プロセスサイクルに合わせて、Initialize, Fainalize を呼ばすのか、
            // ユーザーが明示的に呼べるようにするのかは、検討が必要。
            //Initialize();
            break;
        case DLL_THREAD_ATTACH:
            break;
        case DLL_THREAD_DETACH:
            break;
        case DLL_PROCESS_DETACH:
            // 内部スレッド終了処理
            // プロセスデタッチ時に念のためファイナライズしておく
            Finalize();
            break;
        default:
            break;
    }
    return TRUE;
}
