﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <GL/glew.h>
#include <iostream>
#include <mutex>
#include <opencv2/opencv.hpp>
#include <vector>
#include <windows.h>


class WindowManager
{
public:
    static WindowManager& GetInstance();

    bool OpenImageWindow(const char* windowName, const cv::Mat& image, int imageWidth, int imageHeight, bool isUseMessage);
    bool CloseImageWindow(const char* windowName, bool isUseMessage);
    void UpdateWindowImage(const char* windowName, const cv::Mat& image, int imageWidth, int imageHeight, bool isChangeWindowSize);

    // ウィンドウプロシージャ用
    void AdjustWindowViewPort(HWND windowHandle);
    void RedrawWindow(HWND windowHandle);
    void RemoveWindowList(HWND windowHandle);
    void OpenPopupMenu(HWND windowHandle, int x, int y);
    void RestoreWindowSize(HWND windowHandle);

private:
    // TIPS: ウィンドウプロシージャ、クラス名を重複させられる仕組みとなっていないため、シングルトンで定義
    WindowManager();
    ~WindowManager();
    WindowManager(const WindowManager& copy);
    WindowManager& operator=(const WindowManager& copy);

    // ウィンドウオープンメッセージ
    struct OpenWindowRequestMessage
    {
        std::string windowName;
        int width;
        int height;
        bool isEnableRequest;
    };
    struct OpenWindowResultMessage
    {
        HWND        hwnd;
        bool        result;
        bool        isEnableResult;
    };
    std::mutex               m_OpenWindowMutex;
    std::mutex               m_OpenWindowMessageMutex;
    OpenWindowRequestMessage m_OpenWindowRequest;
    OpenWindowResultMessage  m_OpenWindowResult;

    // ウィンドウクローズメッセージ
    struct CloseWindowRequestMessage
    {
        std::string windowName;
        bool isEnableRequest;
    };
    struct CloseWindowResultMessage
    {
        bool    result;
        bool    isEnableResult;
    };
    std::mutex      m_CloseWindowMutex;
    std::mutex      m_CloseWindowMessageMutex;
    CloseWindowRequestMessage m_CloseWindowRequest;
    CloseWindowResultMessage  m_CloseWindowResult;

    // テクスチャ
    bool CreateTexture(GLuint* pOutTextureId, HDC hdc, HGLRC hglrc, const cv::Mat& image);

    // ウィンドウ情報
    WNDCLASSEX          m_WindowClass;
    ATOM                m_WindowClassAtom;
    void RegisterWindowClass();

    // 管理ウィンドウリスト
    struct WindowInfo
    {
        std::string     name;
        HDC             hdc;
        HWND            hwnd;
        HGLRC           hglrc;
        GLuint          textureId;
        int             imageWidth;
        int             imageHeight;
        int             windowClientWidth;
        int             windowClientHeight;
    };
    std::vector<WindowInfo> m_WindowList;
    std::mutex              m_WindowListMutex;

    // WindowInfo 更新
    void AdjustWindowInfoAspectRatio(WindowInfo& windowInfo);
    void UpdateWindowInfoTexture(WindowInfo& windowInfo, const cv::Mat& image, int imageWidth, int imageHeight, bool isChangeWindowSize);
    void DrawWindowInfo(WindowInfo& windowInfo);

    // ポップアップメニュー
    HMENU m_PopupMenuHandle;
    void InitializePopupMenu();
    void UninitializePoupMenu();

    // ウィンドウメッセージ処理スレッド
    bool m_IsActiveMessageThread;
    std::unique_ptr<std::thread> m_ProcessMessageThreadObject;
    void ProcessMessageThreadBegin();
    void ProcessMessageThreadEnd();
    void ProcessMessage();
    void ProcessOpenWindowMessage();
    void ProcessOpenWindowMessageHelper(const char* title, int width, int height);
    void ProcessCloseWindowMessage();
    void ProcessCloseWindowMessageHelper(const char* title);
};
