﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "common.h"
#include "CaptureGrabber.h"
#include "CaptureProcessorException.h"
#include "FilterManager.h"
#include "WindowManager.h"

#include <autoTestAssistTool/Common/autoTestAssistTool_LibTypes.h>
#include <map>
#include <mutex>
#include <thread>
#include <opencv/cv.hpp>

#pragma warning(push)
#pragma warning(disable:4267)
#pragma warning(disable:4996)
#include <tesseract/baseapi.h>
#include <tesseract/publictypes.h>
#pragma warning(pop)

class ICapture
{
public:
    // 画像取得
    CaptureResult GetImageFromCaptureDevice(ImageIdType *pOutImageId);
    CaptureResult LoadImageFile(ImageIdType *pOutImageId, const char *filepath);

    // 画像解放
    CaptureResult ReleaseImage(ImageIdType id);
    CaptureResult ReleaseAllImages();

    // 画像ファイル出力
    CaptureResult SaveImageFile(std::string filepath, ImageIdType id);

    // テンプレートマッチング
    MatchingResult DetectObject(int *pOutX, int *pOutY, ImageIdType id, double threshold, int method);
    MatchingResult WaitDetectObject(int *pOutX, int *pOutY, ImageIdType id, double threshold, int timeout, int method);
    MatchingResult DetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, FilterIdType filterId, int method);
    MatchingResult WaitDetectObjectWithFilter(int* pOutX, int* pOutY, ImageIdType id, double threshold, int timeout, FilterIdType filterId, int method);

    // ライブラリ呼び出し側への画像リソース受け渡し
    CaptureResult GetImageSize(int* pOutWidth, int* pOutHeight, ImageIdType id);
    CaptureResult GetImageData(void* pOutColorBuffer, size_t colorBufferSize, ImageIdType id);

    ICapture();
    virtual ~ICapture();

    // ドット色検出
    CaptureResult GetDotColor( Rgb24Color* pOutColor,  ImageIdType id, int x, int y );
    MatchingResult DetectDotColor( int x, int y, Rgb24Color lowerColor, Rgb24Color upperColor );
    MatchingResult WaitDetectDotColor( int x, int y,  Rgb24Color lowerColor, Rgb24Color upperColor, int timeout );

    // ラベリング
    CaptureResult ExecuteLabeling( int *pOutResultAreaCount, LabelingResultHandleType *pOutlabelingResultHandle, ImageIdType imageId);
    CaptureResult GetLabelingResults( int *pOutResultsCount, LabelingResult pOutResultArray[], LabelingResultHandleType labelingResultHandle, int arrayCount );
    CaptureResult ReleaseLabelingResult(LabelingResultHandleType labelingResultHandle);

    // OCR
    MatchingResult DetectTextForImage( ImageRect *pOutRect,
                                    ImageIdType id,
                                    const char regexString[],
                                    OcrSeparateLevel separate,
                                    OcrLanguage language);
    CaptureResult ExecuteOcr( int *pOutResultCount, OcrResultHandleType *pOutOcrResultHandle, ImageIdType id, OcrSeparateLevel separate, OcrLanguage language );
    CaptureResult GetOcrResultStringSize( int *pOutResultSize, OcrResultHandleType ocrResultHandle, int number );
    CaptureResult GetOcrResultString( ImageRect *pOutRect, int *pOutResultSize, char pOutResultString[], int bufferSize, OcrResultHandleType ocrResultHandle, int number );
    CaptureResult ReleaseOcrResult(OcrResultHandleType ocrResultHandle);

    // フィルター
    CaptureResult CreateFilter( FilterIdType *pOutFilterId );
    CaptureResult DeleteFilter( FilterIdType filterId );
    CaptureResult ApplyFilter( ImageIdType *pOutId, ImageIdType id, FilterIdType filterId );
    CaptureResult SetBinarizationFilter( FilterIdType filterId, int binarizeThreshold );
    CaptureResult SetColorRangeFilter(FilterIdType filterId, Rgb24Color lowerColor, Rgb24Color upperColor);
    CaptureResult SetMorphologyFilter(FilterIdType filterId, MorphologyConversionType type, int iterations);

    // ウィンドウ描画
    CaptureResult OpenImageWindow( const char* windowName, ImageIdType id );
    CaptureResult CloseImageWindow( const char* windowName );
    CaptureResult ExtractImage(ImageIdType *pOutImageId, ImageIdType id, ImageRect rect);

protected:
    virtual std::string GetPreviewName() = 0;
    virtual int GetOriginalPreviewWidth() = 0;
    virtual int GetOriginalPreviewHeight() = 0;

    // プレビュー
    void            StartPreview();
    void            RunPreview();
    void            EndPreview();

    // プレビュー・キャプチャスレッド
    bool                    m_IsCapture;
    bool                    m_IsPreview;
    std::unique_ptr<std::thread> m_PreviewThread;
    cv::Mat                  m_CaptureImage;
    std::mutex               m_CaptureMutex;

    const int           DefaultWindowHeight = 720;
    const int           DefaultWindowWidth = 1280;
    static const float  PreviewFrameSpan;

    // ラベリング
    std::mutex                                      m_LabelingResultMutex;
    std::map<LabelingResultHandleType, std::vector<LabelingResult>>  m_LabelingResultList;

    // OCR
    std::mutex                      m_TesseractOcrMutex;
    struct OcrResultTypes
    {
        ImageRect rect;
        std::string word;
    };
    tesseract::TessBaseAPI          m_TesseractOcrBaseApi;
    std::map <OcrResultHandleType, std::vector<OcrResultTypes>> m_TesseractOcrResultList;

    // フィルター
    FilterManager                   m_FilterManager;

private:
    cv::Mat         GetCaptureImageMapData(ImageIdType id);
    ImageIdType     SetCaptureImageMapData(cv::Mat& image);
    void            DeleteCaptureImageMapData(ImageIdType id);

    // キャプチャ画像リスト
    std::map<ImageIdType, cv::Mat>    m_CaptureImageMap;
    std::mutex                        m_CaptureImageMapMutex;
};
