﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>


#include "GraphicBase.h"
#include "DisplayData.h"
#include "FileManager.h"
#include "GraphicMain.h"

#include <nn/fs/fs_GameCard.h>

const float ButtonPosx   = 1000;
const float ButtonPosy   = 30;
const float ButtonWidth  = 250;
const float ButtonHeight = 100;

GraphicMain::GraphicMain()
{
    m_pDisplayData    = DisplayData::GetInstance();
    m_StorageCheckerMode = StorageCheckerMode_Menu;
    m_CursorPos = 0;
    m_TouchedX  = 0;
    m_TouchedY  = 0;
    m_ErrorStartPos = 0;
    m_CurrentPagePos = 0;
    m_CurrentMenuPagePos = 0;
    m_DisplayableByteSize = 256;
    m_DisplayableFileNum  = 7;
    m_FileNum = 0;
    m_IsNextPageButtonPushed = false;
    m_IsBackPageButtonPushed = false;
    m_IsExecuteButtonPushed = false;
    m_IsBackButtonPushed = false;
    m_IsEraseMode = false;
}

void GraphicMain::Initialize(size_t applicationHeapSize)
{
    m_ApplicationHeap.Initialize(applicationHeapSize);

    m_pGraphicsSystem = new GraphicsSystem();
    m_pGraphicsSystem->SetApplicationHeap(&m_ApplicationHeap);
    m_pGraphicsSystem->Initialize();

    m_pFontSystem = new FontSystem();
    m_pFontSystem->SetApplicationHeap(&m_ApplicationHeap);
    m_pFontSystem->SetGraphicsSystem(m_pGraphicsSystem);
    m_pFontSystem->Initialize();
}

//------------------------------------------------------------------------------
// 描画部品
//------------------------------------------------------------------------------


bool GraphicMain::SetTouchableTextAndDetectTouch(float x, float y, char* buffer)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetCursor(x, y);
    textWriter.Print(buffer);
    float width  = textWriter.CalculateStringWidth(buffer) + 10;
    float height = textWriter.CalculateStringHeight(buffer) + 20;

    if(m_TouchedX > x &&
            m_TouchedX < x + width &&
            m_TouchedY > y &&
            m_TouchedY < y + height)
    {
        return true;
    }
    return false;
}

void GraphicMain::SetLine(float startX, float startY, float endX, float endY)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    char buffer = '*';
    textWriter.SetScale(1.2, 1.2);

    float dx = endX - startX;
    float dy = endY - startY;

    if(dx == 0 && dy == 0)
    {
        textWriter.SetCursor(startX, startY);
        textWriter.Print(&buffer);
    }
    else
    {
        // 描画する点の個数を決定します
        float lineLength = 1.0f / nn::util::Rsqrt(dx * dx + dy * dy);
        int pointNum = lineLength / 2.5;

        for(int i = 0; i <= pointNum; i++)
        {
            float x = (startX * (pointNum - i) + endX * i) / pointNum;
            float y = (startY * (pointNum - i) + endY * i) / pointNum;
            textWriter.SetCursor(x, y);
            textWriter.Print(&buffer);
        }
    }
}

bool GraphicMain::SetTouchableSquareAndDetectTouch(float x, float y, float width, float height)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Green);
    SetLine(x, y, x + width, y);
    SetLine(x, y, x, y + height);
    SetLine(x + width, y, x + width, y + height);
    SetLine(x, y + height, x + width, y + height);
    if(m_TouchedX > x &&
            m_TouchedX < x + width &&
            m_TouchedY > y &&
            m_TouchedY < y + height)
    {
        return true;
    }
    return false;
}

bool GraphicMain::SetButtonAndBuffer(float x, float y, float width, float height, char* buffer)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    auto color = textWriter.GetTextColor();
    bool ret = false;
    ret = SetTouchableSquareAndDetectTouch(x, y, width, height);
    textWriter.SetScale(2.4, 2.4);
    float textWidth  = textWriter.CalculateStringWidth(buffer) - 10;
    float textHeigth = textWriter.CalculateStringHeight(buffer) - 20;
    textWriter.SetTextColor(color);
    textWriter.SetCursor(x + (width - textWidth) / 2, y + (height - textHeigth) / 2);
    textWriter.Print(buffer);
    return ret;
}



//------------------------------------------------------------------------------
// 描画メイン
//------------------------------------------------------------------------------

/* モード別テキスト描画 */
void GraphicMain::SetAndDrawText()
{
    switch(m_StorageCheckerMode)
    {
        case StorageCheckerMode_Menu:
            SetTouchTextForMenu();
            SetTextForMenu();
            break;
        case StorageCheckerMode_Script:
            SetTextForScript();
            SetTouchTextForCancel();
            break;
        case StorageCheckerMode_ScriptEnd:
            SetTextForScript();
            SetTouchTextForScript();
            break;
        case StorageCheckerMode_ScriptError:
            SetTextForScript();
            SetTouchTextForScript();
            break;
        default :
            NN_ABORT("unexpected in SetAndDrawText\n");
    }
    DrawText();
}

/* メインでループする描画 API*/
void GraphicMain::Execute()
{
    SetAndDrawText();
}

//------------------------------------------------------------------------------
// メニュー画面用描画
//------------------------------------------------------------------------------

/* 選択画面でラベル設定*/
void GraphicMain::SetLabelForMenu(char* buffer, size_t line)
{
    const size_t StartPosX    = 20;
    const size_t StartPosY    = 200;
    const size_t HeightBase   = 60;
    const size_t WidthBase    = 50;

    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    // 選択内容表示
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2.0, 2.0);

    if(m_CursorPos == line)
    {
        textWriter.SetTextColor(Color::Orange);
        textWriter.SetCursor(StartPosX, StartPosY + HeightBase * line);
        textWriter.Print(">");
    }
    if(SetTouchableTextAndDetectTouch(StartPosX + WidthBase, StartPosY + HeightBase * line, buffer))
    {
        m_CursorPos = line;
    }
}

/* 選択画面での描画 */
void GraphicMain::SetTouchTextForMenu()
{
    const float LabelY = 650;
    const float BackPageStartX = 0;
    const float NextPageStartX = 200;
    const float ModeChangeStartX = ButtonPosx + 100;
    const float ModeChangeStartY = ButtonPosy + ButtonHeight + 50;
    char buffer[64];

    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Green);
    textWriter.SetScale(2.0, 2.0);

    // ページ送り 表示
    if(IsExistPreviousPage())
    {
        nn::util::Strlcpy(buffer, "<< Before", sizeof(buffer));
        if(SetTouchableTextAndDetectTouch(BackPageStartX, LabelY, buffer))
        {
            m_IsBackPageButtonPushed = true;
        }
    }
    if(IsExistNextPage())
    {
        nn::util::Strlcpy(buffer, "Next >>", sizeof(buffer));
        if(SetTouchableTextAndDetectTouch(NextPageStartX, LabelY, buffer))
        {
            m_IsNextPageButtonPushed = true;
        }
    }

    // ボタン表示
    if(m_pDisplayData->isFileExist && m_pDisplayData->IsCardWritable() &&
            (m_pDisplayData->IsWritableFile() || m_pDisplayData->IsXciFile()))
    {
        // Write 出来るときは必ず Erase もできるのでここで切り替える
        if(m_IsEraseMode)
        {
            nn::util::Strlcpy(buffer, "WRITE >>", sizeof(buffer));
        }
        else
        {
            nn::util::Strlcpy(buffer, "ERASE >>", sizeof(buffer));
        }
        textWriter.SetTextColor(Color::Gray);
        if(SetTouchableTextAndDetectTouch(ModeChangeStartX, ModeChangeStartY, buffer))
        {
            m_IsModeChangeButtonPushed = true;
        }

        if(!m_IsEraseMode)
        {
            nn::util::Strlcpy(buffer, "WRITE", sizeof(buffer));
            textWriter.SetTextColor(Color::White);
            if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
            {
                m_IsExecuteButtonPushed = true;
            }
        }
        else
        {
            nn::util::Strlcpy(buffer, "ERASE", sizeof(buffer));
            textWriter.SetTextColor(Color::White);
            if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
            {
                m_IsEraseButtonPushed = true;
            }
        }
    }
    else if(m_pDisplayData->cardState == CardState_BrokenCard || m_pDisplayData->IsCardWritable())
    {
        nn::util::Strlcpy(buffer, "ERASE", sizeof(buffer));
        textWriter.SetTextColor(Color::White);
        if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
        {
            m_IsEraseButtonPushed = true;
        }
    }
    else
    {
        nn::util::Strlcpy(buffer, "DISABLE", sizeof(buffer));
        textWriter.SetTextColor(Color::Red);
        SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer);
    }
}

void GraphicMain::DrawCardState()
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();

    // 挿入状態表示
    textWriter.SetCursor(400, 40);
    textWriter.SetScale(2.4, 2.4);

    const float CardDetailIntroStartPosX = 370;
    const float CardDetailStartPosX = 530;
    const float CardDetailStartPosY = 110;

    if(m_pDisplayData->cardState == CardState_NotInserted)
    {
        textWriter.SetTextColor(Color::Gray);
        textWriter.Print("GameCard : Not Inserted");
    }
    else if(m_pDisplayData->cardState == CardState_BrokenCard)
    {
        textWriter.SetTextColor(Color::Red);
        textWriter.Print("GameCard : Broken");
    }
    else if(m_pDisplayData->cardState == CardState_Repairing)
    {
        textWriter.SetTextColor(Color::Blue);
        textWriter.Print("GameCard : Erasing");

        textWriter.SetTextColor(Color::Blue);
        textWriter.SetScale(1.8, 1.8);
        textWriter.SetCursor(CardDetailIntroStartPosX, CardDetailStartPosY);
        char repairText[64] = "Erasing Card";
        char dot[] = ".";
        for(size_t i = 0; i < GetCounter(); i++)
        {
            strncat(repairText, dot, nn::util::Strnlen(dot, sizeof(dot)));
        }
        textWriter.Print(repairText);
    }
    else
    {
        textWriter.SetTextColor(Color::White);
        textWriter.Print("GameCard : Inserted");

        // DeviceID 表示
        textWriter.SetScale(1.8, 1.8);
        if(m_pDisplayData->cardState == CardState_InvalidCardType)
        {
            textWriter.SetCursor(CardDetailIntroStartPosX, CardDetailStartPosY);
            textWriter.SetTextColor(Color::Red);
            textWriter.Print("Card is invalid!! Cofirm card.");
            textWriter.SetCursor(CardDetailIntroStartPosX, CardDetailStartPosY + 40);
            textWriter.Print("You can use only Prod Writable Type Card.");
        }
        else if(m_pDisplayData->cardState == CardState_Accessing)
        {
            textWriter.SetCursor(CardDetailIntroStartPosX, CardDetailStartPosY);
            char accessText[64] = "Accessing Card Info";
            char dot[] = ".";
            for(size_t i = 0; i < GetCounter(); i++)
            {
                strncat(accessText, dot, nn::util::Strnlen(dot, sizeof(dot)));
            }
            textWriter.Print(accessText);
        }
        else
        {
            textWriter.SetCursor(CardDetailIntroStartPosX, CardDetailStartPosY);
            textWriter.Print("Device Id:");
            for(int i = 0; i < 16 / 8; i++)
            {
                for(int j = 0; j < 8; j++)
                {
                    textWriter.SetCursor(CardDetailStartPosX + j * 53, CardDetailStartPosY + i * 40);
                    textWriter.Print("%02X", *(m_pDisplayData->GetDeviceId() + j + i * 8));
                }
            }
        }
    }
}

void GraphicMain::SetStringToMultiLine(char* buffer, size_t lineWidth, size_t lineHeight, size_t maxLineNum, float startX, float startY)
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    const size_t MaxLineWidth = 120;
    char buf[MaxLineWidth];
    size_t restLength = nn::util::Strnlen(buffer, lineWidth * maxLineNum);
    for(size_t  i = 0; i < maxLineNum; i++)
    {
        textWriter.SetCursor(startX, startY + lineHeight * i);
        nn::util::Strlcpy(buf, buffer + (lineWidth - 1) * i, lineWidth);
        textWriter.Print("%s", buf);

        if(restLength >= lineWidth)
        {
            restLength -= lineWidth;
        }
        else
        {
            break;
        }
    }
}


void GraphicMain::SetFixedTextForMenu()
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    // 選択内容表示
    textWriter.SetTextColor(Color::Gray);

    // フォルダ表示
    textWriter.SetScale(1.6, 1.6);
    SetStringToMultiLine(FileManager::GetInstance()->GetRootPath(), 25, 30, 3, 0, 40);

    textWriter.SetScale(2.0, 2.0);
    // Definition files 表示
    textWriter.SetTextColor(Color::Orange);
    textWriter.SetCursor(20, 140);
    textWriter.Print("Definition files");

    // カード挿入時の表示
    DrawCardState();
}

void GraphicMain::SetHostTextForMenu()
{
    // PC 上の該当フォルダ内の txt ファイル表示
    FileManager* pFileManager = FileManager::GetInstance();
    m_FileNum = pFileManager->GetFileNum();
    if(m_FileNum == 0)
    {
        return;
    }
    size_t maxPageSize = m_FileNum / m_DisplayableFileNum;
    if(m_FileNum % m_DisplayableFileNum == 0 && maxPageSize > 0)
    {
        maxPageSize--;
    }
    if(m_CurrentMenuPagePos > maxPageSize)
    {
        m_CurrentMenuPagePos = maxPageSize;
    }

    char buffer[FilePathSize];
    for(int64_t i = 0 ; i < m_DisplayableFileNum; i++)
    {
        int64_t curPos = m_DisplayableFileNum * m_CurrentMenuPagePos + i;
        if(curPos < m_FileNum)
        {
            if(pFileManager->GetFileName(buffer, sizeof(buffer), curPos) == 0)
            {
                return;
            }
            buffer[nn::util::Strnlen(buffer, sizeof(buffer)) - 4] = '\0';// .txt 削除
            SetLabelForMenu(buffer, i);
        }
    }

    // 選択されている txt 内の詳細情報表示

    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2.0, 2.0);
    textWriter.SetCursor(400, 250);
    textWriter.Print("Target File :");
    textWriter.SetScale(1.8, 1.8);
    SetStringToMultiLine(m_pDisplayData->GetFileName(), 38, 40, 3, 410, 300);

    textWriter.SetCursor(400, 420);
    textWriter.SetScale(2.0, 2.0);
    if(m_pDisplayData->isFileExist && (m_pDisplayData->IsWritableFile() || m_pDisplayData->IsXciFile()))
    {
        textWriter.Print("File : %lld MB", m_pDisplayData->fileSize / (1024 * 1024));
        textWriter.SetCursor(700, 420);
        textWriter.Print("Data : %lld MB", m_pDisplayData->validDataSize / (1024 * 1024));
        textWriter.SetCursor(400, 490);
        textWriter.Print("Card Image Hash:");
        textWriter.SetScale(1.8, 1.8);
        for(int i = 0; i < 32 / 16; i++)
        {
            for(int j = 0; j < 16; j++)
            {
                textWriter.SetCursor(410 + j * 53, 540 + i * 40);
                textWriter.Print("%02X", *(m_pDisplayData->GetCardHeaderHash() + i * 16 + j));
            }
        }
    }
    else if(m_pDisplayData->isFileExist)
    {
        textWriter.SetTextColor(Color::Red);
#if defined(NN_XCIE_WRITER_XCIE_MODE)
        textWriter.Print("Target file must be '.xcie' file.");
#elif defined(NN_XCIE_WRITER_XCIR_MODE)
        textWriter.Print("Target file must be '.xcir' file.");
#else
        textWriter.Print("Target file must be '.xci' file.");
#endif

    }
    else
    {
        textWriter.SetTextColor(Color::Red);
        textWriter.Print("Target file Does NOT exist");
    }

    if(*(m_pDisplayData->GetParamPath()) != 0x00)
    {
        textWriter.SetCursor(400, 640);
        textWriter.SetScale(2.0, 2.0);
        if(m_pDisplayData->isParamExist)
        {
            textWriter.SetTextColor(Color::White);
            textWriter.Print("Param File : %s", m_pDisplayData->GetParamPath());
        }
        else
        {
            textWriter.SetTextColor(Color::Red);
            textWriter.Print("Param File : %s", m_pDisplayData->GetParamPath());
            textWriter.SetCursor(400, 680);
            textWriter.Print("Param File Does NOT Exist");
        }
    }

}

void GraphicMain::SetMountHostErrorTextForMenu()
{
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::Red);
    textWriter.SetScale(2.2, 2.2);
    textWriter.SetCursor(100, 300);
    textWriter.Print("Mount Host Failure!");
    textWriter.SetCursor(150, 400);
    textWriter.Print("Confirm PC and TargetManager Connection.");
}

void GraphicMain::SetTextForMenu()
{
    SetTouchTextForMenu();
    SetFixedTextForMenu();

    if(!m_pDisplayData->isMountError)
    {
        SetHostTextForMenu();
    }
    else
    {
        SetMountHostErrorTextForMenu();
    }
}

/** スクリプト画面のタッチ用テキスト表示 **/
void GraphicMain::SetTouchTextForCancel()
{
    char buffer[64];
    nn::util::Strlcpy(buffer, "CANCEL", sizeof(buffer));
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    if(m_pDisplayData->isWriteCancelConfirm)
    {
        textWriter.SetTextColor(Color::Red);
    }
    else
    {
        textWriter.SetTextColor(Color::Gray);
    }
    if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
    {
        m_IsWriteCancelButtonPushed = true;
    }
}

void GraphicMain::SetTouchTextForScript()
{
    char buffer[64];
    nn::util::Strlcpy(buffer, "MENU", sizeof(buffer));
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::White);
    if(SetButtonAndBuffer(ButtonPosx, ButtonPosy, ButtonWidth, ButtonHeight, buffer))
    {
        m_IsBackButtonPushed = true;
    }
}

/** スクリプト画面表示 **/
void GraphicMain::SetTextForScript()
{
    const size_t HeightBase = 100;
    size_t lineNum = 0;
    nn::gfx::util::DebugFontTextWriter& textWriter = m_pFontSystem->GetDebugFontTextWriter();
    textWriter.SetTextColor(Color::White);
    textWriter.SetScale(2.4, 2.4);
    textWriter.SetCursor(30, HeightBase * lineNum++);
    if(m_StorageCheckerMode == StorageCheckerMode_ScriptEnd)
    {
        textWriter.SetTextColor(Color::Blue);
        textWriter.Print("Finished!! Please Touch Menu Button.");
    }
    else if(m_pDisplayData->cardState == CardState_Accessing)
    {
        textWriter.SetTextColor(Color::Red);
        textWriter.Print("Confirming. Don't Remove Card yet.");
    }
    else if(!m_pDisplayData->IsCardWritable())
    {
        m_StorageCheckerMode = StorageCheckerMode_ScriptError;
        textWriter.SetTextColor(Color::Red);
        if(!m_pDisplayData->isFileExist)// PC からの読み込みに失敗している
        {
            textWriter.Print("Error!!! FAIL to READ File From PC.\n");
        }
        else if(m_pDisplayData->cardState == CardState_NotInserted)
        {
            textWriter.Print("Error!!! Card Removed.\n");
        }
        else if(m_pDisplayData->cardState == CardState_InvalidCardType)
        {
            textWriter.Print("Error!!! This card may not be Writable card.\n");
        }
        else if(m_pDisplayData->cardState == CardState_InvalidCardData)
        {
            textWriter.Print("Error!!! This data may be wrong.\n");
        }
    }
    else
    {
        textWriter.SetTextColor(Color::Red);
        textWriter.Print("Now Writing.. Don't Remove Game Card!!");
    }
    lineNum++;

    textWriter.SetTextColor(Color::White);

    // ターゲット名表示
    textWriter.SetCursor(30, HeightBase * lineNum++);
    textWriter.Print("Target File : %s", m_pDisplayData->GetFileName());

    // Progress 表示
    textWriter.SetCursor(30, HeightBase * lineNum++);
    textWriter.Print("Progress  : %lld MB / %lld MB   (%d%)", m_pDisplayData->currentWrittenSize / (1024 * 1024), m_pDisplayData->validDataSize / (1024 * 1024), (m_pDisplayData->currentWrittenSize * 100 / m_pDisplayData->validDataSize));

    // Progress バー
    const float BarStartX = 200;
    const float BarLength = 800;
    float width = (m_pDisplayData->currentWrittenSize * BarLength / m_pDisplayData->validDataSize);
    textWriter.SetTextColor(Color::White);
    SetLine(BarStartX + width, HeightBase * lineNum, BarStartX + BarLength, HeightBase * lineNum);
    textWriter.SetTextColor(Color::Green);
    SetLine(BarStartX, HeightBase * lineNum, BarStartX + width,  HeightBase * lineNum);
}

