﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/util/util_StringUtil.h>

#include "FileAccessor.h"

namespace
{
const uint32_t MaxDirectoryCount = 256;
#if defined(NN_XCIE_WRITER_XCIR_MODE)
const char* DefaultFolderPath = "C:/Xcir/";
#else
const char* DefaultFolderPath = "C:/Xcie/";
#endif
const size_t MaxRootPathNameLength = 400;
}

class FileManager
{
private:
    FileManager()
    {
        m_FileNum = 0;
        m_IsInitialized = false;
        nn::util::Strlcpy(m_FolderPath, DefaultFolderPath, MaxRootPathNameLength);
    }
    int64_t m_FileNum;
    bool m_IsInitialized;
    nn::fs::DirectoryEntry m_DirectoryEntryList[MaxDirectoryCount];
    char m_FolderPath[MaxRootPathNameLength];

public:
    static FileManager* GetInstance()
    {
        static FileManager fileManager;
        return &fileManager;
    }
    nn::Result Mount()
    {
        // マウント
        FileAccessor& fileAccessor = FileAccessor::GetInstance();
        return fileAccessor.Mount();
    }
    void SetRootPath(const char* rootPath)
    {
        if(nn::util::Strnlen(rootPath, MaxRootPathNameLength + 1) > MaxRootPathNameLength)
        {
            NN_ABORT("root path name length must be lower than %d.", MaxRootPathNameLength);
        }
        nn::util::Strlcpy(m_FolderPath, rootPath, MaxRootPathNameLength);
    }
    char* GetRootPath()
    {
        return m_FolderPath;
    }
    void Initialize()
    {
        // ファイルリスト取得
        FileAccessor& fileAccessor = FileAccessor::GetInstance();
        if(fileAccessor.GetFileList(m_DirectoryEntryList, &m_FileNum, MaxDirectoryCount, m_FolderPath).IsFailure())
        {
            NN_LOG("*** [warning] root path (%s) is not found.So, Create the folder.*** \n", m_FolderPath);
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::CreateDirectory(m_FolderPath));
            NN_ABORT_UNLESS_RESULT_SUCCESS(fileAccessor.GetFileList(m_DirectoryEntryList, &m_FileNum, MaxDirectoryCount, m_FolderPath));
        }
        // ".txt" だけ抽出
        size_t listNum = 0;
        for(int64_t i = 0; i < m_FileNum; i++)
        {
            std::string str = m_DirectoryEntryList[i].name;
            int extPos = str.find_last_of(".");
            //"." が見つからない場合はスルー
            if(extPos == std::string::npos)
            {
                continue;
            }
            // 拡張子が ".txt" のときのみリスト入り
            std::string extname = str.substr(extPos, str.size() - extPos);
            if(extname == ".txt")
            {
                m_DirectoryEntryList[listNum++] = m_DirectoryEntryList[i];
            }
        }
        m_FileNum = listNum;
        m_IsInitialized = true;
    }
    uint32_t GetFileName(char* pOutBuffer, size_t outBufferSize, int64_t fileIndex)
    {
        if(!m_IsInitialized)
        {
            NN_ABORT("Not Initialized\n");
        }
        if(fileIndex > m_FileNum || fileIndex < 0)
        {
            return 0;
            //            NN_ABORT("index is over file Num\n");
        }
        size_t nameLength = nn::util::Strnlen(m_DirectoryEntryList[fileIndex].name, outBufferSize) + 1; //'\0'の分
        if(outBufferSize < nameLength)
        {
            NN_ABORT("file name %s\nfile name length must be lower than %d.", m_DirectoryEntryList[fileIndex].name, FilePathSize);
        }
        nn::util::Strlcpy(pOutBuffer, m_DirectoryEntryList[fileIndex].name, nameLength);
        return nameLength;
    }
    size_t ReadFile(char* pOutBuffer, size_t outBufferSize, int64_t fileIndex)
    {
        if(!m_IsInitialized)
        {
            NN_ABORT("Not Initialized\n");
        }
        if(fileIndex > m_FileNum || fileIndex < 0)
        {
            return 0;
            //            NN_ABORT("index is over file Num\n");
        }
        if(m_DirectoryEntryList[fileIndex].name[0] == '\0')
        {
            return 0;
        }
        FileAccessor& fileAccessor = FileAccessor::GetInstance();
        size_t readDataSize = 0;
        char buffer[FilePathSize];
        nn::util::Strlcpy(buffer, m_FolderPath, sizeof(buffer));
        size_t folderPathLength = nn::util::Strnlen(m_FolderPath, FilePathSize);
        size_t nameLength = nn::util::Strnlen(m_DirectoryEntryList[fileIndex].name, sizeof(buffer) - folderPathLength);
        if(nameLength < sizeof(buffer) - folderPathLength)
        {
            strncat(buffer, m_DirectoryEntryList[fileIndex].name, nameLength);
            fileAccessor.Read(pOutBuffer, &readDataSize, outBufferSize, buffer);
            return readDataSize;
        }
        NN_ABORT("file name %s\nfile name length must be lower than %d.", m_DirectoryEntryList[fileIndex].name, FilePathSize);
        return 0;
    }

    int64_t GetFileNum()
    {
        return m_FileNum;
    }
};
