﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include "FileAccessor.h"

using namespace ::nn::fs;

MountPath FileAccessor::defaultMountPath;

FileAccessor::FileAccessor()
{
    m_IsMounted = false;

}

nn::Result FileAccessor::Mount()
{

    nn::Result result = nn::fs::MountHostRoot();
    if(result.IsFailure())
    {
        m_IsMounted = false;
        NN_LOG("nn::fs::MountHostRoot() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    m_IsMounted = true;
    NN_LOG("Host Mount Success\n");
    return result;
}

void FileAccessor::Unmount()
{
    nn::fs::UnmountHostRoot();
}

bool FileAccessor::IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const uint32_t directoryEntryBufferLength, const char* directoryPath, const char* targetName, int openDirectoryMode)
{
    int64_t directoryEntryListLength = 0;

    this->GetDirectoryOrFileList(workDirectoryEntryBuffer, &directoryEntryListLength, directoryEntryBufferLength, directoryPath, openDirectoryMode);
    if ( directoryEntryListLength > 0 )
    {
        for(uint32_t i=0; i<directoryEntryListLength; i++)
        {
            uint32_t entryNameLength = strnlen(workDirectoryEntryBuffer[i].name, sizeof(workDirectoryEntryBuffer[i].name));
            uint32_t targetNameLength = strnlen(targetName, sizeof(workDirectoryEntryBuffer[i].name));
            if ( entryNameLength == targetNameLength && std::memcmp(workDirectoryEntryBuffer[i].name, targetName, targetNameLength) == 0 )
            {
                return true;
            }
        }
    }
    return false;
}

nn::Result FileAccessor::GetDirectoryOrFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const uint32_t directoryEntryBufferLength, const char* directoryPath, int openDirectoryMode)
{
    nn::fs::DirectoryHandle directory;
    nn::Result result = nn::fs::OpenDirectory(&directory, directoryPath, openDirectoryMode);
    if(result.IsFailure())
    {
        NN_LOG("Open Directory '%s' Failure\n", directoryPath);
        return result;
    }

//    NN_LOG("Read Directory '%s'\n", directoryPath);
    result = nn::fs::ReadDirectory(outDirectoryEntryCount, outDirectoryEntryBuffer, directory, directoryEntryBufferLength);
    if(result.IsFailure())
    {
        NN_LOG("Read Directory '%s' Failure\n", directoryPath);
        return result;
    }

    nn::fs::CloseDirectory(directory);
    NN_RESULT_SUCCESS;
}

void FileAccessor::Read(char* outBuffer, size_t* outReadLength, const size_t offset, const size_t bufferLength, const char* filePath)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = OpenFile(&file, filePath, nn::fs::OpenMode::OpenMode_Read);
    if(result.IsFailure())
    {
        outReadLength = 0;
        NN_LOG("Open File '%s' Failure\n", filePath);
        return;
    }
    int64_t fileSize = 0;
    result = GetFileSize(&fileSize, file);
    if(result.IsFailure() || fileSize > bufferLength)
    {
        outReadLength = 0;
        NN_LOG("%s : Size is too large. (file size is %lld, but file size must be lower than %d)\n", filePath, fileSize, bufferLength);
        return;
    }
    result = ReadFile(outReadLength, file, offset, outBuffer, bufferLength);
    if(result.IsFailure())
    {
        outReadLength = 0;
        NN_LOG("Read File '%s' Failure\n", filePath);
        return;
    }
    CloseFile(file);
}
void FileAccessor::Read(char* outBuffer, size_t* outReadLength, const size_t bufferLength, const char* filePath)
{
    this->Read(outBuffer, outReadLength, 0, bufferLength, filePath);
}

void FileAccessor::Write(const char* buffer, const size_t bufferLength, const char* filePath)
{
    nn::Result result;
    nn::fs::FileHandle file;
    result = DeleteFile(filePath);
    if(result.IsFailure())
    {
        NN_LOG("Could not delete file '%s'\n", filePath);
    }
    result = nn::fs::CreateFile(filePath, bufferLength);
    if(result.IsFailure())
    {
        NN_LOG("Create File '%s' Error\n", filePath);
        NN_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return;
    }
    NN_LOG("Create File '%s' Success\n", filePath);
    result = OpenFile(&file, filePath, nn::fs::OpenMode::OpenMode_Write);
    if(result.IsFailure())
    {
        NN_LOG("Open File '%s' Failure\n", filePath);
        return;
    }
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    result = WriteFile(file, 0, buffer, bufferLength, option);
    if(result.IsFailure())
    {
        NN_LOG("Write File '%s' Failure\n", filePath);
    }
    CloseFile(file);
}

nn::Result FileAccessor::Open(nn::fs::FileHandle* handle, const char* filePath, int64_t fileSize, nn::fs::OpenMode mode)
{
    nn::Result result = nn::ResultSuccess();
    if(mode & nn::fs::OpenMode::OpenMode_Write)
    {
        result = DeleteFile(filePath);
        if(result.IsFailure())
        {
            NN_LOG("Could not delete file '%s'\n", filePath);
        }
        result = nn::fs::CreateFile(filePath, fileSize);
        if(result.IsFailure())
        {
            NN_LOG("Create File '%s' Error\n", filePath);
            NN_LOG("failure (Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
            return result;
        }
        NN_LOG("Create File '%s' Success\n", filePath);
    }
    result = OpenFile(handle, filePath, mode);
    if(result.IsFailure())
    {
        NN_LOG("Open File '%s' Failure\n", filePath);
        return result;
    }
    return result;
}
