﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_SysGraphics.h"

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

void Graphics::InitializeGraphicsFramework() NN_NOEXCEPT
{
    const size_t GraphicsSystemMemorySize = 32 * 1024 * 1024;
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem(GraphicsSystemMemorySize);

    const int BufferCount = 2;
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayWidth(DisplayWidth);
    fwInfo.SetDisplayHeight(DisplayHeight);
    fwInfo.SetBufferCount(BufferCount);
    fwInfo.SetSwapChainBufferCount(BufferCount);
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    pFramework->Initialize(fwInfo);
}
void Graphics::FinalizeGraphicsFramework() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    pFramework->Finalize();
}

nn::util::BytePtr g_DebugFontHeap(nullptr);
void Graphics::InitializeDebugFont() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    nn::gfx::util::DebugFontTextWriter* pWriter = (nn::gfx::util::DebugFontTextWriter*)GetWriter();

    const int BufferCount = 2;
    // デバッグフォント初期化
    const bool userMemoryPoolEnable = true;    // true にすると、ユーザーのメモリプールを使用します
    const int charCountMax = 8192;
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(charCountMax);
    info.SetBufferCount(BufferCount);
    info.SetUserMemoryPoolEnabled(userMemoryPoolEnable);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(
        pFramework->GetDevice(), info);
    g_DebugFontHeap = nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

    size_t debugFontMemoryPoolSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemoryPoolSize(
        pFramework->GetDevice(), info);

    nns::gfx::GraphicsFramework::MemoryPoolType type = nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer;
    nn::gfx::MemoryPool* pMemoryPool = pFramework->GetMemoryPool(type);
    ptrdiff_t memoryPoolOffset = nn::gfx::util::MemoryPoolAllocator::InvalidOffset;
    if (NN_STATIC_CONDITION(userMemoryPoolEnable)) {
        memoryPoolOffset = pFramework->AllocatePoolMemory(type, debugFontMemoryPoolSize, 1);

        pWriter->Initialize(pFramework->GetDevice(), info, g_DebugFontHeap.Get(), debugFontHeapSize,
            pMemoryPool, memoryPoolOffset, debugFontMemoryPoolSize);
    }
    else {
        pWriter->Initialize(pFramework->GetDevice(), info, g_DebugFontHeap.Get(), debugFontHeapSize, nullptr, 0, 0);
    }

    int textureDescriptorIndex = pFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, 1);
    int samplerDescriptorIndex = pFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    pWriter->SetDisplayWidth(pFramework->GetDisplayWidth());
    pWriter->SetDisplayHeight(pFramework->GetDisplayHeight());
    pWriter->SetTextureDescriptor(pFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView), textureDescriptorIndex);
    pWriter->SetSamplerDescriptor(pFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler), samplerDescriptorIndex);
    pWriter->SetTextColor(nn::util::Color4u8::Black());
}
void Graphics::FinalizeDebugFont() NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriter* pWriter = (nn::gfx::util::DebugFontTextWriter*)GetWriter();
    pWriter->Finalize();
    delete[] reinterpret_cast<uint8_t*>(g_DebugFontHeap.Get());
    g_DebugFontHeap.Reset(nullptr);
}

//--------------------------------------------------------------------------------------
//  メモリ確保関数
//--------------------------------------------------------------------------------------
void* AllocateFunction(size_t size, size_t alignment, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    nn::mem::StandardAllocator* pAllocator = static_cast<nn::mem::StandardAllocator*>(pUserData);
    return pAllocator->Allocate(size, alignment);
}

//--------------------------------------------------------------------------------------
//  メモリ開放関数
//--------------------------------------------------------------------------------------
void DeallocateFunction(void* ptr, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pUserData);
    nn::mem::StandardAllocator* pAllocator = static_cast<nn::mem::StandardAllocator*>(pUserData);
    pAllocator->Free(ptr);
}

void Graphics::InitializePrimitiveRenderer() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();

    const size_t workMemorySize = 1024 * 1024 * 256;
    m_pPrimitiveRendererAllocatorMemory = malloc(workMemorySize);
    m_PrimitiveRendererAllocator.Initialize(m_pPrimitiveRendererAllocatorMemory, workMemorySize);

    nns::gfx::PrimitiveRenderer::RendererInfo info;
    info.SetDefault();
    info.SetAllocator(AllocateFunction, &m_PrimitiveRendererAllocator);
    info.SetAdditionalBufferSize(1024 * 4);

    // ダブルバッファで運用する場合は、SetMultiBufferQuantity で 2 を指定し、
    // プリミティブレンダラ内部のバッファをダブルにしたうえで、
    // g_pPrimitiveRenderer->Update(); で利用するバッファを選択( 0 -> 1 -> 0 -> 1 )する
    info.SetMultiBufferQuantity(2);

    // PrimitiveRendererのインスタンス
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)
        GetRenderer((void*)nns::gfx::PrimitiveRenderer::CreateRenderer(pFramework->GetDevice(), info));
    pRenderer->SetScreenWidth(DisplayWidth);
    pRenderer->SetScreenHeight(DisplayHeight);
}

void Graphics::FinalizePrimitiveRenderer() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)GetRenderer();

    nns::gfx::PrimitiveRenderer::DestroyRenderer(pRenderer, pFramework->GetDevice(), DeallocateFunction, &m_PrimitiveRendererAllocator);
    m_PrimitiveRendererAllocator.Finalize();
    free(m_pPrimitiveRendererAllocatorMemory);
}

//------------------------------------------------------------------------------
// ユーザーピクセルシェーダ
//------------------------------------------------------------------------------

#define SHADER_SOURCE( ... ) #__VA_ARGS__
// モザイクシェーダ
static const char MOSAIC_PS_SOURCE[] =
"#version 430 \n"
SHADER_SOURCE(

    layout(std140, binding = 1) uniform Model
{
    uniform mat4 u_userMatrix;
    uniform vec4 u_color0;
    uniform vec4 u_color1;
    uniform vec2 u_uv_src;
    uniform vec2 u_uv_size;
    uniform vec4 u_layer;
    float rate;
};

// ユーザー定義のコンスタントバッファ
layout(std140, binding = 2) uniform TextureParam
{
    uniform vec4        u_mul_color;
};

layout(binding = 0) uniform sampler2D texture0;

layout(location = 0) in vec4 v_texCoord;
layout(location = 1) in vec4 v_color;

out vec4 o_Color;

void main()
{
    vec4 out_color = vec4(0);
    vec4 color0 = textureLod(texture0, v_texCoord.xy, 0.0);
    o_Color = vec4(1 - color0.x, 1 - color0.y, 1 - color0.z, color0.w) * u_mul_color;
}
);

// カスタムシェーダで使用するコンスタントバッファの構造体
struct PixelSize
{
    nn::util::Float2 u_pixel_size;
};

//---------------------------------------------------------------
// ユーザーシェーダの初期化
//---------------------------------------------------------------
nn::gfx::Shader g_PixelShader;
void Graphics::InitializeUserShader() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();

    nn::gfx::Shader::InfoType info;
    info.SetDefault();
    info.SetSeparationEnabled(true);

    nn::gfx::ShaderCode psCode;
    psCode.codeSize = static_cast<uint32_t>(strlen(MOSAIC_PS_SOURCE));
    psCode.pCode = MOSAIC_PS_SOURCE;

    info.SetShaderCodePtr(nn::gfx::ShaderStage_Pixel, &psCode);
    info.SetSourceFormat(nn::gfx::ShaderSourceFormat_Glsl);
    info.SetCodeType(nn::gfx::ShaderCodeType_Source);
    nn::gfx::ShaderInitializeResult result = g_PixelShader.Initialize(pFramework->GetDevice(), info);
    NN_SDK_ASSERT(result == nn::gfx::ShaderInitializeResult_Success, "InitializeShader() failed (%d)\n", result);
    NN_UNUSED(result);
}
void Graphics::FinalizeUserShader() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)GetFramework();

    // ユーザーシェーダ破棄
    g_PixelShader.Finalize(pFramework->GetDevice());
}
