﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_SysFileSystem.h"

NN_ALIGNAS(4096) char g_WorkBuffer[8 * 1024 * 1024];
nn::mem::StandardAllocator g_Allocator;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return g_Allocator.Allocate(size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    g_Allocator.Free(p);
}

FileSystem& FileSystem::GetInstance() NN_NOEXCEPT
{
    static FileSystem instance;
    return instance;
}

void FileSystem::Initialize() NN_NOEXCEPT
{
    g_Allocator.Initialize(g_WorkBuffer, sizeof(g_WorkBuffer));

    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));
    m_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(m_MountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("rom", m_MountRomCacheBuffer, cacheSize)
    );
}

void FileSystem::Finalize() NN_NOEXCEPT
{
    nn::fs::Unmount("rom");

    delete[] m_MountRomCacheBuffer;
    m_MountRomCacheBuffer = nullptr;
}

void FileSystem::MountUserSaveData(int userIndex) NN_NOEXCEPT
{
    // アカウントライブラリを初期化します。
    nn::account::Initialize();

    nn::Result result;

    // 例として、リストの先頭のユーザーの識別子を利用します。
    // 実際はユーザーによる選択操作が必要です。 (参照: アカウントガイド)
    nn::account::Uid user = nn::account::InvalidUid;
    int userCount = 0;
    result = nn::account::ListAllUsers(&userCount, &user, 1);
    NN_ASSERT(result.IsSuccess() && userCount > 0);

    // 選択したユーザーのセーブデータを作成します。
    // 既存の場合、何もせず ResultSuccess が返ります。
    result = nn::fs::EnsureSaveData(user);
    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // EnsureSaveData() について、アプリケーションは容量不足のエラーハンドリングを行う必要があります。
        // セーブデータ作成のための容量が不足している旨のメッセージは、システムが自動的にエラービューアで表示します。
        // アプリケーションは、アカウント選択操作をキャンセルして前のシーンに戻るなどの対応をする必要があります。
        NN_ABORT("Usable space not enough.\n");
    }
    // 上記以外の原因で失敗した場合はライブラリ内でアボートするため、
    // これ以上のエラーハンドリングは不要です。

    // マウント名 "save" としてセーブデータをマウントします。
    result = nn::fs::MountSaveData("save", user);
    // 失敗した際は必ずアボートしてください。
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

bool FileSystem::IsUserSaveAvailable() NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType directoryEntryType;
    nn::Result result = nn::fs::GetEntryType(&directoryEntryType, SaveFilePath);
    if (nn::fs::ResultPathNotFound().Includes(result))
    {
        return false;
    }

    return true;
}

void FileSystem::InitializeUserSaveData(char* initData, size_t dataSize) NN_NOEXCEPT
{
    nn::Result result = nn::fs::CreateFile(SaveFilePath, dataSize);
    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // セーブデータのデータ保存領域が不足しています。
        NN_ABORT("Usable space not enough.\n");
        return;
    }

    WriteUserSaveData(initData,dataSize);
}

void FileSystem::ReadUserSaveData(char* readData, size_t dataSize) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;

    // ファイルをオープンします
    result = nn::fs::OpenFile(&fileHandle, SaveFilePath, nn::fs::OpenMode_Read);
    if (nn::fs::ResultPathNotFound::Includes(result))
    {
        // 対象ファイルが存在しません。
        NN_ABORT("Save file not found.\n");
    }
    else if (nn::fs::ResultTargetLocked::Includes(result))
    {
        // 対象ファイルが既にオープンされています。
        NN_ABORT("Save file already opened.\n");
    }

    // ファイルを読み込みます。
    size_t readSize;
    // ファイル読み込み失敗時はライブラリ内でアボートするため、エラーハンドリングは不要です。
    (void)nn::fs::ReadFile(&readSize, fileHandle, 0, readData, dataSize);
    NN_ASSERT_EQUAL(readSize, dataSize);

    nn::fs::CloseFile(fileHandle);
}

void FileSystem::WriteUserSaveData(char* fileData, size_t dataSize) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;

    // ファイルをオープンします
    result = nn::fs::OpenFile(&fileHandle, SaveFilePath, nn::fs::OpenMode_Write);
    if (nn::fs::ResultPathNotFound::Includes(result))
    {
        // 対象ファイルが存在しません。
        NN_ABORT("Save file not found.\n");
    }
    else if (nn::fs::ResultTargetLocked::Includes(result))
    {
        // 対象ファイルが既にオープンされています。
        NN_ABORT("Save file already opened.\n");
    }

    // ファイルに書き込みます。
    result = nn::fs::WriteFile(fileHandle, 0, fileData, dataSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if (nn::fs::ResultUsableSpaceNotEnough::Includes(result))
    {
        // セーブデータのデータ保存領域が不足しています。
        NN_ABORT("Usable space not enough.\n");
    }

    nn::fs::CloseFile(fileHandle);

    // セーブデータの更新内容をコミットします。
    (void)nn::fs::CommitSaveData("save");
}
