﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_AppSceneManager.h"
#include "ShopMonitoringTool_AppColorPalette.h"

SceneManager::SceneManager()
{
    m_pShopCommandManager = new ShopCommandManager();
    m_pExecutor = new CommandExecutor();
    m_pLogView = new LogScene();
    m_pTopmenu = new TopMenu("TopMenu");
    m_pConfigMenu = new ConfigMenu("ConfigMenu");
}

SceneManager::~SceneManager()
{
    for (auto pair : m_ItemList)
    {
        delete m_ItemList[pair.first];
    }

    for (auto pair : m_CommandList)
    {
        delete m_CommandList[pair.first];
    }

    delete m_pShopCommandManager;
    delete m_pExecutor;
    delete m_pLogView;
    delete m_pTopmenu;
    delete m_pConfigMenu;
}

void SceneManager::Initialize()
{
    // 初期化
    nn::nifm::Initialize();
    nn::socket::Initialize(m_SocketConfigWithMemory);
    nn::ssl::Initialize();
    nn::account::Initialize();
    gFileSystem.Initialize();
    gInput.Initialize();
    gGraphics.Initialize();
    gLog.Initialize();
    gKeyboard.Initialize(128 * 1024);
    gDrawer.InitializeTextureResource("rom:/texture/btn.bntx");
    gAudio.Initialize();


    //環境識別子取得
    nn::nsd::GetEnvironmentIdentifier(&m_EnvironmentIdentifier);

    // デフォルト設定
    DefaultConfigData.errorCountLimit = 5;
    DefaultConfigData.autoRestartWaitTime = 60;
    DefaultConfigData.autoRetryWaitTime = 5;
    DefaultConfigData.asyncTimeout = 20;
    DefaultConfigData.userIndex = 0;
    DefaultConfigData.isEnableAutoRestart = true;
    DefaultConfigData.isEnableErrorApplet = false;
    std::strcpy(DefaultConfigData.demoApplicationId, "010083E002F54000");
    std::strcpy(DefaultConfigData.ticketApplicationId, "010083E002F54000");
    std::strcpy(DefaultConfigData.ticketKeyId, "0000000000000000");
    std::strcpy(DefaultConfigData.purchaseNsUid, "70010000000541");

    // セーブデータ
    gFileSystem.MountUserSaveData(0);
    if (!gFileSystem.IsUserSaveAvailable())
    {
        APPLOG_INFO("User save data is not available.\n");
        APPLOG_INFO("Create user save data...\n");
        gFileSystem.InitializeUserSaveData((char*)&DefaultConfigData,sizeof(DefaultConfigData));
    }
    gFileSystem.ReadUserSaveData((char*)&m_ConfigData, sizeof(m_ConfigData));

    // メニューシーンの構築
    ConstructAppScene();
}

bool SceneManager::SaveConfigData()
{
    m_ConfigData.errorCountLimit = *(int*)m_ItemList["Config_ErrorLimit"]->GetValue();
    m_ConfigData.autoRestartWaitTime = *(int*)m_ItemList["Config_AutoRestartWaitTime"]->GetValue();
    m_ConfigData.autoRetryWaitTime = *(int*)m_ItemList["Config_AutoRetryWaitTime"]->GetValue();
    m_ConfigData.asyncTimeout = *(int*)m_ItemList["Config_TimeoutTime"]->GetValue();
    m_ConfigData.userIndex = *(int*)m_ItemList["Config_UserIndex"]->GetValue();
    m_ConfigData.isEnableAutoRestart = *(bool*)m_ItemList["Config_EnableAutoRestart"]->GetValue();
    m_ConfigData.isEnableErrorApplet = *(bool*)m_ItemList["Config_EnableErrApplet"]->GetValue();

    std::strcpy(m_ConfigData.demoApplicationId, (*(std::string*)m_ItemList["Config_DemoId"]->GetValue()).c_str() );
    std::strcpy(m_ConfigData.ticketApplicationId, (*(std::string*)m_ItemList["Config_TicketAppId"]->GetValue()).c_str());
    std::strcpy(m_ConfigData.ticketKeyId, (*(std::string*)m_ItemList["Config_TicketKeyId"]->GetValue()).c_str());
    std::strcpy(m_ConfigData.purchaseNsUid, (*(std::string*)m_ItemList["Config_PurchaseNsuId"]->GetValue()).c_str());

    gFileSystem.WriteUserSaveData((char*)&m_ConfigData, sizeof(m_ConfigData));

    APPLOG_INFO("Config saved\n");

    return true;
}

void SceneManager::Finalize()
{
    gDrawer.FinalizeTextureResource();
    gAudio.Finalize();
    gGraphics.Finalize();
    gKeyboard.Finalize();
    gFileSystem.Finalize();
    nn::ssl::Finalize();
    nn::socket::Finalize();
}

void SceneManager::ConstructAppScene()
{
    // コマンドリストの生成
    m_CommandList["RegDevAcc"]     = new ExecuteItem("RegisterDeviceAccount","ECI にデバイスアカウントを登録し、得られたデバイスアカウント情報をデバイスに保存します");
    m_CommandList["UnregDevAcc"]   = new ExecuteItem("UnregisterDeviceAccount","デバイスアカウント情報をデバイスから削除し、ECI のデバイスアカウントを抹消します");
    m_CommandList["DevRegInfo"]    = new ExecuteItem("DeviceRegistrationInfo","ECI からデバイスアカウントの登録情報を取得します");
    m_CommandList["DevAccStats"]   = new ExecuteItem("DeviceAccountStatus","ECI からデバイスアカウントの状態を取得します");
    m_CommandList["DevAccInfo"]    = new ExecuteItem("DeviceAccountInfo","デバイスに保存されているデバイスアカウント情報を取得します");
    m_CommandList["DevAuthToken"]  = new ExecuteItem("DeviceAuthenticationToken","NAS からショップ用のデバイス認証トークンを取得します");
    m_CommandList["ShopAccStats"]  = new ExecuteItem("ShopAccountStatus","ショップアカウントの情報を取得します");
    m_CommandList["DevLinkStats"]  = new ExecuteItem("DeviceLinkStatus", "機器認証の状態をサーバーから取得します");
    m_CommandList["HasDevLink"]    = new ExecuteItem("HasDeviceLink","機器認証の状態をデバイスから取得します");
    m_CommandList["DelAllRight"]   = new ExecuteItem("DeleteAllRights","サーバー上で管理される全ての権利チケットの情報を抹消します");
    m_CommandList["PrePurchase"]   = new ExecuteItem("Prepurchase","タイトル購入前情報を取得します");
    m_CommandList["Purchase"]      = new ExecuteItem("Purchase","タイトル購入を行います");
    m_CommandList["DownDemo"]      = new ExecuteItem("DownloadDemo","体験版ダウンロードを行います");
    m_CommandList["DownTicket"]    = new ExecuteItem("DownloadTicket","チケットダウンロードを行います");
    m_CommandList["LinkDev"]       = new ExecuteItem("LinkDevice","機器認証を登録します");
    m_CommandList["UnlinkDev"]     = new ExecuteItem("UnlinkDevice","機器認証を解除します");
    m_CommandList["UnlinkDevAll"]  = new ExecuteItem("UnlinkDeviceAll","機器認証を全て解除します");
    m_CommandList["SyncTicket"]    = new ExecuteItem("SyncTicket","チケット同期を行います");
    m_CommandList["Search"]        = new ExecuteItem("Search","タイトル検索を行います");
    m_CommandList["Start"]         = new ExecuteItem("ShopStart", "ログイン処理を行います");

    // コマンドリストへのアタッチ
    m_CommandList["RegDevAcc"]->function       = std::bind(&ShopCommandManager::RegisterDeviceAccount, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["UnregDevAcc"]->function     = std::bind(&ShopCommandManager::UnregisterDeviceAccount, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DevRegInfo"]->function      = std::bind(&ShopCommandManager::DeviceRegistrationInfo, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DevAccStats"]->function     = std::bind(&ShopCommandManager::DeviceAccountStatus, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DevAccInfo"]->function      = std::bind(&ShopCommandManager::DeviceAccountInfo, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DevAuthToken"]->function    = std::bind(&ShopCommandManager::DeviceAuthenticationToken, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["ShopAccStats"]->function    = std::bind(&ShopCommandManager::ShopAccountStatus, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DevLinkStats"]->function    = std::bind(&ShopCommandManager::DeviceLinkStatus, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["HasDevLink"]->function      = std::bind(&ShopCommandManager::HasDeviceLink, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DelAllRight"]->function     = std::bind(&ShopCommandManager::DeleteAllRights, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["PrePurchase"]->function     = std::bind(&ShopCommandManager::Prepurchase, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["Purchase"]->function        = std::bind(&ShopCommandManager::Purchase, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DownDemo"]->function        = std::bind(&ShopCommandManager::DownloadDemo, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["DownTicket"]->function      = std::bind(&ShopCommandManager::DownloadTicket, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["LinkDev"]->function         = std::bind(&ShopCommandManager::LinkDevice, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["UnlinkDev"]->function       = std::bind(&ShopCommandManager::UnlinkDevice, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["UnlinkDevAll"]->function    = std::bind(&ShopCommandManager::UnlinkDeviceAll, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["SyncTicket"]->function      = std::bind(&ShopCommandManager::SyncTicket, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["Search"]->function          = std::bind(&ShopCommandManager::Search, std::ref(*m_pShopCommandManager), std::placeholders::_1);
    m_CommandList["Start"]->function           = std::bind(&ShopCommandManager::Start, std::ref(*m_pShopCommandManager), std::placeholders::_1);


    // TopMenu のセットアップ
    m_ItemList["LinkToMonitoring"] = new MenuItem("Monitoring","自動監視を行います");
    m_ItemList["LinkToMonitoring"]->SetNextScene(m_pExecutor);
    m_ItemList["LinkToMonitoring"]->AtatchedGeneralFunction = std::bind(&SceneManager::SetupMonitoringScene, std::ref(*this));

    m_ItemList["LinkToShopCommand"] = new MenuItem("Command", "各処理を個別に実行します");
    m_ItemList["LinkToShopCommand"]->SetNextScene(m_pExecutor);
    m_ItemList["LinkToShopCommand"]->AtatchedGeneralFunction = std::bind(&SceneManager::SetupCommandScene, std::ref(*this));

    m_ItemList["LinkToConfig"] = new MenuItem("Config","ツールの設定を行います");
    m_ItemList["LinkToConfig"]->SetNextScene(m_pConfigMenu);

    m_pTopmenu->SetDrawPos(50, 100);
    m_pTopmenu->SetState(Scene::State::EnabledActive);
    m_pTopmenu->PushItem(m_ItemList["LinkToMonitoring"]);
    m_pTopmenu->PushItem(m_ItemList["LinkToShopCommand"]);
    m_pTopmenu->PushItem(m_ItemList["LinkToConfig"]);
    m_pTopmenu->SetupScene();


    // ConfigMenu のセットアップ
    const int errorCountLimitMax = 8;
    const int errorCountLimitMin = 1;
    const int waitTimeMax = 300;
    const int waitTimeMin = 0;
    const int asyncTimeoutMax = 60;
    const int asyncTimeoutMin = 1;
    const int userIndexMax = 7;
    const int userIndexMin = 0;

    m_ItemList["Config_ErrorLimit"]  = new IntegerItem("Error count limit", m_ConfigData.errorCountLimit, errorCountLimitMax, errorCountLimitMin);
    m_ItemList["Config_AutoRestartWaitTime"]    = new IntegerItem("Auto restart wait time (sec)", m_ConfigData.autoRestartWaitTime, waitTimeMax, waitTimeMin);
    m_ItemList["Config_AutoRetryWaitTime"] = new IntegerItem("Auto retry wait time (sec)", m_ConfigData.autoRetryWaitTime, waitTimeMax, waitTimeMin);
    m_ItemList["Config_EnableAutoRestart"]  = new BooleanItem("Enable auto restart", m_ConfigData.isEnableAutoRestart);
    m_ItemList["Config_EnableErrApplet"] = new BooleanItem("Enable show error applet", m_ConfigData.isEnableErrorApplet);
    m_ItemList["Config_TimeoutTime"] = new IntegerItem("Async timeout time (sec)", m_ConfigData.asyncTimeout, asyncTimeoutMax, asyncTimeoutMin);
    m_ItemList["Config_UserIndex"] = new IntegerItem("User Index", m_ConfigData.userIndex, userIndexMax, userIndexMin);
    m_ItemList["Config_DemoId"]      = new StringItem("DownloadDemo applicatoinID", m_ConfigData.demoApplicationId);
    m_ItemList["Config_TicketAppId"]      = new StringItem("DownloadTicket applicationID", m_ConfigData.ticketApplicationId);
    m_ItemList["Config_TicketKeyId"] = new StringItem("DownloadTicket keyID", m_ConfigData.ticketKeyId);
    m_ItemList["Config_PurchaseNsuId"] = new StringItem("Purchase NSUID", m_ConfigData.purchaseNsUid);

    m_ItemList["Config_ErrorLimit"]->SetDescription("警報に至るまでの連続エラー回数です");
    m_ItemList["Config_AutoRestartWaitTime"]->SetDescription("自動ループの待機時間です");
    m_ItemList["Config_AutoRetryWaitTime"]->SetDescription("エラー時にリトライするまでの待機時間です");
    m_ItemList["Config_EnableAutoRestart"]->SetDescription("自動ループを有効にします");
    m_ItemList["Config_EnableErrApplet"]->SetDescription("エラーアプレットを利用したエラー表示を有効にします");
    m_ItemList["Config_TimeoutTime"]->SetDescription("一部の非同期処理におけるタイムアウトの時間です");
    m_ItemList["Config_UserIndex"]->SetDescription("使用するユーザーアカウントのインデックス番号です");
    m_ItemList["Config_DemoId"]->SetDescription("体験版ダウンロードを行うタイトルのApplicationIDです");
    m_ItemList["Config_TicketAppId"]->SetDescription("チケットダウンロードを行うタイトルのApplicationIDです");
    m_ItemList["Config_TicketKeyId"]->SetDescription("チケットダウンロードを行うタイトルのKeyIDです");
    m_ItemList["Config_PurchaseNsuId"]->SetDescription("購入確認を行うタイトルのNSUIDです");

    m_ItemList["Header_ExecuteConfig"] = new MenuItem("ExecuteConfig");
    m_ItemList["Header_ExecuteConfig"]->SetSelectable(false);
    m_ItemList["Header_ShopCommandConfig"] = new MenuItem("ShopCommandConfig");
    m_ItemList["Header_ShopCommandConfig"]->SetSelectable(false);

    m_ItemList["SaveConfig"] = new MenuItem("Save");
    m_ItemList["SaveConfig"]->SetDescription("現在の設定をセーブデータに保存します");
    m_ItemList["SaveConfig"]->AtatchedGeneralFunction = std::bind(&SceneManager::SaveConfigData, std::ref(*this));

    m_pConfigMenu->SetDrawPos(350, 70);
    m_pConfigMenu->PushItem(m_ItemList["Header_ExecuteConfig"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_ErrorLimit"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_EnableErrApplet"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_EnableAutoRestart"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_AutoRestartWaitTime"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_AutoRetryWaitTime"]);
    m_pConfigMenu->PushItem(m_ItemList["Header_ShopCommandConfig"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_UserIndex"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_TimeoutTime"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_DemoId"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_TicketAppId"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_TicketKeyId"]);
    m_pConfigMenu->PushItem(m_ItemList["Config_PurchaseNsuId"]);
    m_pConfigMenu->PushItem(m_ItemList["SaveConfig"]);
    m_pConfigMenu->SetupScene();


    // その他メニューのセットアップ
    m_pExecutor->SetDrawPos(350, 100);
    m_pLogView->SetDrawPos(20, 50);

    // ShopCommandManagerのセットアップ
    SetupShopCommandManager();
} // NOLINT (readability/fn_size)

void SceneManager::SetupShopCommandManager()
{
    m_pShopCommandManager->SetUserIndex(*(int*)m_ItemList["Config_UserIndex"]->GetValue());
    m_pShopCommandManager->SetDownloadDemoApplicationIdString(*(std::string*)m_ItemList["Config_DemoId"]->GetValue());
    m_pShopCommandManager->SetDownloadTicketApplicationIdString(*(std::string*)m_ItemList["Config_TicketAppId"]->GetValue());
    m_pShopCommandManager->SetDownloadTicketKeyIdString(*(std::string*)m_ItemList["Config_TicketKeyId"]->GetValue());
    m_pShopCommandManager->SetPurchaseNsUidString(*(std::string*)m_ItemList["Config_PurchaseNsuId"]->GetValue());
    m_pShopCommandManager->SetTimeOut(nn::TimeSpan::FromSeconds(*(int*)m_ItemList["Config_TimeoutTime"]->GetValue()));
}

void SceneManager::UpdateConfigDependance()
{
    // 設定間の依存関係を適用
    m_ItemList["Config_AutoRestartWaitTime"]->SetEditable(*(bool*)m_ItemList["Config_EnableAutoRestart"]->GetValue());
}

bool SceneManager::SetupCommandScene()
{
    SetupShopCommandManager();

    m_pExecutor->SetupScene();
    m_pExecutor->SetExecutionMode(ExecutorScene::ExecuteMode::Selective);
    m_pExecutor->SetEnablShowErrorApplet(*(bool*)m_ItemList["Config_EnableErrApplet"]->GetValue());
    m_pExecutor->ResetCursor();

    m_pExecutor->PushTask(m_CommandList["RegDevAcc"]);
    m_pExecutor->PushTask(m_CommandList["UnregDevAcc"]);
    m_pExecutor->PushTask(m_CommandList["DevRegInfo"]);
    m_pExecutor->PushTask(m_CommandList["DevAccStats"]);
    m_pExecutor->PushTask(m_CommandList["DevAccInfo"]);
    m_pExecutor->PushTask(m_CommandList["DevAuthToken"]);
    m_pExecutor->PushTask(m_CommandList["ShopAccStats"]);
    m_pExecutor->PushTask(m_CommandList["DevLinkStats"]);
    m_pExecutor->PushTask(m_CommandList["HasDevLink"]);
    m_pExecutor->PushTask(m_CommandList["DelAllRight"]);
    m_pExecutor->PushTask(m_CommandList["PrePurchase"]);
    m_pExecutor->PushTask(m_CommandList["Purchase"]);
    m_pExecutor->PushTask(m_CommandList["DownDemo"]);
    m_pExecutor->PushTask(m_CommandList["DownTicket"]);
    m_pExecutor->PushTask(m_CommandList["LinkDev"]);
    m_pExecutor->PushTask(m_CommandList["UnlinkDev"]);
    m_pExecutor->PushTask(m_CommandList["UnlinkDevAll"]);
    m_pExecutor->PushTask(m_CommandList["SyncTicket"]);
    m_pExecutor->PushTask(m_CommandList["Search"]);
    m_pExecutor->PushTask(m_CommandList["Start"]);

    return true;
}

bool SceneManager::SetupMonitoringScene()
{
    SetupShopCommandManager();

    m_pExecutor->SetupScene();
    m_pExecutor->SetExecutionMode(ExecutorScene::ExecuteMode::Continuous);
    m_pExecutor->SetErrorCountLimit(*(int*)m_ItemList["Config_ErrorLimit"]->GetValue());
    m_pExecutor->SetAutoRestartWaitTime(*(int*)m_ItemList["Config_AutoRestartWaitTime"]->GetValue());
    m_pExecutor->SetAutoRetryWaitTime(*(int*)m_ItemList["Config_AutoRetryWaitTime"]->GetValue());
    m_pExecutor->SetEnableAutoRestart(*(bool*)m_ItemList["Config_EnableAutoRestart"]->GetValue());
    m_pExecutor->SetEnablShowErrorApplet(*(bool*)m_ItemList["Config_EnableErrApplet"]->GetValue());

    // 一巡の自動監視で呼ばれるコマンドとその順番
    m_pExecutor->PushTask(m_CommandList["Start"]);
    m_pExecutor->PushTask(m_CommandList["Purchase"]);
    m_pExecutor->PushTask(m_CommandList["DownDemo"]);
    m_pExecutor->PushTask(m_CommandList["DownTicket"]);
    m_pExecutor->PushTask(m_CommandList["SyncTicket"]);
    m_pExecutor->PushTask(m_CommandList["UnlinkDevAll"]);
    m_pExecutor->PushTask(m_CommandList["UnregDevAcc"]);

    return true;
}

void SceneManager::DrawStatusBar()
{
    const int screenWidth = 1280;
    const int screenHeight = 720;

    const int titleBarMarginW = 20;
    const int titleBarMarginH = 50;
    gDrawer.DrawLine(titleBarMarginW, titleBarMarginH, screenWidth - titleBarMarginW, titleBarMarginH, AppColor::DefaultTextColor);

    const int titleTextMarginW = 30;
    const int titleTextMarginH = 15;
    gWriter.SetTextColor(AppColor::DefaultTextColor);
    gWriter.SetCursor(titleTextMarginW, titleTextMarginH);
    gWriter.Print("ShopMonitoringTool Ver 1.0.1");

    {
        int stringWidth = gWriter.CalculateStringWidth("Environment : %s", m_EnvironmentIdentifier.value);
        gWriter.SetCursor(screenWidth - titleTextMarginW - stringWidth, titleTextMarginH);
        gWriter.Print("Environment : ");
        gWriter.SetTextColor(AppColor::ValueColor);
        gWriter.Print("%s", m_EnvironmentIdentifier.value);
    }

    const int footerBarMarginW = 20;
    const int footerBarMarginH = screenHeight - 50;
    gDrawer.DrawLine(footerBarMarginW, footerBarMarginH, screenWidth - footerBarMarginW, footerBarMarginH, AppColor::DefaultTextColor);

    const int footerTextMarginW = 30;
    const int footerTextMarginH = screenHeight - 15 - 20;

    const int iconSize = 32;
    const int textMargin = 10;
    const int descMargin = 30;

    const int descCount = 4;
    const int hOff = -5;

    int strWidth[descCount];

    // ボタン表示
    strWidth[0] = gWriter.CalculateStringWidth("Enter");
    strWidth[1] = gWriter.CalculateStringWidth("Cancel");
    strWidth[2] = gWriter.CalculateStringWidth("Log   ");
    strWidth[3] = gWriter.CalculateStringWidth("Select");
    {
        int sum;
        sum = strWidth[0] + strWidth[1] + strWidth[0] + strWidth[0] + iconSize * 4 + textMargin + descMargin * 3 + footerTextMarginW;

        gWriter.SetTextColor(AppColor::DefaultTextColor);
        gDrawer.DrawTexture(2, screenWidth - footerTextMarginW - sum, footerTextMarginH + hOff, iconSize, iconSize, AppColor::DefaultTextColor);
        sum -= iconSize + textMargin;
        gWriter.SetCursor(screenWidth - footerTextMarginW - sum, footerTextMarginH);
        gWriter.Print("Log");
        sum -= descMargin + strWidth[2];
        gDrawer.DrawTexture(3, screenWidth - footerTextMarginW - sum, footerTextMarginH + hOff, iconSize, iconSize, AppColor::DefaultTextColor);
        sum -= iconSize + textMargin;
        gWriter.SetCursor(screenWidth - footerTextMarginW - sum, footerTextMarginH);
        gWriter.Print("Select");
        sum -= descMargin + strWidth[3];
        gDrawer.DrawTexture(1, screenWidth - footerTextMarginW - sum, footerTextMarginH + hOff, iconSize, iconSize, AppColor::DefaultTextColor);
        sum -= iconSize + textMargin;
        gWriter.SetCursor(screenWidth - footerTextMarginW - sum, footerTextMarginH);
        gWriter.Print("Cancel");
        sum -= descMargin + strWidth[1];
        gDrawer.DrawTexture(0, screenWidth - footerTextMarginW - sum, footerTextMarginH + hOff, iconSize, iconSize, AppColor::DefaultTextColor);
        sum -= iconSize + textMargin;
        gWriter.SetCursor(screenWidth - footerTextMarginW - sum, footerTextMarginH);
        gWriter.Print("Enter");
    }


    // Description表示
    if (!m_IsEnableLogView)
    {
        const nn::util::Color4u8Type descColor = { { 200, 200, 200, 120 } };
        gWriter.SetTextColor(descColor);
        gWriter.SetCursor(footerTextMarginW, footerBarMarginH - 30);
        gWriter.Print(m_CurrentSceneDescription.c_str());
    }
}

void SceneManager::MainLoop()
{
    for(;;)
    {
        gInput.Update();
        gAudio.Update();
        gGraphics.BeginDraw();
        m_pTopmenu->Update();

        if (gInput.isHld(nn::hid::NpadButton::R::Mask))
        {
            m_IsEnableLogView = true;
            if (gInput.isTrg(nn::hid::NpadButton::R::Mask))
            {
                m_pLogView->DisplayRangeReset();
                m_pLogView->SetEnabledAutoScroll(true);
            }
            m_pLogView->Update();
            m_pLogView->Control();
            m_pLogView->Draw();
        }
        else
        {
            m_IsEnableLogView = false;
            m_pTopmenu->Control();
            UpdateConfigDependance();
            m_pTopmenu->Draw();
        }

        m_CurrentSceneDescription = m_pTopmenu->GetDescription();
        DrawStatusBar();
        gGraphics.EndDraw();

        // 終了
        if (gInput.isTrg(nn::hid::NpadButton::Minus::Mask))
        {
            break;
        }
    }
}
