﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <vector>
#include <glv.h>
#include <glv_viewcontainer.h>

#include "QCIT_Image.h"

namespace qcit
{

    struct CommonValue
    {
        static const float         InitialFontSize;  //!< 既定のフォントサイズです。
    };

class RootSurfaceContext;   //!< 前方宣言です。

/**
 * @brief       モーダルビューです。
 */
class ModalView : public glv::ViewContainer
{
    NN_DISALLOW_COPY( ModalView );
    NN_DISALLOW_MOVE( ModalView );

public:
    /**
     * @brief       コンストラクタです。
     */
    explicit ModalView(const glv::Rect& rect) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~ModalView() NN_NOEXCEPT NN_OVERRIDE {}

    /**
     * @brief       タイムアウト時間を設定します。
     *
     * @param[in]   timeout     タイムアウト
     *
     * @return      自身のインスタンスを返します。
     */
    ModalView& SetTimeout(nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief       タイムアウト時間を延長します。
     *
     * @param[in]   timeout     タイムアウト
     *
     * @return      自身のインスタンスを返します。
     */
    ModalView& ExtendTimeout(nn::TimeSpan timeout) NN_NOEXCEPT;

    /**
     * @brief       モーダル処理を開始してからタイムアウト時間を経過したかを判定します。
     *
     * @return      表示されてからタイムアウト時間を経過した場合は真を返します。
     *              モーダル処理が行われていない場合は、必ず真を返します。
     */
    bool IsExpired() const NN_NOEXCEPT;

    /**
     * @brief       有効なタイムアウト時間が設定されているか判定します。
     *
     * @return      有効なタイムアウト時間が設定されている場合は真を返します。
     */
    bool IsTimeoutValid() const NN_NOEXCEPT;

protected:
    /**
     * @brief       アクティブ処理です。必要の応じてオーバーライドしてください。
     *
     * @details     モーダルビューが表示になった直後に呼び出されます。
     *              呼び出された同じフレームの OnLoopBeforeSceneRenderer() が呼ばれない可能性があります。
     */
    virtual void OnActivate() NN_NOEXCEPT {}

    /**
     * @brief       ディアクティブ処理です。必要の応じてオーバーライドしてください。
     *
     * @details     モーダルビューが非表示になる直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() が呼ばれない可能性があります。
     */
    virtual void OnDeactivate() NN_NOEXCEPT {}

    /**
     * @brief       glv シーンレンダラ前のループ処理です。必要の応じてオーバーライドしてください。
     *
     * @details     glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     *              この時点ではまだ glv コンテキストのレンダリングは開始していません。
     *              このメソッドが呼び出されるフレームは OnLoopAfterSceneRenderer() と同じです。
     *              ページが表示されている場合は、先にページの OnLoopBeforeSceneRenderer() が呼ばれます。
     */
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT {}

    /**
     * @brief       glv シーンレンダラ後のループ処理です。必要の応じてオーバーライドしてください。
     *
     * @details     glv シーンレンダラのレンダリングが終わった後に呼び出されます。
     *              このメソッドが呼び出されるフレームは OnLoopBeforeSceneRenderer() と同じです。
     *              ページが表示されている場合は、先にページの OnLoopAfterSceneRenderer() が呼ばれます。
     */
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT {}

    /**
     * @brief       モーダル処理を継続するかを判定します。
     *
     * @return      モーダル処理を継続する場合は真を返します。
     */
    bool IsRunnable() const NN_NOEXCEPT { return (m_IsRunning != false) && (m_IsExiting == false); }

    /**
     * @brief       モーダル処理を終了させます。
     */
    void ExitModal() NN_NOEXCEPT { m_IsExiting = (m_IsRunning != false) ? true : false; }

private:
    friend class InvisibleWall;

    /**
     * @brief       モーダル処理を開始します。
     */
    void StartModal() NN_NOEXCEPT;

    /**
     * @brief       モーダル処理を終了します。
     */
    void EndModal() NN_NOEXCEPT;

private:
    nn::TimeSpan    m_Timeout;
    nn::os::Tick    m_BeginTick;
    bool            m_IsRunning;
    bool            m_IsExiting;
};

/**
 * @brief       メッセージビューです。
 */
class MessageView : public ModalView
{
    NN_DISALLOW_COPY( MessageView );
    NN_DISALLOW_MOVE( MessageView );

public:
    /**
     * @brief       ボタンが選択された際に呼び出される関数の定義です。
     *
     * @param[in]   pParam      ボタン追加時に指定した引数
     * @param[in]   timespan    関数呼び出した後にモーダル状態を継続する時間
     */
    typedef std::function< void( void* pParam, nn::TimeSpan& timespan ) > ActionFunc;

    /**
     * @brief       コンストラクタです。
     */
    MessageView() NN_NOEXCEPT;

    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   isClosable  B ボタン押下で閉じる操作を行う場合は真を指定します。
     */
    explicit MessageView(bool isClosable, bool isCenteringButton = false) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~MessageView() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       子孫要素を全て削除します。
     */
    void Remove() NN_NOEXCEPT;

    /**
     * @brief       メッセージを追加します。
     *
     * @param[in]   message     メッセージ
     */
    MessageView& AddMessage(const std::string& message) NN_NOEXCEPT;

    MessageView& AddMessage(const std::string& message, float fontSize) NN_NOEXCEPT;

    /**
     * @brief       メッセージを追加します。 TORIAEZU: 名前は暫定, 不要なら後で削除する
     *
     * @param[in]   message     メッセージ
     */
    glv::Label* AddMessageReturnLabel(const std::string& message) NN_NOEXCEPT;

    /**
     * @brief       メッセージを追加します。
     *
     * @param[in]   message     メッセージ
     */
    MessageView& AddMessage(const glv::WideString& message) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。一番最後に追加したボタンがフォーカスされます。
     *
     * @param[in]   caption     ボタンのキャプション
     */
    MessageView& AddButton(const std::string& caption) NN_NOEXCEPT { return AddButton(caption, nullptr, nullptr); }

    /**
     * @brief       ボタンを追加します。一番最後に追加したボタンがフォーカスされます。
     *
     * @param[in]   caption     ボタンのキャプション
     * @param[in]   pFunc       ボタン押下時に実行される関数ポインタ
     */
    MessageView& AddButton(const std::string& caption, ActionFunc pFunc) NN_NOEXCEPT { return AddButton(caption, pFunc, nullptr); }

    /**
     * @brief       ボタンを追加します。一番最後に追加したボタンがフォーカスされます。
     *
     * @param[in]   caption     ボタンのキャプション
     * @param[in]   pFunc       ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam      ボタン押下時に実行される関数への引数
     */
    MessageView& AddButton(const std::string& caption, ActionFunc pFunc, void* pParam) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。一番最後に追加したボタンがフォーカスされます。
     *
     * @param[in]   caption     ボタンのキャプション
     * @param[in]   pFunc       ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam      ボタン押下時に実行される関数への引数
     */
    MessageView& AddButton(const glv::WideString& caption) NN_NOEXCEPT { return AddButton(caption, nullptr, nullptr); }

    /**
     * @brief       ボタンを追加します。一番最後に追加したボタンがフォーカスされます。
     *
     * @param[in]   caption     ボタンのキャプション
     * @param[in]   pFunc       ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam      ボタン押下時に実行される関数への引数
     */
    MessageView& AddButton(const glv::WideString& caption, ActionFunc pFunc) NN_NOEXCEPT { return AddButton(caption, pFunc, nullptr); }

    /**
     * @brief       ボタンを追加します。一番最後に追加したボタンがフォーカスされます。
     *
     * @param[in]   caption     ボタンのキャプション
     * @param[in]   pFunc       ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam      ボタン押下時に実行される関数への引数
     */
    MessageView& AddButton(const glv::WideString& caption, ActionFunc pFunc, void* pParam) NN_NOEXCEPT;

    /**
    * @brief       画像を追加します。
    *
    * @param[in]   pImageData   画像データポインタ
    * @param[in]   imageDataSize   画像データサイズ
    */
    qcit::Icon* AddImage(void* imageData, uint32_t imageDataSize, uint16_t imageWidth, uint16_t imageHeight) NN_NOEXCEPT;

protected:
    void Initialize() NN_NOEXCEPT;

    template<typename StringType>
    glv::Label* AddMessageT(const StringType& message) NN_NOEXCEPT;
    template<typename StringType>
    glv::Label* AddMessageT(const StringType& message, float fontSize) NN_NOEXCEPT;

    template<typename StringType>
    glv::Button* AddButtonT(const StringType& caption, ActionFunc pFunc, void* pParam) NN_NOEXCEPT;

    virtual glv::View* focusableView() NN_NOEXCEPT NN_OVERRIDE;

    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

    static void ReceiveUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT;

protected:
    glv::Table          m_MessagesTable;                //!< メッセージ配置テーブル
    glv::Table          m_ButtonsTable;                 //!< ボタン配置テーブル
    glv::Table          m_ImageTable;                   //!< 画像を配置するためのテーブル
    glv::Table          m_ViewTable;                    //!< メッセージとボタンと画像を内包する配置テーブル
    bool                m_IsClosable;                   //!< B ボタン押下で閉じる操作を行うかの可否
    bool                m_IsCenteringButton;

    struct Action
    {
        glv::Button*    pButton;                        //!< 実行対象のボタン
        ActionFunc      pFunc;                          //!< 実行時に呼び出す関数
        void*           pParam;                         //!< 実行時に渡す引数
        bool            isExitModal;                    //!< ExitModalを実行するかどうかのフラグ

        Action() : isExitModal(true) {}
    };
    std::vector< Action > m_Actions;                    //!< 実行情報
};


/**
 * @brief       モーダルビューの親ビューです。
 *              モーダルビューを表示する際は、全面を透明な膜で覆ってページへの入力をブロックします。
 * @details     本クラスはスレッドセーフではありません。RootSurfaceContext が動作するスレッド上で呼んでください。
 */
class InvisibleWall : public glv::View
{
    NN_DISALLOW_COPY( InvisibleWall );
    NN_DISALLOW_MOVE( InvisibleWall );

protected:
    friend class RootSurfaceContext;

    /**
     * @brief       コンストラクタです。
     */
    InvisibleWall() NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~InvisibleWall() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       RootSurfaceContext へアタッチされた時に呼ばれます。
     *
     * @param[in]   pContext    ルートコンテキスト
     */
    void OnAttached(RootSurfaceContext* pContext) NN_NOEXCEPT;

    /**
     * @brief       RootSurfaceContext からデタッチ時に呼ばれます。
     *
     * @param[in]   pContext    ルートコンテキスト
     */
    void OnDetached(RootSurfaceContext* pContext) NN_NOEXCEPT;

    /**
     * @brief       モーダルビューが表示中かを判定します。
     *
     * @return      モーダルビューが表示中ならば真を返します。
     */
    bool IsVisible() const NN_NOEXCEPT { return enabled(glv::Visible) != 0; }

    /**
     * @brief       モーダルビューの表示を開始します。
     *
     * @param[in]   pModalView                          モーダルビュー
     * @param[in]   isDisposable                        表示後にモーダルビューのインスタンスを破棄するなら真
     * @param[in]   isDelayExec                         表示中モーダルビューの EndModal 後に遅延して StartModal するなら真
     * @param[in]   isAutomaticExitAtBackgroundEnabled  表示中モーダルビューをBG遷移時に自動で閉じるなら真
     */
    void StartModal( ModalView* pModalView, bool isDisposable, bool isDelayExec, bool isAutomaticExitAtBackgroundEnabled = false ) NN_NOEXCEPT;
    /**
     * @brief       モーダルビューの表示を終了します。
     */
    void EndModal() NN_NOEXCEPT;

    /**
     * @brief       TORIAEZU: 強制的に Modal を有効化します( DropDown 項目選択時用 )
     */
    void ForceGainFocus(ModalView* pModalView) NN_NOEXCEPT;

    /**
     * @brief       glv シーンレンダラ前のループ処理です。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     *
     * @details     glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     *              この時点ではまだ glv コンテキストのレンダリングは開始していません。
     *              このメソッドが呼び出されるフレームは OnLoopAfterSceneRenderer() と同じです。
     *              ページが表示されている場合は、先にページの OnLoopBeforeSceneRenderer() が呼ばれます。
     */
    void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

    /**
     * @brief       glv シーンレンダラ後のループ処理です。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     *
     * @details     glv シーンレンダラのレンダリングが終わった後に呼び出されます。
     *              このメソッドが呼び出されるフレームは OnLoopBeforeSceneRenderer() と同じです。
     *              ページが表示されている場合は、先にページの OnLoopAfterSceneRenderer() が呼ばれます。
     */
    void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

    /**
     * @brief       アプリケーションのFG→BG遷移が発生した場合に呼び出されます。
     *
     * @details     呼び出されるタイミングはシステム通知及びGPU出画制限状態に依存するため、
     *              必ずしもBG遷移開始時に呼び出されるわけではありません。
     */
    void OnChangeIntoBackground() NN_NOEXCEPT;

    /**
     * @brief       アプリケーションのBG→FG遷移が発生した場合に呼び出されます。
     *
     * @details     呼び出されるタイミングはシステム通知及びGPU出画制限状態に依存するため、
     *              必ずしもFG遷移開始時に呼び出されるわけではありません。
     */
    void OnChangeIntoForeground() NN_NOEXCEPT;

    /**
     * @brief       イベント処理時に呼ばれます。
     *
     * @param[in]   events      イベント
     * @param[in]   context     コンテキスト
     *
     * @return      イベントを処理したら真を返します。
     */
    virtual bool onEvent(glv::Event::t events, glv::GLV& context) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       表示中のモーダルビューの有無を調べます。
     *
     * @return      表示中のモーダルビューがあれば真を返します。
     */
    bool IsModalViewRunning() NN_NOEXCEPT;

private:

    void DisplayModal( ModalView* pModalView ) NN_NOEXCEPT;
    void EndModal( ModalView* pModalView ) NN_NOEXCEPT;

    RootSurfaceContext* m_pRootContext;                         //!< ルートコンテキスト
    ModalView*          m_pModalView;                           //!< 表示中のモーダルビュー
    ModalView*          m_pNextModalView;                       //!< 遅延実行のモーダルビュー
    glv::Style          m_WallStyle;                            //!< 個別スタイル
    bool                m_IsDisposable;                         //!< 表示後に破棄指定
    bool                m_IsDelayExec;                          //!< 既存の ModalView 終了後の遅延実行指定
    bool                m_IsAutomaticExitAtBackgroundEnabled;   //!< BG遷移時にダイアログを閉じるか
};

} // ~namespace qcit
