﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.h"

#include <glv_resources.h>
#include <glv/detail/glv_GfxTextApi.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_CharacterEncoding.h>

#include <nn/fs.h>
#include <nn/ns/ns_InitializationApi.h>
#ifdef NN_BUILD_CONFIG_OS_WIN
#include <nn/ns/ns_ApiForDfc.h>
#else
#include <nn/oe.h>
#include <nn/es/es_InitializationApi.h>
#include <nn/ncm/ncm_Service.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
// For declearation of 'nv::InitializeGraphics' and 'nv::SetGraphicsAllocator'
#include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
// For declearation of 'glslcSetAllocator'
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

#include "QCIT_RootSurface.h"

using namespace std;
using namespace nn::msgpack;

namespace {

    const size_t HeapByteSize = 128 * 1024 * 1024;
    const size_t MallocBytesSize = 64 * 1024 * 1024;
    const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX ) || NN_GFX_IS_TARGET_NVN

    /**
     * @brief 共通アロケータ
     *
     */
    static void* Allocate(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    /**
     * @brief 共通デアロケータ
     *
     */
    static void Deallocate(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    /**
     * @brief 共通リアロケータ
     *
     */
    static void* Reallocate(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif
#endif

    /**
     * @brief ペリフェラルセットアップ
     *
     */
    static void SetupPeripherals() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX )
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
        nv::InitializeGraphics(::malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
#endif
#if NN_GFX_IS_TARGET_NVN
        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
#endif
#endif
    }

    static const glv::InitialConfiguration LocalGfxConfiguration = glv::InitialConfiguration(512 * 1024, 1 * 1024 * 1024, 8);
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration(
        glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary
    );
}

extern "C" void nninitStartup() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(HeapByteSize));
    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, MallocBytesSize));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MallocBytesSize);
}

void Initialize() NN_NOEXCEPT
{
#ifdef NN_BUILD_CONFIG_OS_WIN
    nn::ns::InitializeDependenciesForDfc();
#else
    nn::oe::Initialize();
    nn::ncm::Initialize();
    nn::es::Initialize();
#endif

    nn::ns::Initialize();
}

extern "C" void nnMain() NN_NOEXCEPT
{
    // ペリフェラルを初期化
    SetupPeripherals();

    // ROM マウント
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    std::unique_ptr<char[]> mountRomCacheBuffer(new char[mountRomCacheBufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(mountRomCacheBuffer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom(RomMountName, mountRomCacheBuffer.get(), mountRomCacheBufferSize)
    );

    Initialize();

    // リソースプロバイダ及びリソースの初期化
    glv::Resources::GetSharedResourceContext().Initialize(RomMountName);

    // GLVランタイムエンジンの初期化及びアプリケーションフレームワークの初期化
    glv::ApplicationFrameworkInitialize(LocalHidConfiguration);

    // QCIT エントリポイント呼び出し
    qcit::QCITMain();

    // フォントロード完了したのでマウント解除
    nn::fs::Unmount(RomMountName);
}
