﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "common.h"

#include <cstdlib>
#include <cstring>
#include <glv_viewutility.h>
#include <nn/nn_Assert.h>

#include <nn/ns/ns_InstallApi.h>
#include <nn/ns/ns_Result.h>

#include "QCIT_StringUtil.h"
#include "QCIT_CustomModalView.h"

namespace qcit
{
    const size_t       InstallView::InitialActionCapacity = 4;
    const glv::space_t InstallView::InstallViewWidth = 800.0f;
    const glv::space_t InstallView::ProgressBarMargineX = 8.0f;
    const glv::space_t InstallView::MessageButtonPaddingX = 32.0f;
    const glv::space_t InstallView::MessageButtonPaddingY = 16.0f;
    const glv::space_t InstallView::MessageViewPaddingX = 32.0f;
    const glv::space_t InstallView::MessageViewPaddingY = 32.0f;

    void CreateErrorMessage(std::string* pOutStr, const std::string& errorStr, nn::Result result, bool addsNewLine) NN_NOEXCEPT
    {
        char errorValueStr[32] = { 0 };
        nn::util::SNPrintf(errorValueStr, sizeof(errorValueStr), "(result: 0x%08x)", result.GetInnerValueForDebug());

        if (addsNewLine)
        {
            *pOutStr = errorStr + '\n' + std::string(errorValueStr);
        }
        else
        {
            *pOutStr = errorStr + std::string(errorValueStr);
        }
    }

// ----------------------------------------------------------------------

    Button::Button(const glv::WideString& caption, std::function< void() > callback, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
        : glv::Button(r, true)
        , m_Label(caption, glv::Label::Spec(glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize))
        , m_Callback(callback)
    {
        this->anchor(anchor);
        Initialize();
    }

    Button::Button(const char* caption, std::function< void() > callback, const glv::Rect& r, glv::Place::t anchor) NN_NOEXCEPT
        : glv::Button(r, true)
        , m_Label(caption, glv::Label::Spec(glv::Place::CC, 0.0f, 0.0f, CommonValue::InitialFontSize))
        , m_Callback(callback)
    {
        this->anchor(anchor);
        Initialize();
    }

    Button::Button(const std::string& text, std::function< void() > callback, float fontSize, glv::space_t paddingX, glv::space_t paddingY) NN_NOEXCEPT
        : glv::Button(glv::Rect(80.0f, 20.0f), true)
        , m_Label(text, false)
        , m_Callback(callback)
    {
        FitButton(fontSize, paddingX, paddingY);
        Initialize();
    }

    Button::Button(const glv::WideString& text, std::function< void() > callback, float fontSize, glv::space_t paddingX, glv::space_t paddingY) NN_NOEXCEPT
        : glv::Button(glv::Rect(80.0f, 20.0f), true)
        , m_Label(text, false)
        , m_Callback(callback)
    {
        FitButton(fontSize, paddingX, paddingY);
        Initialize();
    }

    void Button::Initialize() NN_NOEXCEPT
    {
        *this << m_Label;
        changePadClickDetectableButtons(glv::BasicPadEventType::Button::Ok::Mask);
        changePadClickDetectableButtons(glv::DebugPadEventType::Button::Ok::Mask);
        attach([](const glv::Notification& n)->void { n.receiver< Button >()->m_Callback(); }, glv::Update::Clicked, this);

        m_StyleLabel[ButtonStyle_Focusable] = m_Label.style();
        m_StyleLabel[ButtonStyle_NotFocusable].color = glv::Style::standard().color;
        m_StyleLabel[ButtonStyle_NotFocusable].color.set(glv::StyleColor::BlackOnWhite);
        m_StyleLabel[ButtonStyle_NotFocusable].color.fore.set(0.8f, 0.8f, 0.8f);
        m_StyleLabel[ButtonStyle_FocusableButInvalid] = glv::Style(m_StyleLabel[ButtonStyle_NotFocusable]);

        m_StyleButton[ButtonStyle_Focusable] = style();
        m_StyleButton[ButtonStyle_NotFocusable].color = glv::Style::standard().color;
        m_StyleButton[ButtonStyle_NotFocusable].color.fore.set(0.3f, 0.3f, 0.3f);
        m_StyleButton[ButtonStyle_NotFocusable].color.border.set(0.3f, 0.3f, 0.3f);
        m_StyleButton[ButtonStyle_NotFocusable].color.back.set(0.15f, 0.15f, 0.15f);
        m_StyleButton[ButtonStyle_FocusableButInvalid] = glv::Style(m_StyleButton[ButtonStyle_NotFocusable]);
        m_StyleButton[ButtonStyle_FocusableButInvalid].color.back.set(0.2f, 0.2f, 0.2f);
    }

        void Button::FitButton(float fontSize, glv::space_t paddingX, glv::space_t paddingY) NN_NOEXCEPT
    {
        m_Label.size(fontSize);
        m_Label.paddingX(paddingX);
        m_Label.paddingY(paddingY);
        m_Label.anchor(glv::Place::CC);
        m_Label.pos(glv::Place::CC, 0.0f, 0.0f).anchor(glv::Place::CC);
        const glv::space_t width = (m_Label.width() + 1.0f) / 2.0f * 2.0f;   // ピクセルだから、センタリング時に 2 の倍数じゃないと...
        const glv::space_t height = (m_Label.height() + 1.0f) / 2.0f * 2.0f; // ピクセルだから、センタリング時に 2 の倍数じゃないと...
        glv::Button::extent(width, height);
    }

        void Button::UpdateFocusAndColor(bool isValid, bool isFocusable) NN_NOEXCEPT
    {
        const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
        const glv::Property::t focusableButInvalidProperty = glv::Property::t::Controllable;

        if (isValid)
        {
            style(&m_StyleButton[ButtonStyle_Focusable]);
            m_Label.style(&m_StyleLabel[ButtonStyle_Focusable]);
            enable(focusableProperty);
        }
        else
        {
            m_Label.disable(glv::Property::DrawBack);

            if (!isFocusable)
            {
                style(&m_StyleButton[ButtonStyle_NotFocusable]);
                m_Label.style(&m_StyleLabel[ButtonStyle_NotFocusable]);
                disable(focusableProperty);
            }
            else
            {
                style(&m_StyleButton[ButtonStyle_FocusableButInvalid]);
                m_Label.style(&m_StyleLabel[ButtonStyle_FocusableButInvalid]);
                disable(focusableButInvalidProperty);
            }
        }
    }

        void Button::UpdateLabelText(const char* text) NN_NOEXCEPT
    {
        m_Label.setValue(text);
    }

        void Button::IncreaseLabelSize(glv::space_t increasedX, glv::space_t increasedY) NN_NOEXCEPT
    {
        glv::Button::extent(m_Label.width() + increasedX, m_Label.height() + increasedY);
    }

        void Button::ChangeLabelSize(glv::space_t width, glv::space_t height) NN_NOEXCEPT
    {
        glv::Button::extent(width, height);
    }

ConfirmView::ConfirmView() NN_NOEXCEPT
    : MessageView( false )
{
    // MessageView で構築されたビューツリーをクリア
    Remove();

    // Table を登録
    m_ViewTable.arrangement("<, x");
    *this << (m_ViewTable << m_MessagesTable << m_ButtonsTable);

    m_ViewTable.arrange();
}

/**
 * @brief       コンストラクタ
 */
InstallView::InstallView( std::function< void() > closeCallback, std::function< void() > cancelCallback, std::function< void() > cleanupCallback, std::function< void() > nextCallback ) NN_NOEXCEPT
    :   MessageView( false ),
    m_ProgressBarsTable( "", MessageViewPaddingX, MessageViewPaddingY, glv::Rect( 0.0f ) ),
    m_pProgressBar(nullptr),
    m_pTotalProgressBar(nullptr),
    m_pProgressCaption( new glv::Label( "[Install Progress]", glv::Label::Spec( glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize ) ) ) ),
    m_pProgressFileName( new glv::Label( "", glv::Label::Spec( glv::Label::Spec( glv::Place::TL, m_pProgressCaption->right() + 10.0f, 0.0f, CommonValue::InitialFontSize ) ) ) ),
    m_pProgressLabel( new glv::Label( "Prepare installing...", glv::Label::Spec( glv::Place::TL, 25.0f, m_pProgressCaption->bottom() + 20.0f, CommonValue::InitialFontSize ) ) ),
    m_pTotalProgressLabel(nullptr),
    m_pCloseButton(new Button("Close", closeCallback, CommonValue::InitialFontSize, 26.0f, 12.0f)),
    m_pCancelButton(new Button("Cancel", cancelCallback, CommonValue::InitialFontSize, 26.0f, 12.0f)),
    m_pNextButton(new Button("Next", nextCallback, CommonValue::InitialFontSize, 26.0f, 12.0f)),
    m_pParentResultListView(nullptr),
    m_pInstallResultListView(nullptr),
    m_InstallResult( nn::ResultSuccess() )
{
    Initialize();
}

InstallView::~InstallView() NN_NOEXCEPT
{
}

void InstallView::Initialize() NN_NOEXCEPT
{
    enable( glv::Property::DrawBorder | glv::Property::DrawBack );
    disable( glv::Property::FocusHighlight );
    m_Actions.reserve( InstallView::InitialActionCapacity );

    // MessageView で構築されたビューツリーをクリア
    Remove();

    // Table
    m_ViewTable.arrangement( "<,^,x" );
    *this << ( m_ViewTable << m_MessagesTable << m_ProgressBarsTable << m_ButtonsTable );

    // (SIGLO-71432) インストール結果画面の表示崩れを防止するためタッチ(フォーカス)できないようにしておく
    m_MessagesTable.disable(glv::Property::HitTest);

    // メッセージ
    AddProgressMessage( m_pProgressCaption, m_pProgressFileName, m_pProgressLabel );

    // プログレスバー
    const glv::Rect sliderRect( InstallView::InstallViewWidth - InstallView::ProgressBarMargineX - InstallView::ProgressBarMargineX, CommonValue::InitialFontSize );
    m_pProgressBar = new glv::Slider( sliderRect, 0.0f  );
    m_pTotalProgressLabel = new glv::Label("Total Progress : Processing ...", glv::Label::Spec(glv::Place::TL, 25.0f, m_pProgressBar->bottom(), CommonValue::InitialFontSize));
    m_pTotalProgressBar = new glv::Slider(sliderRect, 0.0f);
    AddProgressBar(m_pProgressBar, m_pTotalProgressLabel, m_pTotalProgressBar);

    const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
    m_pProgressBar->disable( focusableProperty );
    // (SIGLO-71432) 全体の進捗バーがタッチ(フォーカス)できないように
    m_pTotalProgressBar->disable( focusableProperty );

    // ModalView の表示位置調整のために非表示で追加しておく
    // 表示予定のボタンを登録(状態により表示/非表示を切り替える)
    AddButton(m_pCancelButton, false);
    AddButton(m_pNextButton, false);
    // 全て非表示にしておく
    RemoveButtons();
}

InstallView& InstallView::AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT
{
    m_pProgressCaption = pProgressCaption;
    m_pProgressFileName = pProgressFileName;
    m_pProgressLabel = pProgressLabel;

    // レイアウト調整
    m_MessagesTable << pProgressCaption;
    m_MessagesTable << pProgressFileName;
    m_MessagesTable << pProgressLabel;
    m_MessagesTable.arrange();
    m_ViewTable.arrange();

    return *this;
}

InstallView& InstallView::AddButton(glv::Button* pButton, bool isExitModal) NN_NOEXCEPT
{
    return AddButton(pButton, nullptr, nullptr, isExitModal);
}

InstallView& InstallView::AddButton(glv::Button* pButton, ActionFunc pFunc, void* pParam, bool isExitModal) NN_NOEXCEPT
{
    pButton->attach(ReceiveUpdateNotification, glv::Update::Clicked, this);

    // 配置方法指定
    const std::string align = m_ButtonsTable.arrangement() + "x";
    m_ButtonsTable.arrangement(align.c_str());

    // レイアウト調整
    m_ButtonsTable << pButton;
    m_ButtonsTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    // 実行情報を登録
    Action action;
    action.pButton = pButton;
    action.pFunc = pFunc;
    action.pParam = pParam;
    action.isExitModal = isExitModal;
    m_Actions.push_back(action);

    return *this;
}

InstallView& InstallView::AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT
{
    m_pProgressBar = pProgressBar;

    // 配置方法指定
    const std::string align = m_ProgressBarsTable.arrangement() + "x";
    m_ProgressBarsTable.arrangement( align.c_str() );

    // レイアウト調整
    m_ProgressBarsTable << pProgressBar;
    m_ProgressBarsTable.arrange();
    m_ViewTable.arrange();
    extent( m_ViewTable.width(), m_ViewTable.height() );
    return *this;
}

InstallView& InstallView::AddProgressBar(glv::Slider* pProgressBar, glv::Label* pTotalProgressLabel, glv::Slider* pTotalProgressBar) NN_NOEXCEPT
{
    // 配置方法指定
    const char* align = "^,p,^";
    m_ProgressBarsTable.arrangement(align);

    // レイアウト調整
    m_ProgressBarsTable << pProgressBar << pTotalProgressLabel << pTotalProgressBar;
    m_ProgressBarsTable.arrange();
    m_ViewTable.arrange();
    extent(m_ViewTable.width(), m_ViewTable.height());

    // プログレスバー＆ラベル位置の微調整
    pProgressBar->posAdd(0.0f, -40.0f);
    pTotalProgressLabel->posAdd(0.0f, -10.0f);
    pTotalProgressBar->posAdd(0.0f, -20.0f);

    return *this;
}

void InstallView::SetErrorMessageView( bool isLastTarget ) NN_NOEXCEPT
{
    SetProgressCaption( "[Install Failed]" );

    std::string errorMessage;
    bool requestsCleanup = false;
    bool isOnlyCancelButtonDisplayed = false;

    if ( nn::ncm::ResultContentNotFound::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "Installable content is not found. Confirm the nsp file.", m_InstallResult, true );
    }
    else if ( nn::ncm::ResultNotEnoughInstallSpace::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "There is not enough free space. More than nsp file size is required.", m_InstallResult, true );
        // (SIGLO-76759) 空き容量が足りない系エラーは Cancel ボタンしか出ないようにする
        // (次にコンテンツが控えていても必ずスキップさせる)
        isOnlyCancelButtonDisplayed = true;
    }
    else if (nn::ncm::ResultContentAlreadyExists::Includes(m_InstallResult))
    {
        CreateErrorMessage(&errorMessage, "The same or new version of the content has already been installed.", m_InstallResult, true);
    }
    else if ( nn::ns::ResultSdCardNoOwnership::Includes( m_InstallResult ) || nn::ns::ResultSdCardDatabaseCorrupted::Includes( m_InstallResult) )
    {
        requestsCleanup = true;
        CreateErrorMessage( &errorMessage,
            "SD card was initialized on another devine or database is corrupted.\nCleanup is required to use SD card.", m_InstallResult, true );
    }
    else if ( nn::ns::ResultSdCardNotMounted::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage,
            "SD card was inserted after device boot.\nDevice reboot is required to install contents to the card.", m_InstallResult, true );
    }
    else if ( nn::ncm::ResultContentHashNotMatched::Includes( m_InstallResult ) )
    {
        CreateErrorMessage( &errorMessage, "nsp file is corrupted. ", m_InstallResult, false );
    }
    else
    {
        CreateErrorMessage( &errorMessage, "Failed to install the nsp file. ", m_InstallResult, false );
    }

    SetProgressLabel( errorMessage.c_str() );
    m_ProgressBarsTable.disable( glv::Property::Visible );

    RemoveButtons();

    auto& g = reinterpret_cast< glv::GLV& >( root() );

    if ( !isLastTarget )
    {
        m_pCancelButton->enable(glv::Property::Visible);
    }
    else
    {
        // 最後のコンテンツである場合はNextボタンのみが中央に配置されるように調整する
        m_pCancelButton->remove();
        m_pNextButton->remove();
        AddButton(m_pNextButton, false);
        isOnlyCancelButtonDisplayed = false;
    }

    if (isOnlyCancelButtonDisplayed)
    {
        g.setFocus(m_pCancelButton);
    }
    else
    {
        m_pNextButton->enable(glv::Property::Visible);
        g.setFocus( m_pNextButton );
    }
}

void InstallView:: RecoverErrorToProgressMessage() NN_NOEXCEPT
{
    SetProgressCaption("[Install Progress]");

    m_ProgressBarsTable.enable(glv::Property::Visible);
    SetProgressLabel("Prepare installing...");

    RemoveButtons();
}

void InstallView::SetInstallResultMessageView() NN_NOEXCEPT
{
    SetProgressCaption("[Install Result]");

    auto beforeCenterX = this->centerX();
    auto beforeCenterY = this->centerY();

    m_pProgressFileName->remove();
    m_pProgressLabel->remove();
    m_ProgressBarsTable.remove();

    // シザーボックスの領域設定
    static const glv::space_t InstallResultListWidth = 950.0f;
    static const glv::space_t InstallResultListHeight = 480.0f;
    static const glv::space_t InstallSceneHeaderRegion = 20.0f;
    static const glv::space_t InstallSceneFooterRegion = 130.0f;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginL = 16.0f;
    static const glv::space_t ListMarginR = 16.0f;
    static const glv::space_t ListMarginY = 2.0f;

    auto pResultStringLabel = new glv::Label("Success", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize + 8.0f));
    m_pParentResultListView = new glv::ScissorBoxView(0, InstallSceneHeaderRegion, InstallResultListWidth, InstallResultListHeight - (InstallSceneHeaderRegion + InstallSceneFooterRegion));
    {
        glv::Rect clipRegion(ListMarginL, ListMarginY, m_pParentResultListView->width() - (ListMarginL + ListMarginR), m_pParentResultListView->height() - (ListMarginY * 2));
        auto pInstallResultListView = new InstallResultListView(clipRegion);

        m_pInstallResultListView = pInstallResultListView;
        *m_pParentResultListView << pInstallResultListView;

        m_MessagesTable << m_pParentResultListView;

        auto pElapsedLabel = new glv::Label(m_TotalElapsedTime, glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize - 1.0f));
        auto pTotalResultLabel = new glv::Label("Total Result :", glv::Label::Spec(glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize - 1.0f));

        m_MessagesTable << pElapsedLabel << pTotalResultLabel << pResultStringLabel;
        m_MessagesTable.arrangement("< - -, x - -, < > <");
        m_MessagesTable.arrange();

        m_ViewTable.arrangement("<,x");
        m_ViewTable.arrange();
        extent(m_ViewTable.width(), m_ViewTable.height());

        // インストール結果の画面を中央に持っていく(ずらす)
        auto addX = (beforeCenterX - this->centerX());
        auto addY = (beforeCenterY - this->centerY());
        this->posAdd(addX, addY);
    }

    {
        bool isIncludeFailure = false;
        auto listSize = m_InstallResultList.size();
        if (listSize > 0)
        {
            // SyncList 用の Failure が入っているため、Total Result は Failure 扱いとする
            isIncludeFailure = true;
        }

        m_InstallResultList.resize(listSize + m_FileResultMap.size());

        int i = listSize;
        for (auto& item : m_FileResultMap)
        {
            auto& refResult = m_InstallResultList.at(i);
            BuildUtf16< 48 >(refResult.fileName, ClipMessage(item.first, 40).c_str());
            BuildUtf16< 64 >(refResult.result, item.second.resultString.c_str());
            if (item.second.isSuccess == false)
            {
                // ひとつでも Failure があれば全体の結果は Failure としておく
                isIncludeFailure = true;
                // 一応、Failure 扱いのものは目立つように赤文字で表示する
                refResult.isSyncFailure = true;
            }
            ++i;
        }

        {
            // 緑色で "Success" を表示
            std::string resultStr = "Success";
            m_ResultStringStyle.color.text.set(0.0f, 0.8f, 0.0f);
            if (isIncludeFailure)
            {
                // 赤色で "Failure" を表示
                resultStr = "Failure";
                m_ResultStringStyle.color.text.set(0.8f, 0.0f, 0.0f);
            }

            // Total Result の結果を設定する
            pResultStringLabel->style(&m_ResultStringStyle);
            pResultStringLabel->setValue(resultStr.c_str());
            // 位置を微調整
            pResultStringLabel->posAdd(-20.0f, 0.0f);
        }

        m_pInstallResultListView->EntryCollection(m_InstallResultList);
        //const glv::Property::t focusableProperty = glv::Property::t::Controllable | glv::Property::t::HitTest;
        const glv::Property::t focusableProperty = glv::Property::t::HitTest;

        m_pInstallResultListView->enable(focusableProperty);
        m_pInstallResultListView->disable(glv::Property::t::DrawSelectionBox);
    }

    SetCloseButton();
}

void InstallView::SetCloseButton() NN_NOEXCEPT
{
    RemoveButtons();
    // 不要なボタンは削除しておく
    m_pCancelButton->remove();
    m_pNextButton->remove();

    // Close ボタンのみが中央に配置されるように調整する
    AddButton(m_pCloseButton);
    m_pCloseButton->enable( glv::Property::Visible );

    auto& g = reinterpret_cast< glv::GLV& >( root() );
    g.setFocus( m_pCloseButton );
}

/**
 * @brief       ButtonsTable に追加された Button を全て削除します。
 */
void InstallView::RemoveButtons() NN_NOEXCEPT
{
    // ボタンそのものを消すのではなく、単純に見えないようにする
    m_pCancelButton->disable(glv::Property::Visible);
    m_pCloseButton->disable(glv::Property::Visible);
    m_pNextButton->disable(glv::Property::Visible);
}

void InstallView::SetInstallFileNameList(std::set<std::string>& inSetList) NN_NOEXCEPT
{
    for (auto& name : inSetList)
    {
        // ResultString の初期値は Skip にしておく
        ResultView rv(": Skip");
        m_FileResultMap.insert(std::make_pair(std::string(name), rv));
    }
}

void InstallView::AddInstallResultList(const std::string& inFileName, nn::Result inResult, int64_t inElapsedSecond) NN_NOEXCEPT
{
    auto iter = m_FileResultMap.find(inFileName);
    if (iter == std::end(m_FileResultMap))
    {
        // ありえないはずだが、登録されていないファイル名が入ってきた場合、抜ける
        return;
    }

    char resultStr[32] = ": Success";
    if (inResult.IsFailure())
    {
        nn::util::SNPrintf(resultStr, 32, ": Failure ( 0x%08x )", inResult.GetInnerValueForDebug());
    }
    else
    {
        iter->second.isSuccess = true;
    }

    char resultElapsedStr[64] = { 0 };
    nn::util::SNPrintf(resultElapsedStr, 64, "%s [ %s ]", resultStr, ConvertToFormatTime(inElapsedSecond).c_str());

    iter->second.resultString = resultElapsedStr;
}

void InstallView::SetSyncFailIdList(std::set<nn::Bit64>& inFaildIdList) NN_NOEXCEPT
{
    if (inFaildIdList.empty())
    {
        return;
    }

    auto listSize = m_InstallResultList.size();
    m_InstallResultList.resize(listSize + inFaildIdList.size());

    int i = listSize;
    for (auto& id : inFaildIdList)
    {
        BuildUtf16< 48 >(m_InstallResultList.at(i).fileName, "SyncList ( 0x%016llx )", id);
        BuildUtf16< 32 >(m_InstallResultList.at(i).result, ": [ Content Not Found ]");
        m_InstallResultList.at(i).isSyncFailure = true;
        ++i;
    }
}

void InstallView::SetSyncFailAocIndexList(nn::Bit64 inTargetAppId, std::set<int>& inFaildIndexList) NN_NOEXCEPT
{
    if (inFaildIndexList.empty())
    {
        return;
    }

    auto listSize = m_InstallResultList.size();
    m_InstallResultList.resize(listSize + inFaildIndexList.size());

    int i = listSize;
    for (auto& index : inFaildIndexList)
    {
        BuildUtf16< 48 >(m_InstallResultList.at(i).fileName, "SyncList ( 0x%016llx, Index = %d )", inTargetAppId, index);
        BuildUtf16< 32 >(m_InstallResultList.at(i).result, ": [ Content Not Found ]");
        m_InstallResultList.at(i).isSyncFailure = true;
        ++i;
    }
}

void InstallView::SetTotalElapsedSecond(int64_t inElapsedSecond) NN_NOEXCEPT
{
    char totalElapsedStr[64] = { 0 };
    nn::util::SNPrintf(totalElapsedStr, 64, " Total Elapsed Time : %s", ConvertToFormatTime(inElapsedSecond).c_str());
    m_TotalElapsedTime = totalElapsedStr;
}

} // ~namespace qcit
