﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/repair.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nn/fs/fs_ApiPrivate.h>
#include <nn/fs/fs_MmcPrivate.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/manu/manu_Api.h>
#include <nn/repair/repair_Authentication.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_CommandLineOption.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/settings/system/settings_ProductModel.h>

#include <nn/util/util_ScopeExit.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/nn_SdkLog.h>
#include <nn/util/util_StringUtil.h>

#define RETURN_IF_FAILURE(result, ...) \
    do { \
    if (result.IsFailure()){SendString(__VA_ARGS__); return result;} \
    } while(NN_STATIC_CONDITION(false))

namespace {

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
const size_t messageBufferSize = 1024;
char messageBuffer[messageBufferSize] = {};

static nn::os::MutexType s_Mutex;

static const int TOOL_MAJOR_VERSION = 8;
static const int TOOL_MINOR_VERSION = 0;
static const int BACKUP_THREAD_STACK_SIZE = 0x10000;

static glv::GLV           s_GlvRootView;
static nn::repair::StreamTextView*    s_pTextView;
static nn::os::ThreadType s_SdWaiterThreadType;
static nn::os::ThreadType s_ProcessingThreadType;
static nn::os::EventType  s_SdInsertionEvent;

static nn::repair::LabelButton* s_StartButton;
static nn::repair::LabelButton* s_ExitButton;

static const char* const s_AutoExitOption = "--autoexit";

enum BackupConfig
{
    BackupConfig_Normal      = 0x00, // 通常モード
    BackupConfig_ErrorReport = 0x01, // エラーレポートを抜き出す
};
static int s_Config = BackupConfig_Normal;

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

static const std::string MountName = "Target";
static const std::string DirectoryPath = MountName + ":/" + nn::repair::ArchiveDirectoryPath; // 'NintendoRepairArchive'

void SendString(const char* pFormat, ...) NN_NOEXCEPT
{
    nn::os::LockMutex(&s_Mutex);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlockMutex(&s_Mutex);
    };

    va_list args;
    va_start(args, pFormat);
    auto len = vsnprintf(messageBuffer, messageBufferSize, pFormat, args);
    va_end(args);

    #if !defined(NN_REPAIR_DEVELOP_MODE)
        size_t writeSize;
        auto result = nn::manu::UsbWrite(&writeSize, messageBuffer, messageBufferSize, len);
        if (result.IsFailure())
        {
            NN_SDK_LOG("%s:%d %016x\n", __FILE__, __LINE__, result.GetInnerValueForDebug());
        }
    #else
        NN_UNUSED(len);
    #endif

    s_pTextView->AppendValue(messageBuffer);
}

void SendStringToScreen(const char* pFormat, ...) NN_NOEXCEPT
{
    nn::os::LockMutex(&s_Mutex);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::UnlockMutex(&s_Mutex);
    };

    va_list args;
    va_start(args, pFormat);
    auto len = vsnprintf(messageBuffer, messageBufferSize, pFormat, args);
    va_end(args);

    NN_UNUSED(len);

    s_pTextView->AppendValue(messageBuffer);
}

void ShowDeviceId() NN_NOEXCEPT
{
    char id[nn::repair::DeviceIdLength];
    nn::repair::GetDeviceId(id);
    SendString("Device Id : %s\n", id);
}

void ShowSerialNumber() NN_NOEXCEPT
{
    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);
    SendString("Serial Number : %s\n", number);
}

void ShowInternalInformation() NN_NOEXCEPT
{
    // デバイス ID 表示
    ShowDeviceId();

    // シリアルナンバー表示
    ShowSerialNumber();
}

nn::Result SaveDeviceId() NN_NOEXCEPT
{
    char id[nn::repair::DeviceIdLength];
    nn::repair::GetDeviceId(id);

    // 保存先のファイルパス
    std::string path = DirectoryPath + std::string(nn::repair::DeviceIdFileName);

    // 保存用ファイルを新規に作成
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsSuccess())
    {
        NN_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
    }
    NN_RESULT_DO(nn::fs::CreateFile(path.c_str(), 0));

    nn::fs::FileHandle handle;
    NN_RESULT_DO(
            nn::fs::OpenFile(
                &handle, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(
                handle, 0, id, strnlen(id, nn::repair::DeviceIdLength), nn::fs::WriteOption()));
    NN_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}

nn::Result SaveSerialNumber() NN_NOEXCEPT
{
    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);

    // 保存先のファイルパス
    std::string path = DirectoryPath + std::string(nn::repair::SerialNumberFileName);

    // 保存用ファイルを新規に作成
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsSuccess())
    {
        NN_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
    }
    NN_RESULT_DO(nn::fs::CreateFile(path.c_str(), 0));

    nn::fs::FileHandle handle;
    NN_RESULT_DO(
            nn::fs::OpenFile(
                &handle, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::WriteFile(
                handle, 0, number, strnlen(number, nn::repair::SerialNumberLength), nn::fs::WriteOption()));
    NN_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}

nn::Result SaveInternalInformation() NN_NOEXCEPT
{
    // デバイス ID の保存
    SendString("Save device id\n");
    NN_RESULT_DO(SaveDeviceId());

    // シリアルナンバーの保存
    SendString("Save serial number\n");
    NN_RESULT_DO(SaveSerialNumber());

    NN_RESULT_SUCCESS;
}

nn::Result CheckAccountSaveData() NN_NOEXCEPT
{
    bool isAccountExported;
    NN_RESULT_DO(nn::repair::IsAccountExported(&isAccountExported, DirectoryPath.c_str()));

    if (!isAccountExported)
    {
        SendString(" WARNING : Account save data was broken !! \n");
    }
    else
    {
        SendString("Account save data is alive\n");
    }

    bool isVirtualAccountExported;
    NN_RESULT_DO(nn::repair::IsVirtualAccountExported(&isVirtualAccountExported, DirectoryPath.c_str()));

    if (!isVirtualAccountExported)
    {
        SendString(" WARNING : Virtual account save data was broken !! \n");
    }
    else
    {
        SendString("Virtual account save data is alive\n");
    }

    if (!isAccountExported || !isVirtualAccountExported)
    {
        SendString("-------------------------------------------------\n");
        SendString(" Please unlink device\n");
        SendString("-------------------------------------------------\n");
    }

    NN_RESULT_SUCCESS;
}

nn::Result ShowCorruptedData(bool *pOutHasCorruption) NN_NOEXCEPT
{
    SendString("CORRUPTED DATA ----------------------------\n");
    NN_RESULT_DO(nn::repair::ShowCorruptedData(pOutHasCorruption, DirectoryPath.c_str(), SendString, nullptr));
    SendString("-------------------------------------------------\n");

    NN_RESULT_SUCCESS;
}

nn::Result RecordTargetInformation() NN_NOEXCEPT
{
    // 保存先のファイルパス
    const std::string path = DirectoryPath + std::string(nn::repair::HistoryFileName);

    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsFailure())
    {
        NN_RESULT_DO(nn::fs::CreateFile(path.c_str(), 0));
    }

    nn::fs::FileHandle handle;
    NN_RESULT_DO(
            nn::fs::OpenFile(&handle, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t offset;
    NN_RESULT_DO(nn::fs::GetFileSize(&offset, handle));

    // デバイス ID の取得
    char id[nn::repair::DeviceIdLength];
    nn::repair::GetDeviceId(id);

    // シリアルナンバーの取得
    char number[nn::repair::SerialNumberLength];
    nn::repair::GetSerialNumber(number);

    // SystemUpdateVersion の取得
    uint32_t rebootlessNupVersion;
    NN_RESULT_DO(nn::repair::GetSystemUpdateVersion(&rebootlessNupVersion));

    // ProductModel の取得
    auto model = nn::settings::system::GetProductModel();

    std::ostringstream history;
    history << "[backup] " << "device_id=" << id << ", " << "serial_number=" << number << ", ";
    history << "version=" << TOOL_MAJOR_VERSION << "." << TOOL_MINOR_VERSION << ", ";
    history << "rebootless-NUP=" << rebootlessNupVersion << ", ";
    history << "model=" << model << "\n";
    NN_RESULT_DO(nn::fs::WriteFile(handle, offset, history.str().c_str(), history.str().size(), nn::fs::WriteOption()));
    NN_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}

nn::Result CreateBlackList(nn::repair::BlackListStruct* pOutList) NN_NOEXCEPT
{
    static nn::Bit64 BlackListIds[] = { BLACK_LIST_IDS };
    auto count = sizeof(BlackListIds) / sizeof(BlackListIds[0]);

    // エラーレポート転送モードのときは
    // ブラックリスト内の該当 ID をゼロでつぶす
    if (s_Config & BackupConfig_ErrorReport)
    {
        for (auto index = 0; index < count; index++)
        {
            if (BlackListIds[index] == ERROR_REPORT_SAVEDATA_ID)
            {
                BlackListIds[index] = 0;
            }
        }
    }

    pOutList->pIds = BlackListIds;
    pOutList->count = count;

    NN_RESULT_SUCCESS;
}

nn::Result ReadConfigFile() NN_NOEXCEPT
{
    // 実行時に消されないように設定ファイルは直下に置く
    std::string path = MountName + ":/" + nn::repair::ConfigFileName;

    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsSuccess())
    {
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile(
                    &handle, path.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile(handle);
        };

        int64_t size;
        NN_RESULT_DO(nn::fs::GetFileSize(&size, handle));

        std::unique_ptr<char []> buffer(new char[static_cast<int>(size + 1)]);
        NN_RESULT_DO(
                nn::fs::ReadFile(handle, 0, buffer.get(), static_cast<size_t>(size)));
        *(buffer.get() + size) = '\0';

        std::stringstream stream(buffer.get());
        std::string line;

        while (std::getline(stream, line, '\n'))
        {
            if (line.find("ErrorReport") != std::string::npos)
            {
                s_Config |= BackupConfig_ErrorReport;
            }
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result CheckMigrationState() NN_NOEXCEPT
{
    SendString("- Checking Migration.\n");
    {
        nn::repair::ReportCallbackStruct callback = { SendString, nullptr };
        auto result = nn::repair::DetectMigration(DirectoryPath.c_str(), &callback);

        RETURN_IF_FAILURE(result, "- Detecting a Migration!\n");
    }
    SendString("- Migration is not found.\n");

    NN_RESULT_SUCCESS;
}

void ShowStorageSppedMode() NN_NOEXCEPT
{
    SendString("Storage speed mode : ");

    const char* eMmcSpeedMode = nullptr;
    nn::fs::MmcSpeedMode eMmcMode;
    auto result = nn::fs::GetMmcSpeedMode(&eMmcMode);
    if (result.IsFailure())
    {
        eMmcSpeedMode = "Failed to get the speed mode";
    }
    else
    {
        switch (eMmcMode)
        {
        case nn::fs::MmcSpeedMode_Identification:
            eMmcSpeedMode = "Identification";
            break;
        case nn::fs::MmcSpeedMode_LegacySpeed:
            eMmcSpeedMode = "Legacy";
            break;
        case nn::fs::MmcSpeedMode_HighSpeed:
            eMmcSpeedMode = "HighSpeed";
            break;
        case nn::fs::MmcSpeedMode_Hs200:
            eMmcSpeedMode = "HS200";
            break;
        case nn::fs::MmcSpeedMode_Hs400:
            eMmcSpeedMode = "HS400";
            break;
        case nn::fs::MmcSpeedMode_Unknown:
            eMmcSpeedMode = "Unknown";
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    const char* sdCardSpeedMode = nullptr;
    nn::fs::SdCardSpeedMode sdcardMode;
    result = nn::fs::GetSdCardSpeedMode(&sdcardMode);

    if (result.IsFailure())
    {
        sdCardSpeedMode = "Failed to get the speed mode";
    }
    else
    {
        switch (sdcardMode)
        {
        case nn::fs::SdCardSpeedMode_Identification:
            sdCardSpeedMode = "Identification";
            break;
        case nn::fs::SdCardSpeedMode_DefaultSpeed:
            sdCardSpeedMode = "Default";
            break;
        case nn::fs::SdCardSpeedMode_HighSpeed:
            sdCardSpeedMode = "HighSpeed";
            break;
        case nn::fs::SdCardSpeedMode_Sdr12:
            sdCardSpeedMode = "SDR12";
            break;
        case nn::fs::SdCardSpeedMode_Sdr25:
            sdCardSpeedMode = "SDR25";
            break;
        case nn::fs::SdCardSpeedMode_Sdr50:
            sdCardSpeedMode = "SDR50";
            break;
        case nn::fs::SdCardSpeedMode_Sdr104:
            sdCardSpeedMode = "SDR104";
            break;
        case nn::fs::SdCardSpeedMode_Ddr50:
            sdCardSpeedMode = "DDR50";
            break;
        case nn::fs::SdCardSpeedMode_Unknown:
            sdCardSpeedMode = "Unknown";
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    SendString("eMMC = %s, SdCard = %s", eMmcSpeedMode, sdCardSpeedMode);
}

nn::Result Backup(bool* pOutHasCorruption) NN_NOEXCEPT
{
    auto result = nn::fs::MountSdCard(MountName.c_str());
    RETURN_IF_FAILURE(result, "Failed to mount device\n");
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    result = ReadConfigFile();
    RETURN_IF_FAILURE(result, "Failed to read config file\n");

    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, DirectoryPath.c_str()).IsSuccess())
    {
        result = nn::fs::DeleteDirectoryRecursively(DirectoryPath.c_str());
        RETURN_IF_FAILURE(result, "Fialed to delete the directory\n");
    }
    result = nn::fs::CreateDirectory(DirectoryPath.c_str());
    RETURN_IF_FAILURE(result, "Failed to create the directory\n");

    nn::repair::Initialize();
    NN_UTIL_SCOPE_EXIT
    {
        nn::repair::Finalize();
    };

    SendString("- Save internal information\n");
    result = SaveInternalInformation();
    RETURN_IF_FAILURE(result, "Failed to save internal information\n");
    SendString("- Save done\n");

    result = CheckMigrationState();
    RETURN_IF_FAILURE(result, "Suspended by migration check.\n");

    SendString("- Backup save data\n");

    nn::repair::BlackListStruct list;
    result = CreateBlackList(&list);
    RETURN_IF_FAILURE(result, "Failed to create black list\n");

    nn::repair::ReportCallbackStruct callback = { SendString, nullptr, SendStringToScreen, nullptr };

    #if !defined(NN_REPAIR_DEVELOP_MODE)
        result = nn::repair::ExportSaveDataSecurely(
                DirectoryPath.c_str(), &list, &callback);
    #else
        result = nn::repair::ExportSaveData(
                DirectoryPath.c_str(), &list, &callback);
    #endif
    RETURN_IF_FAILURE(result, "Failed to export save data\n");
    {
        uint64_t size;
        int64_t time;
        int64_t count;
        nn::repair::GetTransportRateInfo(&size, &time, &count);
        if( time > 0 )
        {
            SendString("- Total %lld KiB %lld file %lld sec (%d KiB/sec) \n", size / 1024, count, time / 1000, (int)(size * 1000 / time) / 1024 );
        }
    }

    SendString("- Backup done\n");

    SendString("- Show corrupted data\n");
    result = ShowCorruptedData(pOutHasCorruption);
    RETURN_IF_FAILURE(result, "Failed to show the corruption data");
    SendString("- Show done\n");

    SendString("- Check account save data\n");
    result = CheckAccountSaveData();
    RETURN_IF_FAILURE(result, "Failed to check the account save data\n");
    SendString("- Check done\n");

    // バックアップ元の本体情報を書き出す
    SendString("- Record information\n");
    result = RecordTargetInformation();
    RETURN_IF_FAILURE(result, "Failed to record target information\n");
    SendString("- Record done\n");

    NN_RESULT_SUCCESS;
}

void ApplicationExit()
{
    // SD 挿入待ちイベントの終了
    nn::os::FinalizeEvent(&s_SdInsertionEvent);

#if !defined(NN_REPAIR_DEVELOP_MODE)
    // 中断時は Repair サービスへの入力待ち状態になっているので適当な値を送って終了させる
    auto message = nn::repair::BackupRequestMessage::MakeZero();
    nn::manu::WriteToHost(&message, sizeof(message), nn::repair::GetOptionArgument(nn::repair::RequestOption), 0, sizeof(message));
    nn::manu::FinalizeUfio();
#endif

    nn::os::FinalizeMutex(&s_Mutex);

    std::quick_exit(0);
}

void ProcessingThread(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    SendString("=== Backup Start ===\n");

    bool hasCorruption = false;
    auto result = Backup(&hasCorruption);
    SendString("M: %d, D: %d, I: %d\n",
        result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());

    if (nn::repair::ResultInMigration().Includes(result))
    {
        SendString("This console is maybe in migration mode.\n");
        SendString("Please run proper tool before backup.\n");
    }

    if (result.IsSuccess() && (!hasCorruption))
    {
        SendString("[[ SUCCESS ]]\n");
    }
    else if (result.IsSuccess() && hasCorruption)
    {
        SendString("[[ LIMITED SUCCESS ]]\n");
    }
    else
    {
        SendString("[[ FAILED ]]\n");
    }

    SendStringToScreen("\n\n\n\n\n --- \n");

    // eMMC 及び microSD の速度表示
    ShowStorageSppedMode();

    // お尻が表示されない問題への暫定対応
    // （glv::TextView にバッファされた未表示の残りカスを明示的に flush させる手段が不明)
    SendStringToScreen("\n\n\n\n\n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");
    SendStringToScreen(" ----------------------- \n");

    if (nn::repair::HasOption(s_AutoExitOption))
    {
        ApplicationExit();
    }
}

void BackupMouseDownHandler() NN_NOEXCEPT
{
}

void SdcardWaiter(void* arg) NN_NOEXCEPT
{
    NN_UNUSED(arg);

    nn::os::InitializeEvent(&s_SdInsertionEvent, false, nn::os::EventClearMode_AutoClear);

    if (nn::repair::HasOption(s_AutoExitOption))
    {
        SendString("=== Auto Exit mode ===\n");
    }

    SendString("Please insert your micro sd card\n");

    std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
    auto result = sdcard->WaitAttach();
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    SendString("Ready to backup !\n");
    SendString("Push START button\n");

    nn::os::SignalEvent(&s_SdInsertionEvent);

    // バックアップツールは SD カードの挿入を検知して動作開始
    nn::os::StartThread(&s_ProcessingThreadType);
}

void BackupWindowCreateHandler() NN_NOEXCEPT
{
    nn::os::InitializeEvent(&s_SdInsertionEvent, false, nn::os::EventClearMode_ManualClear);

    // 本体情報の表示
    ShowInternalInformation();

    // SD カードの挿入待ち
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ThreadStack[0x1000];
    auto result = nn::os::CreateThread(
            &s_SdWaiterThreadType, SdcardWaiter, nullptr, s_ThreadStack,
            sizeof(s_ThreadStack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::os::StartThread(&s_SdWaiterThreadType);
}

void BackupWindowDestroyHandler() NN_NOEXCEPT
{
    nn::os::DestroyThread(&s_SdWaiterThreadType);
}

void ExitMouseDownHandler() NN_NOEXCEPT
{
    // 処理スレッドの破棄
    nn::os::DestroyThread(&s_ProcessingThreadType);

    ApplicationExit();
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 512), 24.f);
    s_pTextView->pos(120, 100);
    *pGlvRootView << s_pTextView;

    // ツール情報ラベル表示
    auto titleLabel = new glv::Label("Repair Backup Tool", false);
    nn::repair::GetToolInformationLabel(
            titleLabel, "Repair Backup Tool", TOOL_MAJOR_VERSION, TOOL_MINOR_VERSION);
    *pGlvRootView << titleLabel;

    // 開始ボタン表示
    s_StartButton = new nn::repair::LabelButton(
            "START",
            [&]{BackupMouseDownHandler();},
            [&]{BackupWindowCreateHandler();},
            [&]{BackupWindowDestroyHandler();});
    s_StartButton->pos(120, 640);
    *pGlvRootView << s_StartButton;

    // 終了ボタン表示
    s_ExitButton = new nn::repair::LabelButton(
            "EXIT",
            [&]{ExitMouseDownHandler();});
    s_ExitButton->pos(1030, 640);
    *pGlvRootView << s_ExitButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

#if !defined(NN_REPAIR_DEVELOP_MODE)
    extern "C" void nninitInitializeSdkModule()
    {
        nn::fs::InitializeWithMultiSessionForTargetTool();
    }

    extern "C" void nninitFinalizeSdkModule()
    {
    }

    extern "C" void nndiagStartup()
    {
    }
#endif

extern "C" void nnMain()
{
    SetupPeripherals();

    #if !defined(NN_REPAIR_DEVELOP_MODE)
        nn::manu::InitializeUfio();
    #endif

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    nn::os::InitializeMutex(&s_Mutex, false, 0);

    // バックアップ処理スレッド
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ProcessingThreadStack[BACKUP_THREAD_STACK_SIZE];
    auto result = nn::os::CreateThread(
            &s_ProcessingThreadType, ProcessingThread, nullptr, s_ProcessingThreadStack,
            sizeof(s_ProcessingThreadStack), nn::os::DefaultThreadPriority);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}

