﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/* 多重インクルードの回避 */
#pragma once

#include <nn/gpio/gpio.h>

namespace ptd {
    namespace lib {

        struct GpioTypes
        {
            GpioTypes()
            {
                direction = nn::gpio::Direction_Input;
                value = nn::gpio::GpioValue_Low;
                interruptStatus = nn::gpio::InterruptStatus_Inactive;
                interruptMode = nn::gpio::InterruptMode_AnyEdge;
            }
            nn::gpio::Direction direction;
            nn::gpio::GpioValue value;
            nn::gpio::InterruptStatus interruptStatus;
            nn::gpio::InterruptMode interruptMode;
            NN_PADDING4;
        };

        class Gpio
        {
        public:
            Gpio();
            ~Gpio();

            /*!
            * @brief		GPIO のバッドの値を取得します。
            * @param[in]	direction	入出力方向を設定します。
            */
            nn::gpio::GpioValue GetValue(nn::gpio::Direction direction);

            /*!
            * @brief		GPIO の各列挙型をまとめて設定します。
            * @param[in]	direction	入出力方向を設定します。
            * @param[in]	value		状態(H/L)を設定します。
            * @param[in]	status		割り込みステータスを設定します。
            * @param[in]	mode		割り込みモードを設定します。
            */
            void SetGpioTypes(nn::gpio::Direction direction,
                nn::gpio::GpioValue value,
                nn::gpio::InterruptStatus status = nn::gpio::InterruptStatus_Inactive,
                nn::gpio::InterruptMode mode = nn::gpio::InterruptMode_AnyEdge) NN_NOEXCEPT;

            /*!
            * @brief		nn::gpio::OpenSession() を呼び出して指定した GPIO パッドとのセッションを開きます。
            * @param[in]	pad		GPIO のパッド
            */
            void OpenSession(nn::gpio::GpioPadName pad) NN_NOEXCEPT;

            /*!
            * @brief		nn::gpio::CloseSession() を呼び出して指定した GPIO パッドとのセッションを閉じます。
            */
            void CloseSession(void) NN_NOEXCEPT;

            /*!
            * @brief		割り込みステータスをクリアして許可状態に戻します。
            */
            void ResetInterrupt(void) NN_NOEXCEPT;

            /*!
            * @brief		CPU ファンの GpioPadName_FanTach で信号が一定回数発生するまでの時間を取得します。
            * @return		サンプル時間を返します。無回転の場合は -1 を返します。
            * @param[in]	sampling	サンプリング数を指定します。デフォルトは 10 です。
            * @param[in]	timeout		タイムアウト時間を秒で指定します。デフォルトは 1 秒です。
            * @pre			CPU ファンの回転速度はせいぜい 15000rpm 程度と予想。
            *				15000rpm の場合 1ms あたり 1/4 回転、信号は 1/2 周期なのでイベントの更新は余裕のはず。
            */
            int64_t GetSamplingTimeCpuFan(int sampling = 10, int timeoutSec = 1) NN_NOEXCEPT;

        protected:

        private:
            GpioTypes m_types;
            nn::gpio::GpioPadSession m_session;
            nn::os::SystemEventType m_event;
        };

    }
}
