﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nns {
namespace audio {

/**
 * @brief       ParseWavFormat() に関する処理結果を表す列挙型です。
 */
enum WavResult
{
    WavResult_Success,                  //!< 成功しました
    WavResult_InvalidDataFormat,        //!< データフォーマットが Wav ではありません
    WavResult_InsufficientBuffer        //!< データバッファが不十分で Wav フォーマットが読み込めませんでした
};

/**
 * @brief       Wav データのエンディアンを表す列挙型です。
 */
enum WavEndian
{
    WavEndian_Little,                   //!< リトルエンディアン
    WavEndian_Big                       //!< ビッグエンディアン
};

/**
 * @brief       Wav フォーマット情報です。
 */
struct WavFormat
{
    int64_t dataOffset;                 //!< PCM データの開始位置のデータ先頭からのバイトオフセット
    int64_t dataSize;                   //!< PCM データのサイズ
    int32_t sampleRate;                 //!< サンプルレート
    int16_t channelCount;               //!< チャンネル数
    int16_t bitsPerSample;              //!< 1 サンプルあたりのビット数
    WavEndian endian;                   //!< エンディアン
};

/**
 * @brief       Wav データからフォーマット情報を取得します。
 * @param[out]  pOutWavFormat           Wav フォーマット情報
 * @param[in]   buffer                  Wav データバッファ
 * @param[in]   size                    Wav データバッファサイズ
 * @return      処理の結果を返します。
 * @retval      WavResult_Success 成功しました
 * @retval      WavResult_InvalidDataFormat データフォーマットが Wav ではありません
 * @retval      WavResult_InsufficientBuffer データバッファが不十分で Wav フォーマットが読み込めませんでした
 * @pre
 * - pOutWavFormat != nullptr
 * - buffer != nullptr
 * @details
 * buffer には Wav ファイルの先頭から size バイトの情報が含まれている必要があります。
 * 本関数は buffer 内の riff ヘッダ、fmt チャンク、および data チャンクのヘッダ部分のみを参照して、その内容を WavFormat 構造体に格納します。
 * buffer 内にこれらの情報が含まれない場合は nns::audio::WavResult_InsufficientBuffer が返りますが、この場合は buffer により多くのデータを置いて再度関数を呼ぶようにしてください。
 * 正常な Wav ファイルの内容すべてが buffer に格納されている場合 nns::audio::WavResult_InsufficientBuffer は返りません。
 */
WavResult ParseWavFormat(WavFormat* pOutWavFormat, const void* buffer, size_t size) NN_NOEXCEPT;

}  //  namespace audio
}  //  namespace nns
