﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/hid/hid_Npad.h>
#include <nn/util/util_CharacterEncoding.h>

#include "../NoftWriter_Util.h"
#include "../npad/NoftWriter_NpadController.h"
#include "NoftWriter_WriteNtfWindow.h"

namespace noftwriter { namespace ui {

namespace {

struct NtfHeaderFormatEx
{
    nn::Bit8 headerVersion;         // 0x00
    nn::Bit8 submissionVersion;     // 0x01
    nn::Bit8 reserve1[3];           // 0x02 - 0x04 (NtfHeaderFormat: reserve[0] ~ tagID[0])
    nn::Bit8 productCode[8];        // 0x05 - 0x0C (NtfHeaderFormat: tagID[1] ~ makerCode[0])
    nn::Bit8 reserve2[3];           // 0x0D - 0x0F (NtfHeaderFormat: makerCode[1] ~ makerCode[4])
    nn::Bit8 identifyCode[4];       // 0x10 - 0x13
    nn::Bit8 reserve3[1];           // 0x14
    nn::Bit8 characterId[3];        // 0x15 - 0x17
    nn::Bit8 numberingId[2];        // 0x18 - 0x19
    nn::Bit8 seriesId;              // 0x1A
    nn::Bit8 nfpType;               // 0x1B
    nn::Bit8 reserve4[68];          // 0x1C - 0x5F
    nn::Bit8 hash[32];              // 0x60 - 0x7F
};

}

const int WriteNtfWindow::ItemCountMax = 7;
const int WriteNtfWindow::ItemForDecideStartIndex = 5;

void WriteNtfWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsRepeated(nn::hid::NpadButton::Up::Mask))
    {
        if(m_CursorIndex == ItemForDecideStartIndex)
        {
            m_CursorIndex = ItemCountMax - 1;
        }
        else
        {
            --m_CursorIndex;
        }
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Down::Mask))
    {
        if(m_CursorIndex == ItemCountMax - 1)
        {
            m_CursorIndex = ItemForDecideStartIndex;
        }
        else
        {
            ++m_CursorIndex;
        }
    }
    else if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        if (m_DecideHandler != nullptr)
        {
            m_DecideHandler(controller.GetNpadId(), reinterpret_cast<uintptr_t>(m_Ntf), m_CursorIndex - ItemForDecideStartIndex);
        }
    }

    Window::UpdateInput(controller);
}

void WriteNtfWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    renderer.SetTextScale(1.2f, 1.2f);

    DrawCurrentInfo();

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void WriteNtfWindow::DrawCurrentInfo() NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float dx = rect.x + 12;
    float dy = rect.y + 2;

    DrawCursor(
        dx - 8,
        dy + m_CursorIndex * renderer.GetTextLineHeight(),
        rect.width - 4,
        renderer.GetTextLineHeight() - 8,
        GetCursorFadeColor());

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    float offsetX = 220;

    NtfHeaderFormatEx* header = reinterpret_cast<NtfHeaderFormatEx*>(m_Ntf->data);

    renderer.DrawText(dx, dy, "Product Code");
    renderer.DrawText(dx + offsetX, dy,
                      "%c%c%c%c%c%c%c%c",
                      header->productCode[0],
                      header->productCode[1],
                      header->productCode[2],
                      header->productCode[3],
                      header->productCode[4],
                      header->productCode[5],
                      header->productCode[6],
                      header->productCode[7]);

    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Character ID");
    renderer.DrawText(dx + offsetX, dy,
                      "0x%02X%02X%02X",
                      header->characterId[0],
                      header->characterId[1],
                      header->characterId[2]);

    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Numbering ID");
    renderer.DrawText(dx + offsetX, dy,
                      "0x%02X%02X",
                      header->numberingId[0],
                      header->numberingId[1]);

    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Series ID");
    renderer.DrawText(dx + offsetX, dy,
                      "0x%02X",
                      header->seriesId);

    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "NFP Type");
    renderer.DrawText(dx + offsetX, dy,
                      "0x%02X",
                      header->nfpType);

    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Write(except lock bits) >");

    dy += renderer.GetTextLineHeight();

    renderer.DrawText(dx, dy, "Write >");
}

void WriteNtfWindow::SetNtf(uintptr_t ntf) NN_NOEXCEPT
{
    m_Ntf = reinterpret_cast<Ntf*>(ntf);
}

nn::nfp::NtfWriteType WriteNtfWindow::GetNtfWriteType() NN_NOEXCEPT
{
    if(m_CursorIndex == ItemForDecideStartIndex)
    {
        return nn::nfp::NtfWriteType_NoLock;
    }
    else
    {
        return nn::nfp::NtfWriteType_Lock;
    }
}

}}  // noftwriter::ui
