﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "../graphics/NoftWriter_Renderer.h"
#include "NoftWriter_Spinner.h"

namespace
{

const int SegmentCount = 8;

}  // anonymous

namespace noftwriter { namespace ui {

void Spinner::Update() NN_NOEXCEPT
{
    DisplayObject::Update();

    int frames = SegmentCount * m_Interval;
    m_Duration = (m_Duration + 1) % frames;
};

void Spinner::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    DrawSegments();
}

void Spinner::DrawSegments() NN_NOEXCEPT
{
    const int SegmentPosition[] = { 0, 3, 6, 7, 8, 5, 2, 1 };
    const int LineSegmentCount = 3;

    graphics::Point2D padding = {};
    padding.x = GetWidth()  / 12;
    padding.y = GetHeight() / 12;

    graphics::Size segmentSize = {};
    segmentSize.width  = (GetWidth()  - (LineSegmentCount - 1) * padding.x) / LineSegmentCount;
    segmentSize.height = (GetHeight() - (LineSegmentCount - 1) * padding.y) / LineSegmentCount;

    graphics::Point2D offset = {};
    offset.x = segmentSize.width  + padding.x;
    offset.y = segmentSize.height + padding.y;

    auto CalculateColor = [&](int element, int step)
    {
        return step * (m_EndColor.v[element] - m_StartColor.v[element]) / SegmentCount;
    };
    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);
    renderer.SetTextScale(1.0f, 1.0f);

    for (int i = 0; i < SegmentCount; i++)
    {
        float dx = GetX() + offset.x * (SegmentPosition[i] % LineSegmentCount);
        float dy = GetY() + offset.y * (SegmentPosition[i] / LineSegmentCount);

        int step = (i + m_Duration / m_Interval) % SegmentCount;
        graphics::Color color = m_StartColor;
        color.v[0] += CalculateColor(0, step);
        color.v[1] += CalculateColor(1, step);
        color.v[2] += CalculateColor(2, step);
        color.v[3] += CalculateColor(3, step);
        renderer.SetTextColor(color);
        renderer.DrawText(dx, dy, "■");
    }

    // 中央
    float dx = GetX() + offset.x;
    float dy = GetY() + offset.y;
    renderer.SetTextColor(m_StartColor);
    renderer.DrawText(dx, dy, "■");

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

}}  // noftwriter::ui
