﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define _USE_MATH_DEFINES

#include <cmath>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>

#include "../graphics/NfpDebugTool_Renderer.h"
#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_MenuWindow.h"

namespace nfpdebug { namespace ui {

void MenuWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsRepeated(nn::hid::NpadButton::Down::Mask))
    {
        m_CursorIndex = (m_CursorIndex + 1) % m_ItemCount;
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Up::Mask))
    {
        m_CursorIndex = (m_CursorIndex + m_ItemCount - 1) % m_ItemCount;
    }

    if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        Invoke(m_CursorIndex, controller.GetNpadId());
    }

    Window::UpdateInput(controller);
}

void MenuWindow::Render() NN_NOEXCEPT
{
    if (m_pItems == nullptr || !IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    renderer.SetTextScale(1.2f, 1.2f);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float dx = rect.x + 4;
    float dy = rect.y + 2;
    for (int i = 0; i < m_ItemCount; i++)
    {
        auto&& item = m_pItems[i];
        if (m_CursorIndex == i)
        {
            // 選択中の項目
            DrawCursor(dx, dy, rect.width - 4, renderer.GetTextLineHeight() - 8, GetCursorFadeColor());

            renderer.SetTextColor(GetDisplayColor(
                item.handler == nullptr ?
                graphics::Colors::Silver :
                graphics::Colors::Turquoise));
        }
        else if (item.handler == nullptr)
        {
            // 無効な項目
            renderer.SetTextColor(GetDisplayColor(graphics::Colors::Gray));
        }
        else
        {
            // 通常の項目
            renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));
        }

        renderer.DrawText(dx + 8, dy, "%s", item.caption);

        dy += renderer.GetTextLineHeight();
    }

    renderer.SetTextColor(graphics::Colors::White);
    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void MenuWindow::Invoke(int cursorIndex, const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    if (m_pItems == nullptr ||
        cursorIndex < 0 ||
        cursorIndex >= m_ItemCount)
    {
        return;
    }

    auto&& item = m_pItems[cursorIndex];
    if (item.handler != nullptr)
    {
        item.handler(npadId, item.argument);
    }
}

}}  // nfpdebug::ui
