﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>

#include "NfpDebugTool_Window.h"

namespace nfpdebug { namespace ui {

/**
 * @brief   入力完了を通知するコールバック関数型です。
 */
typedef void (*InputWindowHandlerType)(const nn::hid::NpadIdType&, uint64_t value);

/**
 * @brief   数値の基数です。
 */
enum class NumberBase
{
    Decimal,    //!< 10 進
    Hex         //!< 16 進
};

/**
 * @brief   数値を入力するウィンドウです。
 */
class InputNumberWindow : public Window
{
public:
    InputNumberWindow() NN_NOEXCEPT :
        m_Number(0),
        m_Digits(1),
        m_Base(NumberBase::Decimal),
        m_Text(nullptr),
        m_CursorIndex(0),
        m_DecideHandler(nullptr)
    {
        SetSize(380, 120);
    }

    /**
     * @brief   入力する値を直接指定します。
     */
    void SetNumber(uint64_t number) NN_NOEXCEPT
    {
        m_Number = number;
    }

    /**
     * @brief   入力された値を取得します。
     */
    uint64_t GetNumber() const NN_NOEXCEPT
    {
        return m_Number;
    }

    /**
     * @brief   入力する数値に関する設定を行います。
     *          設定を行うと、入力中の値は 0 に初期化されます。
     */
    void SetNumberRestriction(int digits, NumberBase base) NN_NOEXCEPT
    {
        NN_ASSERT_MINMAX(digits, 1, 16);

        m_Number      = 0;
        m_Digits      = digits;
        m_Base        = base;
        m_CursorIndex = 0;
    }

    /**
     * @brief   表示する文字列を設定します。
     */
    void SetText(const char* text) NN_NOEXCEPT
    {
        m_Text = text;
    }

    /**
     * @brief   決定時のコールバック関数を登録します。
     */
    void SetDecideHandler(InputWindowHandlerType handler) NN_NOEXCEPT
    {
        m_DecideHandler = handler;
    }

    virtual void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT NN_OVERRIDE;
    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    virtual float GetRenderHeight() const NN_NOEXCEPT NN_OVERRIDE
    {
        return GetDefaultLineHeight() * 4 + GetHeaderHeight() + 8;
    }

private:
    void RenderDigits(float x, float y) NN_NOEXCEPT;

private:
    uint64_t                m_Number;           //!< 入力した値
    int                     m_Digits;           //!< 入力する値の桁数
    NumberBase              m_Base;             //!< 入力する値の基数
    const char*             m_Text;             //!< コメントとして表示する文字列
    int                     m_CursorIndex;      //!< カーソルの位置
    InputWindowHandlerType  m_DecideHandler;    //!< 決定時のコールバック
};

}}  // nfpdebug::ui
