﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Macro.h>
#include <nn/util/util_BitFlagSet.h>
#include <nn/nfp/nfp_Types.h>

namespace nfpdebug
{

/**
 * @brief   cabinet の戻り値の有効状態を示すフラグ定義です。
 */
struct CabinetResultFlags
{
    typedef nn::util::BitFlagSet<32, CabinetResultFlags>::Flag<0> DeviceHandle;     //!< NFP デバイスハンドルが有効か
    typedef nn::util::BitFlagSet<32, CabinetResultFlags>::Flag<1> TagInfo;          //!< TagInfo が有効か
    typedef nn::util::BitFlagSet<32, CabinetResultFlags>::Flag<2> RegisterInfo;     //!< RegisterInfo が有効か
};

/**
 * @brief   cabinet の戻り値有効状態の集合を扱う型です。
 */
typedef nn::util::BitFlagSet<32, CabinetResultFlags> CabinetResultFlagSet;

/**
 * @brief   cabinet の処理結果を格納する構造体です。
 */
struct CabinetResult
{
    CabinetResultFlagSet    infoFlags;      //!< メンバの有効状態を示すフラグ
    nn::nfp::DeviceHandle   deviceHandle;   //!< 使用した NFC デバイスのハンドル
    nn::nfp::TagInfo        tagInfo;        //!< 検出したタグ
    nn::nfp::RegisterInfo   registerInfo;   //!< 登録された情報

    /**
     * @brief   メンバの値を初期化します。
     */
    void Initialize() NN_NOEXCEPT
    {
        infoFlags.Reset();
        std::memset(&deviceHandle, 0, sizeof(deviceHandle));
        std::memset(&tagInfo, 0, sizeof(tagInfo));
        std::memset(&registerInfo, 0, sizeof(registerInfo));
    }

    /**
     * @brief   デバイスハンドル情報が存在するか判定します。
     */
    bool HasDeviceHandle() const NN_NOEXCEPT
    {
        return infoFlags.Test<CabinetResultFlags::DeviceHandle>();
    }

    /**
     * @brief   タグ情報が存在するか判定します。
     */
    bool HasTagInfo() const NN_NOEXCEPT
    {
        return infoFlags.Test<CabinetResultFlags::TagInfo>();
    }

    /**
     * @brief   登録情報が存在するか判定します。
     */
    bool HasRegisterInfo() const NN_NOEXCEPT
    {
        return infoFlags.Test<CabinetResultFlags::RegisterInfo>();
    }
};

}  // nfpdebug
