﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo_VibrationSwitch.h"

namespace VibrationDemo
{
    VibrationSwitch::VibrationSwitch()
    {
        SetDefault();
    }

    void VibrationSwitch::SetDefault()
    {
        static_cast<BaseItem*>(this)->SetDefault();
        m_IsEnableSwitch = false;
        m_KbobPosition = 0.f;

        m_BorderColor = nn::util::Color4u8(64, 64, 64, 255);
        m_MainColor = nn::util::Color4u8(216, 216, 216, 255);
        m_SubColor = Color::Green;
        m_EffectColor = Color::Cyan;
    }

    void VibrationSwitch::Update()
    {
        ++m_FrameCount;

        // スイッチの切り替え条件
        //
        // OFF |---------- ---------- ----------| ON
        //               ↑          ↑
        //          [ON > OFF]  [OFF > ON]
        //
        // タップ [ ON <=> OFF ]
        // ボタン単押し [ ON <=> OFF ]

        bool isTapSwitch = false;

        if (gTouch.IsEnable(m_Pos, m_Size))
        {
            static int64_t prevEventNumber = 0;
            nn::hid::GestureState state[nn::hid::GestureStateCountMax];
            auto count = gTouch.GetGestureStates(state, nn::hid::GestureStateCountMax, m_Pos, m_Size);
            for (int i = 0; i < count; ++i)
            {
                if ((state[i].eventNumber > prevEventNumber))
                {
                    switch (state[i].GetGestureType())
                    {
                    case nn::hid::GestureType_Pan:
                        m_KbobPosition = (state[i].x - m_Pos.x - m_Size.y / 2.f) / (m_Size.x - m_Size.y);;
                        prevEventNumber = state[i].eventNumber;
                        break;
                    case nn::hid::GestureType_Tap:
                        isTapSwitch = true;
                        break;
                    default:
                        break;
                    }
                }
            }
        }
        else
        {
            m_KbobPosition = std::max(0.f, std::min(1.f, m_IsEnableSwitch ? (m_KbobPosition + 0.025f) : (m_KbobPosition - 0.025f)));
        }

        m_KbobPosition = std::max(0.f, std::min(1.f, m_KbobPosition));

        const auto oldSwitch = m_IsEnableSwitch;

        if (m_IsEnableSwitch && m_KbobPosition < 0.25f) { m_IsEnableSwitch = false; }
        if (!m_IsEnableSwitch && m_KbobPosition > 0.75f) { m_IsEnableSwitch = true; }

        if (!m_Option.Test<nns::hidfw::layout::LayoutOption::OnOffMode>() && m_State.Test<nns::hidfw::layout::LayoutState::Selected>())
        {
            m_IsEnableSwitch ^= true;
            m_KbobPosition = m_IsEnableSwitch ? 0.8f : 0.2f;
            Cancel();
        }

        if (oldSwitch != m_IsEnableSwitch)
        {
            SetVibrationBalance(m_IsEnableSwitch ? 1.f : -1.f);
            int key = nns::hidfw::layout::LayoutState::Selected::Index;
            auto itr = m_BnvibFileName.find(key);
            if (itr != m_BnvibFileName.end())
            {
                SetBnvibFile(itr->second.fileName.c_str(), itr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            itr = m_SoundFileName.find(key);
            if (itr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(itr->second.fileName.c_str(), itr->second.volume);
            }
        }

        SetVibrationBalance(0.f);

        if (
            !m_OldState.Test<nns::hidfw::layout::LayoutState::Hover>() &&
            m_State.Test<nns::hidfw::layout::LayoutState::Hover>()
            )
        {
            int key = nns::hidfw::layout::LayoutState::Hover::Index;
            auto vibItr = m_BnvibFileName.find(key);
            if (vibItr != m_BnvibFileName.end())
            {
                SetBnvibFile(vibItr->second.fileName.c_str(), vibItr->second.volume);
                PlayVibration(gControllerSequenceManager.GetMasterControllerId());
            }
            auto seItr = m_SoundFileName.find(key);
            if (seItr != m_SoundFileName.end())
            {
                gAudioManager.PlayWav(seItr->second.fileName.c_str(), 0.5f);
            }
        }
    }

    void VibrationSwitch::PrintText()
    {
        m_OldState = m_State;
        m_State.Set(nns::hidfw::layout::LayoutState::Canceled::Index, false);
    }

    void VibrationSwitch::Draw()
    {
        const auto radius = m_Size.y / 2.f;
        const auto knobCenterPos = nn::util::MakeFloat2(std::min(m_Pos.x + radius + (m_Size.x - m_Size.y) * m_KbobPosition, m_Pos.x + m_Size.x - radius), m_Pos.y + radius);
        const auto shadowPos = nn::util::MakeFloat2(knobCenterPos.x + 0.5f, knobCenterPos.y + 0.5f);

        const nn::util::Color4u8 bgColor = nn::util::Color4u8(48, 48, 48, 255);
        const nn::util::Color4u8 transparenceBgColor = nn::util::Color4u8(bgColor.GetR(), bgColor.GetG(), bgColor.GetB(), 0);

        const nn::util::Color4u8 disableColor = nn::util::Color4u8(96, 96, 96, 255);
        const nn::util::Color4u8 transparenceDisableColor = nn::util::Color4u8(disableColor.GetR(), disableColor.GetG(), disableColor.GetB(), 0);

        const nn::util::Color4u8 shadowColor = nn::util::Color4u8(32, 32, 32, 192);

        const nn::util::Color4u8 knobColor = m_IsEnableSwitch ? m_MainColor : nn::util::Color4u8(m_MainColor.GetR() / 2, m_MainColor.GetG() / 2, m_MainColor.GetB() / 2, 255);

        const nn::util::Color4u8 transparenceEffectColor = nn::util::Color4u8(m_EffectColor.GetR(), m_EffectColor.GetG(), m_EffectColor.GetB(), 0);
        const nn::util::Color4u8 transparenceSubColor = nn::util::Color4u8(m_SubColor.GetR(), m_SubColor.GetG(), m_SubColor.GetB(), 0);

        auto pos = m_Pos;
        pos.x = knobCenterPos.x - radius;

        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, bgColor, transparenceBgColor);
        gDrawer.Draw2DRoundedFrame(pos, nn::util::MakeFloat2(m_Size.x - (knobCenterPos.x - m_Pos.x - radius), m_Size.y), 1.f, 64, 2.f);
        gDrawer.SetColor(bgColor);
        gDrawer.Draw2DRoundedRect(pos, nn::util::MakeFloat2(m_Size.x - (knobCenterPos.x - m_Pos.x - radius), m_Size.y), 1.f, 64);

        if (m_IsEnableSwitch)
        {
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, m_SubColor, transparenceSubColor);
        }
        else
        {
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, disableColor, transparenceDisableColor);
        }
        gDrawer.Draw2DRoundedFrame(m_Pos, nn::util::MakeFloat2(knobCenterPos.x - m_Pos.x + radius, m_Size.y), 1.f, 64, 2.f);
        gDrawer.SetColor(m_IsEnableSwitch ? m_SubColor : disableColor);
        gDrawer.Draw2DRoundedRect(m_Pos, nn::util::MakeFloat2(knobCenterPos.x - m_Pos.x + radius, m_Size.y), 1.f, 64);

        gDrawer.SetColor(shadowColor);
        gDrawer.Draw2DCircleFrame(shadowPos, radius * 0.8f, 1.5f, 64);

        gDrawer.SetColor(knobColor);
        gDrawer.Draw2DCircle(knobCenterPos, radius * 0.8f, 64);

        if (m_State.Test<nns::hidfw::layout::LayoutState::Hover>())
        {
            pos = nn::util::MakeFloat2(m_Pos.x - 7.f, m_Pos.y - 7.f);
            auto size = nn::util::MakeFloat2(m_Size.x + 14.f, m_Size.y + 14.f);
            gDrawer.SetColor(m_EffectColor);
            gDrawer.Draw2DRoundedFrame(pos, size, 1.f, 64, 2.f);

            pos = nn::util::MakeFloat2(m_Pos.x - 10.f, m_Pos.y - 10.f);
            size = nn::util::MakeFloat2(m_Size.x + 20.f, m_Size.y + 20.f);
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, transparenceEffectColor, m_EffectColor);
            gDrawer.Draw2DRoundedFrame(pos, size, 1.f, 64, -1.f);

            pos = nn::util::MakeFloat2(m_Pos.x - 6.f, m_Pos.y - 6.f);
            size = nn::util::MakeFloat2(m_Size.x + 12.f, m_Size.y + 12.f);
            gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Out, transparenceEffectColor, m_EffectColor);
            gDrawer.Draw2DRoundedFrame(pos, size, 1.f, 64, 1.f);
        }
    }
}
