﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

 /**
 * @file
 * @brief       振動の再生やファイル読み込みを行うための API の宣言
 */

#pragma once

#include <string>
#include <map>

#include <nn/mem.h>
#include "hidfw/hid.h"

namespace VibrationDemo
{
    /**
    * @brief       振動機能で利用するメモリの管理を行うクラスです
    * @details     Bnvib ファイルをメモリ上に展開する場合に利用します
    */
    template<size_t MemoryHeapSize>
    class BnvibFileHeap
    {
        NN_DISALLOW_COPY(BnvibFileHeap);
        NN_DISALLOW_MOVE(BnvibFileHeap);

    public:
        BnvibFileHeap() NN_NOEXCEPT
        {
            Allocator.Initialize(VibrationHeap, MemoryHeapSize);
        }
        ~BnvibFileHeap() NN_NOEXCEPT
        {
            Allocator.Finalize();
        }
        void* Allocate(const size_t size) NN_NOEXCEPT
        {
            return Allocator.Allocate(size);
        }

    private:
        NN_OS_ALIGNAS_THREAD_STACK char  VibrationHeap[MemoryHeapSize];
        nn::mem::StandardAllocator       Allocator;
    };

    /**
    * @brief       Bnvib ファイルの情報を扱う構造体
    * @details     Bnvib ファイルの格納先とファイルサイズが保存されます
    */
    struct BnvibFile
    {
        uint8_t*    address;
        size_t      fileSize;
    };

    /**
    * @brief       NpadID とスタイルからキーを生成します
    * @details     振動させる振動子を判別するため NpadId とスタイルからキーを生成します
    */
    struct VibrationKeyType
    {
        bool operator < (const VibrationKeyType& rhs) const NN_NOEXCEPT
        {
            // ID を優先して比較します
            if (id < rhs.id)
            {   // 自分のIDの方が小さければ true
                return true;
            }
            else if (id > rhs.id)
            {   // 自分のIDの方が大きければ false
                return false;
            }
            // ID が同じ場合はスタイル値を基準に比較します
            int index = 0;
            // 自身と比較対象の有効な操作スタイルで最もインデックスが若いものを検索
            for (; index < style.GetCount() && (!style.Test(index) && !rhs.style.Test(index)); ++index) {}

            return (style.Test(index) && !rhs.style.Test(index));
        }

        static VibrationKeyType Make(nn::hid::NpadIdType id, nn::hid::NpadStyleSet style) NN_NOEXCEPT
        {
            NN_ASSERT_LESS_EQUAL(style.CountPopulation(), 1);
            VibrationKeyType result;
            result.id = id;
            result.style = style;
            return result;
        }

        nn::hid::NpadIdType id;
        nn::hid::NpadStyleSet style;
    };

    template<const int HANDLECOUNT>
    struct VibartionDeviceType
    {
        int                                 handleCount;
        nn::hid::VibrationDeviceHandle      handles[HANDLECOUNT];
        nn::hid::VibrationNodeConnection    connections[HANDLECOUNT];   // デフォルトで1ハンドルにつき一つの接続を持つ
    };

    template<const int STATECOUNT, const int HANDLECOUNT>
    struct VibrationValues
    {
        int64_t samplingNumber[STATECOUNT];
        nn::hid::VibrationValue vibrationValue[STATECOUNT][HANDLECOUNT];
    };

    //=================================================
    // 振動管理クラス
    //=================================================
    //
    // VibrationNode_1 --                          -- FullKey_Target
    //                    ＞ [ NpadId::No1 Mixer ] -- JoyDual_Target
    // VibrationNode_2 --                          -- JoyLeft_Target ...etc
    //
    class VibrationManager
    {
    public:
        // 各種最大値の定義
        static const int                                            HandleCountMax = 2;
        static const int                                            NpadCountMax = 9;
        static const int                                            NpadStyleCountMax = 5;
        static const int64_t                                        VibrationValueCountMax = 5;

    public:
        typedef VibartionDeviceType<HandleCountMax>                 VibartionDevice;

    public:
        static VibrationManager& GetInstance() NN_NOEXCEPT;

        static void UpdateVibrationNode(void *arg) NN_NOEXCEPT;

        static const std::map<std::string, BnvibFile>& GetBnvibFileList() NN_NOEXCEPT;

        void UpdateNpadStyle() NN_NOEXCEPT;

        void Initialize(const nn::hid::NpadIdType* npadIds, const int npadIdCount, const nn::hid::NpadStyleSet npadStyleSet) NN_NOEXCEPT;

        size_t LoadBnvibForMemory() NN_NOEXCEPT;

        bool GetBnvibFile(const char* fileName, void** address, size_t* outFileSize) NN_NOEXCEPT;

        void StartUpdateThread() NN_NOEXCEPT;

        void StopUpdateThread() NN_NOEXCEPT;

        void StopDefaultVibrationNodes() NN_NOEXCEPT;

        void DisconnectAll() NN_NOEXCEPT;

        nn::hid::VibrationNode* GetVibrationMixer(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType type) NN_NOEXCEPT;

        bool IsEnableVibrationNpadId(nn::hid::NpadIdType id) NN_NOEXCEPT;

        int GetEnableVibrationNpadIds(nn::hid::NpadIdType* pOutIds, const int count) NN_NOEXCEPT;

        void EnableVibration(nn::hid::NpadIdType* pIds, const int count) NN_NOEXCEPT;

        void EnableVibration() NN_NOEXCEPT;

        void DisableVibration(nn::hid::NpadIdType* pIds, const int count) NN_NOEXCEPT;

        void DisableVibration() NN_NOEXCEPT;

        const std::map<VibrationKeyType, VibartionDevice*>& GetVibrationDevices() NN_NOEXCEPT
        {
            return m_pVibrationDevice;
        }

        nn::hid::VibrationWriter& GetVibrationWriter(nn::hid::NpadIdType id) NN_NOEXCEPT;

        int ConnectVibrationNode(nn::hid::VibrationNode* pNode, nn::hid::VibrationNodeConnection* pConnections, nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType type) NN_NOEXCEPT;

        int ConnectVibrationNode(nn::hid::VibrationNode* pNode, nn::hid::VibrationNodeConnection* pConnections, nn::hid::NpadIdType id) NN_NOEXCEPT;

        void AddSourceConnections(nn::hid::VibrationNodeConnection* pConnection) NN_NOEXCEPT;

        void RemoveSourceConnections(nn::hid::VibrationNodeConnection* pConnection) NN_NOEXCEPT;

        nn::hid::VibrationValue GetActualVibrationValue(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType deviceType) NN_NOEXCEPT;

        int64_t GetActualVibrationValues(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType deviceType, nn::hid::VibrationValue* pValues, int64_t maxCount, int64_t samplingNumber, int64_t* outLastSamplingNumber = nullptr) NN_NOEXCEPT;
    private:
        // 振動の更新スレッドで使用
        static const size_t                                         VibrationThreadStackSize = 64 * 1024;
        static const size_t                                         VibrationMemoryHeapSize = 64 * 1024 * 1024;
    private:
        static NN_OS_ALIGNAS_THREAD_STACK char                      ThreadStack[VibrationThreadStackSize];
        static nn::os::ThreadType                                   VibrationThread;
        static nn::os::TimerEventType                               VibrationTimerEvent;
        static std::map<VibrationKeyType, VibrationValues<VibrationValueCountMax, HandleCountMax>> CurrentVibrationValues;
        static int64_t                                              SamplingNumber;

        static BnvibFileHeap<VibrationMemoryHeapSize>               VibrationHeap;
        static std::map<std::string, BnvibFile>                     VibrationFileList;

        // 初期化時に何台のNpadを利用するよう指定されたか
        int                                                         m_NpadIdCount;

        // 最大何台分のスタイルが利用可能か
        int                                                         m_NpadStyleCount;

        // 有効化されているNpadIdのインデックスリスト
        std::map<nn::hid::NpadIdType, int>                          m_NpadIndices;

        // 更新が有効設定されているNpadId
        std::vector<nn::hid::NpadIdType>                            m_IsEnableVibrationNpadIds;

        VibartionDevice*                                            m_pVibrationDeviceData[NpadCountMax];
        std::map<VibrationKeyType, VibartionDevice*>                m_pVibrationDevice;
        std::map<VibrationKeyType, std::vector<nn::hid::VibrationTarget*>>  m_VibrationTargets;
        std::map<VibrationKeyType, std::vector<nn::hid::VibrationNodeConnection*>>       m_VibrationMixerConnections;

        bool                                                        m_IsInitialized;

        // 標準装備の Player と Writer
        nn::hid::VibrationWriter                                    m_DefaultVibrationWriter[NpadCountMax];
        nn::hid::VibrationPlayer                                    m_DefaultVibrationPlayer[NpadCountMax];

        // 最終出力結果を格納するミキサー
        nn::hid::VibrationMixer                                     m_VibrationMixer[NpadCountMax][HandleCountMax];

        std::vector<nn::hid::VibrationNodeConnection*>              m_SourceConnections;
    };

    //=================================================
    // 振動ソース
    //=================================================
    template <typename T>
    struct VibrationSource
    {
        VibrationSource() NN_NOEXCEPT
        {
            connectionNodes = nullptr;
            isSelectDeviceType = false;
            deviceType = nn::hid::NpadJoyDeviceType_Left;
            SetDefaultModulation(nn::hid::VibrationModulation::Make());
            for (auto& connection : connections)
            {
                VibrationManager::GetInstance().AddSourceConnections(&connection);
            }
        }

        ~VibrationSource() NN_NOEXCEPT
        {
            for (auto& connection : connections)
            {
                VibrationManager::GetInstance().RemoveSourceConnections(&connection);
            }
        }

        int Connection(nn::hid::NpadIdType id) NN_NOEXCEPT
        {
            npadId = id;

            isSelectDeviceType = false;
            return VibrationManager::GetInstance().ConnectVibrationNode(&node, connections, id);
        }

        int Connection(nn::hid::NpadIdType id, nn::hid::NpadJoyDeviceType type) NN_NOEXCEPT
        {
            npadId = id;

            isSelectDeviceType = true;
            deviceType = type;

            return VibrationManager::GetInstance().ConnectVibrationNode(&node, connections, id, type);
        }

        bool SetModulation(const nn::hid::VibrationModulation& modulation, nn::hid::NpadJoyDeviceType deviceType) NN_NOEXCEPT
        {
            auto pNode = VibrationManager::GetInstance().GetVibrationMixer(npadId, deviceType);
            if (pNode != nullptr)
            {
                static_cast<nn::hid::VibrationNode*>(&node)->SetModulationTo(pNode, modulation);
                return true;
            }
            return false;
        }

        bool GetModulation(nn::hid::VibrationModulation* outModulation, nn::hid::NpadJoyDeviceType deviceType) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(outModulation);
            auto pNode = VibrationManager::GetInstance().GetVibrationMixer(npadId, deviceType);
            if (pNode != nullptr)
            {
                *outModulation = static_cast<nn::hid::VibrationNode*>(&node)->GetModulationTo(pNode);
                return true;
            }
            return false;
        }

        void Disconnection() NN_NOEXCEPT
        {
            for (auto i = 1; i < VibrationManager::HandleCountMax; ++i)
            {
                if (connections[i].IsConnected())
                {
                    connections[i].Disconnect();
                }
            }
        }

        bool GetConnectedDeviceType(nn::hid::NpadJoyDeviceType* outDeviceType) NN_NOEXCEPT
        {
            NN_ASSERT_NOT_NULL(outDeviceType);
            *outDeviceType = deviceType;
            return isSelectDeviceType;
        }

        void SetDefaultModulation(const nn::hid::VibrationModulation& modulation) NN_NOEXCEPT
        {
            for (auto& mod : defaultModulation)
            {
                mod = modulation;
            }
        }

        void SetDefaultModulation(const nn::hid::VibrationModulation& modulation, nn::hid::NpadJoyDeviceType deviceType) NN_NOEXCEPT
        {
            NN_ASSERT_GREATER_EQUAL(deviceType, 0);
            NN_ASSERT_LESS(deviceType, 2);
            defaultModulation[deviceType == nn::hid::NpadJoyDeviceType_Left ? 0 : 1] = modulation;
        }

        void ResetModulation() NN_NOEXCEPT
        {
            SetModulation(defaultModulation[0], nn::hid::NpadJoyDeviceType_Left);
            SetModulation(defaultModulation[1], nn::hid::NpadJoyDeviceType_Right);
        }

        T node;
        nn::hid::NpadIdType npadId;
        nn::hid::NpadJoyDeviceType deviceType;
        nn::hid::VibrationNodeConnection connections[VibrationManager::HandleCountMax];
        nn::hid::VibrationNode* connectionNodes;
        nn::hid::VibrationModulation defaultModulation[2];
        bool isSelectDeviceType;
    };
}
#define gVibrationManager (VibrationDemo::VibrationManager::GetInstance())
