﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/system/hid_UniquePad.h>

#include "util.h"

namespace nns { namespace hid { namespace util {

/**
 * @brief   接続されている UniquePad を管理するクラスです。
 */
class UniquePadManager final
{
    NN_DISALLOW_COPY(UniquePadManager);
    NN_DISALLOW_MOVE(UniquePadManager);

public:
    UniquePadManager() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;

    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   接続されているコントローラーが変化したときに通知するイベントを設定します。
     */
    void BindUpdateEvent(nn::os::LightEventType* pEvent) NN_NOEXCEPT;

    /**
     * @brief   接続されているコントローラーが変化したときに通知するイベントを解除します。
     */
    void UnbindUpdateEvent() NN_NOEXCEPT;

    /**
     * @brief   接続されているコントローラーの台数を返します。
     */
    int GetPadCount() const NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        return m_PadCount;
    }

    /**
     * @brief   接続可能なコントローラーの台数を返します。
     */
    int GetPadCountMax() const NN_NOEXCEPT
    {
        return static_cast<int>(NN_ARRAY_SIZE(m_PadInfos));
    }

    /**
     * @brief   コントローラーが 1 台以上接続されているか判定します。
     */
    bool IsDeviceConnected() const NN_NOEXCEPT
    {
        return GetPadCount() > 0;
    }

    /**
     * @brief   指定したコントローラーインデックスに対応する NpadId を取得します。
     */
    nn::Result GetNpadId(nn::hid::NpadIdType* pOutId, int padIndex) NN_NOEXCEPT;

    /**
     * @brief   指定したコントローラーインデックスに対応する UniquePadId を取得します。
     */
    nn::Result GetUniquePadId(nn::hid::system::UniquePadId* pOutId, int padIndex) NN_NOEXCEPT;

    /**
     * @brief   指定したコントローラーの FW バージョンを取得します。
     */
    bool GetFirmwareVersion(nn::hid::system::FirmwareVersion* pOutVersion, int padIndex) NN_NOEXCEPT;

    /**
     * @brief   指定したコントローラーが FW 更新中か確認します。
     */
    bool IsFirmwareUpdating(nn::hid::system::UniquePadId id) NN_NOEXCEPT;

    /**
     * @brief   コントローラーの FW バージョン再取得リクエストを発行します。
     */
    void RequestInvalidateFirmwareVersion() NN_NOEXCEPT;

    /**
     * @brief   FW 更新中のコントローラー情報を更新します。
     */
    void UpdateFirmwareUpdatingDevice() NN_NOEXCEPT;

    /**
     * @brief   コントローラー一覧を再取得します。
     */
    void Refresh() NN_NOEXCEPT;

    /**
     * @brief   コントローラーの情報を更新します。
     */
    void Update() NN_NOEXCEPT;

    /**
     * @brief   FW バージョン再取得リクエストが出ていた場合、FW バージョンを再取得します。
     */
    void InvalidateFirmwareVersionIfNeeded() NN_NOEXCEPT;

private:
    struct PadInfo
    {
        nn::hid::system::UniquePadId id;
        FirmwareVersionReader        versionReader;
        bool                         isUpdating;

        // メンバのクリア
        void Clear() NN_NOEXCEPT;
    };

private:
    mutable nn::os::Mutex m_Mutex;

    bool                    m_IsInitialized;
    bool                    m_IsInvalidateVersionRequested;
    nn::os::SystemEventType m_UniquePadConnectionEvent;
    nn::os::LightEventType* m_pUpdateEvent;
    PadInfo                 m_PadInfos[nn::hid::system::UniquePadIdCountMax];
    int                     m_PadCount;
};

}}}  // nns::hid::util
