﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "Config.h"

namespace
{

const char ConfigSaveFile[] = "save:/config.dat";

// 設定値の内部構造
struct ConfigData
{
    bool isForceUpdateEnabled;

    char _reserved[1023];

    void Reset() NN_NOEXCEPT
    {
        isForceUpdateEnabled = false;
        std::memset(_reserved, 0, sizeof(_reserved));
    }
};

NN_STATIC_ASSERT(sizeof(ConfigData) == 1024);

ConfigData* GetConfig(void* pData)
{
    return reinterpret_cast<ConfigData*>(pData);
}

const ConfigData* GetConfig(const void* pData)
{
    return reinterpret_cast<const ConfigData*>(pData);
}

}  // anonymous

namespace nns { namespace hid { namespace util {

Config* Config::GetInstance() NN_NOEXCEPT
{
    static Config s_Instance;
    return &s_Instance;
}

Language Config::GetLanguage() NN_NOEXCEPT
{
    static char s_Language[10] = "";

    if (s_Language[0] == '\0')
    {
        // 本体の言語設定を取得
        auto language = nn::oe::GetDesiredLanguage();
        nn::util::Strlcpy(s_Language, language.string, sizeof(s_Language));
    }

    return nn::util::Strncmp(s_Language, "ja", sizeof(s_Language)) == 0 ?
        nns::hid::util::Language::Japanese :
        nns::hid::util::Language::English;
}

Config::Config() NN_NOEXCEPT
    : m_Mutex(false)
    , m_DataStorage()
    , m_IsChanged(false)
{
}

nn::Result Config::Load() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    reinterpret_cast<ConfigData*>(m_DataStorage)->Reset();
    m_IsChanged = false;

    nn::fs::FileHandle file;
    NN_RESULT_DO(
        nn::fs::OpenFile(&file, ConfigSaveFile, nn::fs::OpenMode_Read)
    );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };

    NN_RESULT_DO(
        nn::fs::ReadFile(file, 0, m_DataStorage, sizeof(m_DataStorage));
    );

    NN_RESULT_SUCCESS;
}

nn::Result Config::Save() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (!IsDirty())
    {
        // 保存不要
        NN_RESULT_SUCCESS;
    }

    nn::fs::FileHandle file;
    {
        auto result = nn::fs::CreateFile(ConfigSaveFile, sizeof(m_DataStorage));
        if (result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            // AlreadyExists 以外の失敗はエラー
            NN_RESULT_THROW(result);
        }
    }

    bool isSuccess = false;

    NN_RESULT_DO(
        nn::fs::OpenFile(&file, ConfigSaveFile, nn::fs::OpenMode_Write)
    );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
        if (isSuccess)
        {
            nn::fs::Commit("save");
        }
    };

    NN_RESULT_DO(
        nn::fs::WriteFile(
            file,
            0,
            m_DataStorage,
            sizeof(m_DataStorage),
            nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)
        )
    );

    isSuccess = true;

    m_IsChanged = false;

    NN_RESULT_SUCCESS;
}

bool Config::IsDirty() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return m_IsChanged;
}

bool Config::IsForceUpdateEnabled() const NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    return GetConfig(m_DataStorage)->isForceUpdateEnabled;
}

void Config::SetForceUpdateEnabled(bool isEnabled) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    GetConfig(m_DataStorage)->isForceUpdateEnabled = isEnabled;
    m_IsChanged = true;
}

}}}  // nns::hid::util
