﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_ScopeExit.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/gui/SimpleGfx_Gui.h"
#include "../util/FirmwareImage.h"
#include "../util/StringTable.h"
#include "../util/util.h"
#include "FirmwareImageButton.h"

namespace nns { namespace hid { namespace ui {

FirmwareImageButton::FirmwareImageButton() NN_NOEXCEPT
    : m_pImage(nullptr)
{
    SetButtonMaskForPress(util::DecideButtonMask);
}

void FirmwareImageButton::SetImage(const util::FirmwareImage* pImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImage);

    m_pImage = pImage;
}

void FirmwareImageButton::RenderContents() NN_NOEXCEPT
{
    nns::sgx::ScopedFontContextSaver saver;

    nns::sgx::SetTextSize(24);

    auto& str = nns::hid::util::StringTable::GetInstance();

    if (!IsValid())
    {
        nns::sgx::SetTextColor(nns::sgx::Colors::Gray());
        nns::sgx::DrawText(0, 16, str.Get("FlashWriter_NoFileText"));
        return;
    }

    nns::sgx::SetTextColor(GetTextColor());
    nns::sgx::DrawText(0, 0, m_pImage->GetFileName());

    nns::sgx::SetTextSize(20);
    nns::sgx::SetTextColor(nns::sgx::Colors::Gray());
    nns::sgx::DrawText(
        16, 32, "%s - %s  %s",
        m_pImage->GetTargetDeviceName(),
        m_pImage->GetTargetChipName(),
        m_pImage->GetVersion()
    );
    char sizeText[100];
    nn::util::SNPrintf(
        sizeText,
        sizeof(sizeText),
        "%.2f KB",
        m_pImage->GetFileSize() / 1024.0f
    );
    nns::sgx::DrawText(0, 32, GetClientAreaSize().width, 24, nns::sgx::TextAlignment::Right, sizeText);
}

}}}  // nns::hid::ui
