﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   プログレスバーを扱うクラスです。
 */
class ProgressBar :
    public DisplayObject
{
    NN_DISALLOW_COPY(ProgressBar);
    NN_DISALLOW_MOVE(ProgressBar);

public:
    ProgressBar() NN_NOEXCEPT
        : m_Value(0)
        , m_MinValue(0)
        , m_MaxValue(100)
        , m_DispValue(0.0f)
    {
        SetSize(100, 32);
    }

    /**
     * @brief   現在値を取得します。
     */
    int GetValue() const NN_NOEXCEPT
    {
        return m_Value;
    }

    /**
     * @brief   現在値を設定します。
     */
    void SetValue(int value) NN_NOEXCEPT
    {
        SetValue(value, false);
    }

    /**
     * @brief   現在値を設定します。
     */
    void SetValue(int value, bool isSmooth) NN_NOEXCEPT
    {
        m_Value = std::min(std::max(value, m_MinValue), m_MaxValue);

        if (!isSmooth)
        {
            m_DispValue = m_Value;
        }
    }

    /**
     * @brief   最小値を取得します。
     */
    int GetMinValue() const NN_NOEXCEPT
    {
        return m_MinValue;
    }

    /**
     * @brief   最小値を設定します。
     */
    void SetMinValue(int value) NN_NOEXCEPT
    {
        m_MinValue = std::min(value, m_MaxValue);

        // 現在値を補正
        SetValue(GetValue());
    }

    /**
     * @brief   最大値を取得します。
     */
    int GetMaxValue() const NN_NOEXCEPT
    {
        return m_MaxValue;
    }

    /**
     * @brief   最大値を設定します。
     */
    void SetMaxValue(int value) NN_NOEXCEPT
    {
        m_MaxValue = std::max(value, m_MinValue);

        // 現在値を補正
        SetValue(GetValue());
    }

    /**
     * @brief   最大値に対する現在値の割合を取得します。
     */
    float GetRate() const NN_NOEXCEPT
    {
        return (m_Value - m_MinValue) * 1.0f / (m_MaxValue - m_MinValue);
    }

    /**
     * @brief   現在値が最大値に達しているか判定します。
     */
    bool IsMaximum() const NN_NOEXCEPT
    {
        return m_Value == m_MaxValue;
    }

    /**
     * @brief   クライアント領域の矩形を取得します。
     */
    virtual Rectangle GetClientArea() const NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    /**
     * @brief   表示上の現在値を取得します。
     */
    int GetDisplayValue() const NN_NOEXCEPT
    {
        return static_cast<int>(m_DispValue);
    }

    /**
     * @brief   最大値に対する現在値の表示上の割合を取得します。
     */
    float GetDispRate() const NN_NOEXCEPT
    {
        return (m_DispValue - m_MinValue) * 1.0f / (m_MaxValue - m_MinValue);
    }

    /**
     * @brief   バー本体を描画します。
     */
    virtual void DrawBar() NN_NOEXCEPT;

private:
    int m_Value;        //!< 現在値
    int m_MinValue;     //!< 最小値
    int m_MaxValue;     //!< 最大値
    float m_DispValue;  //!< 表示上の現在値
};

}}}  // nns::sgx::gui
