﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommonDefs.h"
#include "SimpleGfx_GuiAttributes.h"

namespace nns { namespace sgx { namespace gui {

// 前方宣言
class UiContainer;

/**
 * @brief   可視オブジェクトのベースクラスです。
 */
class DisplayObject :
    public Prohibitable,
    public Draggable,
    public Focusable
{
    NN_DISALLOW_COPY(DisplayObject);
    NN_DISALLOW_MOVE(DisplayObject);

public:
    DisplayObject() NN_NOEXCEPT;

    virtual ~DisplayObject() NN_NOEXCEPT {}

    /**
     * @brief   オブジェクトの所有者を取得します。
     */
    UiContainer* GetOwner() NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_pOwner;
    }

    /**
     * @brief   オブジェクトの所有者を設定します。
     */
    void SetOwner(UiContainer* pOwner) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_pOwner = pOwner;
    }

    /**
     * @brief   コンテナ内の相対位置の X 座標を取得します。
     */
    float GetX() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Position.x;
    }

    /**
     * @brief   コンテナ内の相対位置の Y 座標を取得します。
     */
    float GetY() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Position.y;
    }

    /**
     * @brief   画面上の絶対表示位置の X 座標を取得します。
     */
    float GetAbsoluteX() const NN_NOEXCEPT;

    /**
     * @brief   画面上の絶対表示位置の Y 座標を取得します。
     */
    float GetAbsoluteY() const NN_NOEXCEPT;

    /**
     * @brief   画面上の絶対表示位置を取得します。
     */
    void GetAbsolutePosition(Point2D* pOutPosition) const NN_NOEXCEPT;

    /**
     * @brief   コンテナ内の相対描画位置を取得します。
     */
    void GetRenderPosition(Point2D* pOutPosition) const NN_NOEXCEPT;

    /**
     * @brief   表示位置を設定します。
     */
    void SetPosition(const Point2D& position) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_Position = position;
    }

    /**
     * @brief   表示位置を設定します。
     */
    void SetPosition(float x, float y) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_Position.x = x;
        m_Position.y = y;
    }

    /**
     * @brief   幅を取得します。
     */
    virtual float GetWidth() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Size.width;
    }

    /**
     * @brief   高さを取得します。
     */
    virtual float GetHeight() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Size.height;
    }

    /**
     * @brief   サイズを取得します。
     */
    virtual void GetSize(Size* pOutSize) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutSize);

        *pOutSize = GetSize();
    }

    /**
     * @brief   サイズを取得します。
     */
    virtual Size GetSize() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_Size;
    }

    /**
     * @brief   サイズを設定します。
     */
    virtual void SetSize(const Size& size) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_Size = size;
    }

    /**
     * @brief   サイズを設定します。
     */
    virtual void SetSize(float width, float height) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(width,  0.0f);
        NN_ASSERT_GREATER_EQUAL(height, 0.0f);
        NNS_SGX_GUI_SCOPED_LOCK;

        m_Size.width  = width;
        m_Size.height = height;
    }

    /**
     * @brief   Z オーダーを取得します。
     */
    int GetZ() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_OrderZ;
    }

    /**
     * @brief   Z オーダーを設定します。
     */
    void SetZ(int z) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_OrderZ = z;
    }

    /**
     * @brief   指定した方向の接続点を取得します。
     */
    Point2D GetAnchorPoint(Direction dir) NN_NOEXCEPT;

    /**
     * @brief   ドラッグを認識する領域を取得します。
     */
    virtual void GetDraggableArea(Rectangle* pOutArea) const NN_NOEXCEPT;

    /**
     * @brief   タッチ操作を受け付ける領域を取得します。
     */
    virtual Rectangle GetHitArea() const NN_NOEXCEPT;

    void GetHitArea(Rectangle* pOutValue) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutValue);

        *pOutValue = GetHitArea();
    }

    /**
     * @brief   可視状態を取得します。
     */
    virtual bool IsVisible() const NN_NOEXCEPT;

    /**
     * @brief   可視状態を設定します。
     */
    virtual void SetVisible(bool isVisible) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_IsVisible = isVisible;
    }

    /**
     * @brief   可視状態にします。
     */
    virtual void Show() NN_NOEXCEPT
    {
        SetVisible(true);
    }

    /**
     * @brief   不可視状態にします。
     */
    virtual void Hide() NN_NOEXCEPT
    {
        SetVisible(false);
    }

    /**
     * @brief   決定エフェクトの有効状態を設定します。
     */
    void SetDecideEffectEnabled(bool isEnabled) NN_NOEXCEPT
    {
        m_IsDecideEffectEnabled = isEnabled;
    }

    /**
     * @brief   決定エフェクトが有効化されているか取得します。
     */
    bool IsDecideEffectEnabled() const NN_NOEXCEPT
    {
        return m_IsDecideEffectEnabled;
    }

    /**
     * @brief   不透明度を取得します。
     */
    uint8_t GetOpacity() const NN_NOEXCEPT
    {
        return m_Opacity;
    }

    /**
     * @brief   オーナー要素の設定を反映した不透明度を取得します。
     */
    uint8_t GetDisplayOpacity() const NN_NOEXCEPT;

    /**
     * @brief   不透明度を設定します。
     */
    void SetOpacity(uint8_t opacity) NN_NOEXCEPT
    {
        m_Opacity = opacity;
    }

    /**
     * @brief   有効状態を取得します。
     */
    virtual bool IsEnabled() const NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   キー操作によってフォーカス可能かどうか取得します。
     */
    virtual bool IsKeyFocusable() const NN_NOEXCEPT
    {
        return false;
    }

    /**
     * @brief   コンテナかどうか判定します。
     */
    virtual bool IsContainer() const NN_NOEXCEPT
    {
        return false;
    }

    /**
     * @brief   フレーム毎の更新処理を行います。
     */
    virtual void Update() NN_NOEXCEPT;

    /**
     * @brief   キー入力の処理を行います。
     *
     * @return  入力がハンドリングされたら true
     */
    virtual bool UpdateKeyInput() NN_NOEXCEPT { return false; }

    /**
     * @brief   タッチパネル入力の処理を行います。
     *
     * @return  入力がハンドリングされたら true
     */
    virtual bool UpdateTouchInput() NN_NOEXCEPT { return false; }

    /**
     * @brief   タッチパネル入力の処理をキャンセルし、非タッチ状態に戻します。
     */
    virtual void CancelTouchInput() NN_NOEXCEPT {}

    /**
     * @brief   画面への描画を行います。継承先で定義されます。
     */
    virtual void Render() NN_NOEXCEPT = 0;

    /**
     * @brief   指定したオフセットを適用して画面描画を行います。
     */
    void RenderWithOffset(const Point2D& offset) NN_NOEXCEPT;

protected:
    virtual void ReservePostUpdateEvent(const GuiEventHandlerType& handler) NN_NOEXCEPT;

    virtual void UpdateDrag() NN_NOEXCEPT NN_OVERRIDE;

    virtual void InvokeDragEventHandler(GuiEventHandlerType* pHandler) NN_NOEXCEPT NN_OVERRIDE;

    virtual void BeginDrag(const Point2D& point) NN_NOEXCEPT NN_OVERRIDE;

    virtual void MoveDrag(const Point2D& point) NN_NOEXCEPT NN_OVERRIDE;

    virtual void EndDrag() NN_NOEXCEPT NN_OVERRIDE;

protected:
    mutable nn::os::Mutex m_Mutex;

private:
    UiContainer*        m_pOwner;                   //!< このオブジェクトを所有しているコンテナ
    Point2D             m_Position;                 //!< 表示位置
    Size                m_Size;                     //!< 表示サイズ
    int                 m_OrderZ;                   //!< Z オーダー
    Point2D             m_DragOffset;               //!< ドラッグ開始位置とオブジェクト位置の差分
    bool                m_IsVisible;                //!< 可視状態
    bool                m_IsDecideEffectEnabled;    //!< 決定エフェクト有効フラグ
    uint8_t             m_Opacity;                  //!< 不透明度
    GuiEventHandlerType m_PostUpdateGuiEvent;       //!< Update の最後に呼ばれる GUI イベント
};

}}}  // nns::sgx::gui
