﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiKeyHelp.h"
#include "SimpleGfx_GuiUtil.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

namespace {

const Point2D HelpOffsetFromBottomLeft = { { 32.0f, 72.0f } };

const float HelpItemOffsetFromBottom = 46.0f;
const float HelpKeyOffsetY           = -2.0f;

const float HelpKeySize  = 26.0f;
const float HelpTextSize = 21.2f;

const float HelpKeyGap   = 12.0f;
const float HelpItemGap  = 36.0f;

// ボタンに対応する外字を取得
uint32_t GetButtonChar(nn::hid::NpadButtonSet button) NN_NOEXCEPT
{
    NN_ASSERT_EQUAL(button.CountPopulation(), 1);

    if (button.Test<nn::hid::NpadButton::A>())
    {
        return ExtensionCharacter_ButtonAInvert;
    }
    else if (button.Test<nn::hid::NpadButton::B>())
    {
        return ExtensionCharacter_ButtonBInvert;
    }
    else if (button.Test<nn::hid::NpadButton::X>())
    {
        return ExtensionCharacter_ButtonXInvert;
    }
    else if (button.Test<nn::hid::NpadButton::Y>())
    {
        return ExtensionCharacter_ButtonYInvert;
    }
    else if (button.Test<nn::hid::NpadButton::StickL>())
    {
        return ExtensionCharacter_StickPushLInvert;
    }
    else if (button.Test<nn::hid::NpadButton::StickR>())
    {
        return ExtensionCharacter_StickPushRInvert;
    }
    else if (button.Test<nn::hid::NpadButton::L>())
    {
        return ExtensionCharacter_ButtonLInvert;
    }
    else if (button.Test<nn::hid::NpadButton::R>())
    {
        return ExtensionCharacter_ButtonRInvert;
    }
    else if (button.Test<nn::hid::NpadButton::ZL>())
    {
        return ExtensionCharacter_ButtonZlInvert;
    }
    else if (button.Test<nn::hid::NpadButton::ZR>())
    {
        return ExtensionCharacter_ButtonZrInvert;
    }
    else if (button.Test<nn::hid::NpadButton::Plus>())
    {
        return ExtensionCharacter_ButtonPlusInvert;
    }
    else if (button.Test<nn::hid::NpadButton::Minus>())
    {
        return ExtensionCharacter_ButtonMinusInvert;
    }
    else if (button.Test<nn::hid::NpadButton::Left>())
    {
        return ExtensionCharacter_ButtonLeft;
    }
    else if (button.Test<nn::hid::NpadButton::Up>())
    {
        return ExtensionCharacter_ButtonUp;
    }
    else if (button.Test<nn::hid::NpadButton::Right>())
    {
        return ExtensionCharacter_ButtonRight;
    }
    else if (button.Test<nn::hid::NpadButton::Down>())
    {
        return ExtensionCharacter_ButtonDown;
    }
    else
    {
        return ExtensionCharacter_QuestionCircleInvert;
    }
}

// ボタンヘルプを取得
void GetButtonHelpText(uint32_t* pOutText, int textLengthMax, nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutText);

    int textIndex      = 0;
    int buttonCount    = 0;
    int maxButtonCount = std::min(buttons.CountPopulation(), textLengthMax / 2);

    for (int i = 0;
        i < buttons.GetCount() && buttonCount < maxButtonCount;
        i++)
    {
        if (!buttons.Test(i))
        {
            continue;
        }

        if (textIndex > 0)
        {
            // 2 ボタン目以降は '/' で区切る
            char slash[4] = "/";
            nn::util::ConvertCharacterUtf8ToUtf32(&pOutText[textIndex], slash);
            textIndex++;
        }

        auto flag = nn::hid::NpadButtonSet().Set(i);
        pOutText[textIndex] = GetButtonChar(flag);
        textIndex++;
        buttonCount++;
    }

    // 終端文字
    pOutText[textIndex] = 0u;
}

}  // anonymous

void KeyHelp::AddHelp(nn::hid::NpadButtonSet buttons, const char* description) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(description);

    NNS_SGX_GUI_SCOPED_LOCK;

    HelpItem help = {};
    help.buttons = buttons;
    GetButtonHelpText(help.key, NN_ARRAY_SIZE(HelpItem::key), buttons);
    nn::util::ConvertStringUtf8ToUtf32(
        help.description,
        NN_ARRAY_SIZE(HelpItem::description),
        description);

    m_HelpList.push_back(help);
    m_NeedsUpdateTouchArea = true;
}

void KeyHelp::AddHelp(const char* key, const char* description, GuiEventHandler handler, uintptr_t userArg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(key);
    NN_ASSERT_NOT_NULL(description);

    NNS_SGX_GUI_SCOPED_LOCK;

    HelpItem help = {};
    nn::util::ConvertStringUtf8ToUtf32(
        help.key,
        NN_ARRAY_SIZE(HelpItem::key),
        key);
    nn::util::ConvertStringUtf8ToUtf32(
        help.description,
        NN_ARRAY_SIZE(HelpItem::description),
        description);
    help.touchHandler.handler  = handler;
    help.touchHandler.argument = userArg;

    m_HelpList.push_back(help);
    m_NeedsUpdateTouchArea = true;
}

void KeyHelp::Clear() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_NeedsUpdateTouchArea = false;
    m_HelpList.clear();
    m_TouchedIndex = -1;
    m_TouchState   = TouchState::Idle;
}

void KeyHelp::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    DisplayObject::Update();
}

bool KeyHelp::UpdateTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    auto* tp = input::TouchPanel::GetInstance();

    bool isTouched = false;
    for (int i = 0; i < m_HelpList.size(); i++)
    {
        if (m_TouchedIndex >= 0 && m_TouchedIndex != i)
        {
            continue;
        }

        auto& help = m_HelpList[i];
        const auto& area = help.touchArea.ToFloat4();

        if (tp->IsTouched(area))
        {
            isTouched = true;
        }

        if (tp->IsTouchTriggered(area))
        {
            if (m_TouchState == TouchState::Idle)
            {
                // ボタン押下開始
                m_TouchedIndex = i;
                m_TouchState   = TouchState::Press;
                PlaySystemSe(SystemSe::Press);
                break;
            }
        }
        else if (!tp->IsTouched(area) &&
            m_TouchState == TouchState::Press)
        {
            m_TouchedIndex = -1;
            m_TouchState   = TouchState::Idle;

            // 押下状態からリリースされたらボタン押し判定
            if (tp->IsTouchReleased(area))
            {
                help.touchHandler.Invoke(this);
                if (help.buttons.IsAnyOn())
                {
                    input::PerformPressDummyController(help.buttons);
                }
                return true;
            }
        }
    }

    if (!isTouched)
    {
        CancelTouchInput();
    }

    return false;
}

void KeyHelp::CancelTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_TouchedIndex = -1;
    m_TouchState   = TouchState::Idle;
}

void KeyHelp::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    if (m_NeedsUpdateTouchArea)
    {
        UpdateTouchArea();
    }

    ScopedFontContextSaver saver;

    Size canvasSize;
    GetCanvasSize(&canvasSize);

    // ライン
    const auto& offset = HelpOffsetFromBottomLeft;
    DrawLine(
        offset.x,
        canvasSize.height - offset.y,
        canvasSize.width  - offset.x,
        canvasSize.height - offset.y,
        Colors::Shadow(),
        1.0f);

    SetTextColor(Colors::Shadow());
    for (int i = 0; i < m_HelpList.size(); i++)
    {
        const auto& help = m_HelpList[i];
        const auto& area = help.touchArea;

        float dx = area.x + 16;
        float dy = area.y + 16;

        // キー
        SetTextSize(HelpKeySize);
        Size keySize;
        GetTextDrawSize(&keySize, help.key);
        DrawText(dx, dy + HelpKeyOffsetY, help.key);

        // 説明
        SetTextSize(HelpTextSize);
        Size descSize;
        GetTextDrawSize(&descSize, help.description);
        DrawText(dx + keySize.width + HelpKeyGap, dy, help.description);

        // タッチ表示
        if (m_TouchedIndex == i)
        {
            nns::sgx::FillRectangle(area, nns::sgx::Colors::NeonBlue().BlendAlpha(48));
        }
    }
}

void KeyHelp::UpdateTouchArea() NN_NOEXCEPT
{
    ScopedFontContextSaver saver;

    Size canvasSize;
    GetCanvasSize(&canvasSize);

    const auto& offset = HelpOffsetFromBottomLeft;
    float dx = canvasSize.width  - offset.x * 2;
    float dy = canvasSize.height - HelpItemOffsetFromBottom;
    for (auto iter = m_HelpList.rbegin(); iter != m_HelpList.rend(); iter++)
    {
        auto& help = *iter;
        float rightX = dx;

        // 説明文の幅
        SetTextSize(HelpTextSize);
        Size descSize;
        GetTextDrawSize(&descSize, help.description);
        dx -= descSize.width;

        // キー部分の幅
        SetTextSize(HelpKeySize);
        Size keySize;
        GetTextDrawSize(&keySize, help.key);
        dx -= keySize.width + HelpKeyGap;

        // タッチ領域確定
        help.touchArea.Set(dx - 16, dy + HelpKeyOffsetY - 16, rightX - dx + 32, 56);

        dx -= HelpItemGap;
    }

    m_NeedsUpdateTouchArea = false;
}

}}}  // nns::sgx::gui
