﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>

#include "SimpleGfx_GuiDialog.h"
#include "../SimpleGfx.h"
#include "../SimpleGfx_PostEffect.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

namespace {

const int AnimationFrameCount = 8;

const Size DefaultSize = { { 640, 480 } };

const float WholePadding = 16.0f;

const float ButtonPadding = 16.0f;

const float ButtonHeight = 80.0f;

}  // anonymous

Dialog::Dialog() NN_NOEXCEPT
    : m_IsInitialized(false)
    , m_ButtonCount(1)
    , m_Buttons()
    , m_ButtonSkin()
    , m_CancelButtonMask(nn::hid::NpadButton::B::Mask)
    , m_HandlerForCancel()
    , m_HandlerForRenderTextArea()
    , m_DisplayState(DisplayState::Hide)
    , m_AnimDuration(0)
{
    SetZ(DefaultDialogZ);
}

void Dialog::Initialize() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (m_IsInitialized)
    {
        return;
    }

    SetSize(DefaultSize);

    m_Buttons[0].SetFocus();
    m_ButtonSkin.SetDefault();

    for (auto& button : m_Buttons)
    {
        //button.SetDecideEffectEnabled(false);
        AddChild(&button);
    }

    Centering();
    SetVisible(false);

    m_IsInitialized = true;
}

Size Dialog::GetTextAreaSize() const NN_NOEXCEPT
{
    Size size;
    size.width  = GetWidth()  - WholePadding * 2;
    size.height = GetHeight() - WholePadding * 2;
    if (GetButtonCount() > 0)
    {
        size.height -= (WholePadding + ButtonPadding + ButtonHeight);
    }

    return size;
}

Rectangle Dialog::GetTextArea() const NN_NOEXCEPT
{
    Rectangle rect;
    rect.x    = GetX() + WholePadding;
    rect.y    = GetY() + WholePadding;
    rect.size = GetTextAreaSize();

    return rect;
}

void Dialog::Appear() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized);

    NNS_SGX_GUI_SCOPED_LOCK;

    if (m_DisplayState == DisplayState::Showing ||
        m_DisplayState == DisplayState::Show)
    {
        return;
    }

    SetOpacity(0);
    SetVisible(true);
    SetDefaultFocus();

    m_DisplayState = DisplayState::Showing;
    m_AnimDuration = AnimationFrameCount;
}

void Dialog::Disappear() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized);

    NNS_SGX_GUI_SCOPED_LOCK;

    if (m_DisplayState == DisplayState::Hiding ||
        m_DisplayState == DisplayState::Hide)
    {
        return;
    }

    m_DisplayState = DisplayState::Hiding;
    m_AnimDuration = AnimationFrameCount;
}

void Dialog::Update() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized);

    NNS_SGX_GUI_SCOPED_LOCK;

    if (IsFocused())
    {
        // キャンセル判定
        auto controllers = input::GetNpadControllers();
        for (auto iter = controllers.begin(); iter != controllers.end(); iter++)
        {
            auto pController = *iter;
            if (pController->IsAnyTriggered(m_CancelButtonMask))
            {
                nns::sgx::gui::PlaySystemSe(nns::sgx::gui::SystemSe::Cancel);
                m_HandlerForCancel.Invoke(this);
                return;
            }
        }
    }

    UiContainer::Update();

    UpdateAnimation();
}

void Dialog::Render() NN_NOEXCEPT
{
    NN_ASSERT(m_IsInitialized);

    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    auto opacity = GetDisplayOpacity();

    // 背景をぼかす
    if (nns::sgx::IsPostEffectInitialized())
    {
        float blurOffset = opacity * 2.0f / 255.0f;
        nns::sgx::ApplyBlurEffect(blurOffset, blurOffset);
    }

    // 画面全体をグレーアウト
    {
        Rectangle rect;
        rect.x = rect.y = 0;
        nns::sgx::GetCanvasSize(&rect.size);

        int alpha = 192 * opacity / 255;
        FillRectangle(rect, nns::sgx::Colors::Smoke().BlendAlpha(alpha));
    }

    // フェード中の位置ずらし
    float prevX = GetX();
    float prevY = GetY();
    SetPosition(prevX, prevY + 16 * (255 - opacity) / 255);
    NN_UTIL_SCOPE_EXIT
    {
        // ずらした位置を戻す
        SetPosition(prevX, prevY);
    };

    // クライアント領域の描画
    {
        auto rect = GetClientArea();

        // 背景を描画
        // TODO: 色変更機能
        int alpha = 240 * opacity / 255;
        DrawRectangle(rect, nns::sgx::Colors::Shadow().BlendAlpha(alpha / 2), 4.0f);
        FillRectangle(rect, nns::sgx::Colors::WhiteSmoke().BlendAlpha(alpha));
        DrawRectangle(rect, nns::sgx::Colors::White().BlendAlpha(alpha), 2.0f);

        // 子要素を描画
        UiContainer::Render();
    }

    // テキスト領域の描画
    {
        nns::sgx::ScopedFontContextSaver saver;

        auto rect = GetTextArea();
        nns::sgx::ApplyRenderArea(rect);

        RenderTextArea();
        m_HandlerForRenderTextArea.Invoke(this);

        nns::sgx::RestoreRenderArea();
        NNS_SGX_GUI_DRAW_CLIENT_AREA_DEBUG(rect);
    }
}

void Dialog::UpdateLayout() NN_NOEXCEPT
{
    // ボタンの位置とサイズを調整
    float padding     = WholePadding + ButtonPadding;
    float buttonWidth = (GetWidth() - (m_ButtonCount + 1) * padding) / std::max<int>(m_ButtonCount, 1);
    float buttonX     = padding;
    float buttonY     = GetHeight() - ButtonHeight - padding;
    for (int i = 0; i < GetButtonCountMax(); i++)
    {
        auto& button = m_Buttons[i];
        if (i < m_ButtonCount)
        {
            button.Show();
            button.SetPosition(buttonX, buttonY);
            button.SetSize(buttonWidth, ButtonHeight);
            buttonX += buttonWidth + padding;
        }
        else
        {
            button.Hide();
        }
    }

    // フォーカス可否判定を正しく行うため、一時的に表示状態にする
    {
        bool prevVisible = IsVisible();
        SetVisible(true);

        RebuildKeyFocusableChildren();
        SetDefaultFocus();

        SetVisible(prevVisible);
    }

    UpdateCursorPosition();
}

void Dialog::DrawCenteringText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    auto textArea = GetTextAreaSize();
    nns::sgx::Size textSize;
    nns::sgx::GetTextDrawSize(&textSize, text);

    nns::sgx::DrawText(
        std::max(textArea.width  - textSize.width,  0.0f) / 2,
        std::max(textArea.height - textSize.height, 0.0f) / 2,
        text
    );
}

void Dialog::DrawCenteringText(float y, const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    auto textArea = GetTextAreaSize();
    nns::sgx::Size textSize;
    nns::sgx::GetTextDrawSize(&textSize, text);

    nns::sgx::DrawText(
        std::max(textArea.width  - textSize.width,  0.0f) / 2,
        y,
        text
    );
}

void Dialog::SetDefaultFocus() NN_NOEXCEPT
{
    // 必要ならフォーカスを当て直す
    if (GetFocusedChild() == nullptr ||
        !GetFocusedChild()->IsFocused())
    {
        for (auto& button : m_Buttons)
        {
            if (button.IsKeyFocusable())
            {
                SetFocusedChild(&button);
                break;
            }
        }
    }
}

void Dialog::Centering() NN_NOEXCEPT
{
    Size canvasSize;
    nns::sgx::GetCanvasSize(&canvasSize);

    Point2D position;
    position.x = (canvasSize.width  - GetWidth())  / 2.0f;
    position.y = (canvasSize.height - GetHeight()) / 2.0f;
    SetPosition(position);

    UpdateLayout();
}

void Dialog::UpdateAnimation() NN_NOEXCEPT
{
    if (m_AnimDuration <= 0)
    {
        return;
    }

    m_AnimDuration--;

    switch (m_DisplayState)
    {
    case DisplayState::Hiding:
        if (m_AnimDuration <= 0)
        {
            m_DisplayState = DisplayState::Hide;
            SetVisible(false);
        }
        else
        {
            SetOpacity(255 * m_AnimDuration / AnimationFrameCount);
        }
        break;

    case DisplayState::Showing:
        if (m_AnimDuration <= 0)
        {
            m_DisplayState = DisplayState::Show;
            SetOpacity(255);
        }
        else
        {
            SetOpacity(255 * (AnimationFrameCount - m_AnimDuration) / AnimationFrameCount);
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}}  // nns::sgx::gui
