﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "SimpleGfx_GuiCommonDefs.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   ドラッグ操作の開始トリガーです。
 */
enum class DragTrigger
{
    Press,      //!< タッチした直後にドラッグを開始
    LongPress   //!< 長押し後にドラッグを開始
};

/**
 * @brief   オブジェクトにフォーカス設定用の機能を追加するためのクラスです。
 */
class Focusable
{
public:
    Focusable() NN_NOEXCEPT :
        m_IsFocused(false),
        m_IsFocusRejected(false)
    {}

    /**
     * @brief   フォーカス状態を取得します。
     */
    bool IsFocused() const NN_NOEXCEPT
    {
        return m_IsFocused;
    }

    /**
     * @brief   フォーカス拒否状態を取得します。
     */
    bool IsFocusRejected() const NN_NOEXCEPT
    {
        return m_IsFocused;
    }

    /**
     * @brief   フォーカスを設定します。
     */
    virtual void SetFocus() NN_NOEXCEPT
    {
        m_IsFocused = true;
    }

    /**
     * @brief   フォーカスを解除します。
     */
    virtual void UnsetFocus() NN_NOEXCEPT
    {
        m_IsFocused = false;
    }

    /**
     * @brief   フォーカス拒否状態を設定します。
     */
    void SetFocusRejected(bool isRejected) NN_NOEXCEPT
    {
        m_IsFocusRejected = isRejected;
    }

private:
    bool    m_IsFocused;        //!< フォーカス状態
    bool    m_IsFocusRejected;  //!< フォーカス拒否状態
};

/**
 * @brief   オブジェクトに有効/無効状態設定用の機能を追加するためのクラスです。
 */
class Prohibitable
{
public:
    Prohibitable() NN_NOEXCEPT :
        m_IsEnabled(true)
    {}

    /**
     * @brief   有効状態を取得します。
     */
    virtual bool IsEnabled() const NN_NOEXCEPT
    {
        return m_IsEnabled;
    }

    /**
     * @brief   有効状態を設定します。
     */
    virtual void SetEnabled(bool isEnabled) NN_NOEXCEPT
    {
        m_IsEnabled = isEnabled;
    }

    /**
     * @brief   有効状態にします。
     */
    void Enable() NN_NOEXCEPT
    {
        SetEnabled(true);
    }

    /**
     * @brief   無効状態にします。
     */
    void Disable() NN_NOEXCEPT
    {
        SetEnabled(false);
    }

private:
    bool    m_IsEnabled;    //!< 有効状態
};

/**
 * @brief   オブジェクトにドラッグ移動機能を追加するためのクラスです。
 */
class Draggable
{
public:
    Draggable() NN_NOEXCEPT :
#if defined(S2D_GUI_ENABLE_LOCK)
        m_MutexForDraggable(true),
#endif  // if defined(S2D_GUI_ENABLE_LOCK)
        m_IsDraggable(false),
        m_DragState(DragState::Idle),
        m_DragTrigger(DragTrigger::Press),
        m_DragArea(),
        m_DragStartPoint(),
        m_LastDragPoint(),
        m_HandlerForDragEvent(),
        m_HandlerForDragMoveEvent(),
        m_HandlerForDropEvent()
    {}

public:
    /**
     * @brief   ドラッグ可否状態を取得します。
     */
    bool IsDraggable() const NN_NOEXCEPT
    {
        return m_IsDraggable;
    }

    /**
     * @brief   ドラッグ可否状態を設定します。
     */
    void SetDraggable(bool isDraggable) NN_NOEXCEPT
    {
        m_IsDraggable = isDraggable;
    }

    /**
     * @brief   ドラッグ開始イベントを通知するハンドラを登録します。
     */
    void RegisterDragEventHandler(const GuiEventHandlerType& handler) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        m_HandlerForDragEvent = handler;
    }

    /**
     * @brief   ドラッグ中の移動イベントを通知するハンドラを登録します。
     */
    void RegisterDragMoveEventHandler(const GuiEventHandlerType& handler) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        m_HandlerForDragMoveEvent = handler;
    }

    /**
     * @brief   ドラッグ終了イベントを通知するハンドラを登録します。
     */
    void RegisterDropEventHandler(const GuiEventHandlerType& handler) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        m_HandlerForDropEvent = handler;
    }

protected:
    /**
     * @brief   ドラッグ範囲の取得
     */
    void GetDragArea(Rectangle* pOutArea) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        *pOutArea = m_DragArea;
    }

    /**
     * @brief   ドラッグ範囲の設定
     */
    void SetDragArea(const Rectangle& area) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        m_DragArea = area;
    }

    /**
     * @brief   ドラッグ開始位置の取得
     */
    void GetDragStartPoint(Point2D* pOutPoint) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutPoint);
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        *pOutPoint = m_DragStartPoint;
    }

    /**
     * @brief   ドラッグ位置が移動したか判定
     */
    bool IsDragMoved(const Point2D& point) const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        if (std::fabs(m_LastDragPoint.x - point.x) >= 1.0f &&
            std::fabs(m_LastDragPoint.y - point.y) >= 1.0f)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief   ドラッグ処理の更新
     */
    virtual void UpdateDrag() NN_NOEXCEPT = 0;

    /**
     * @brief   イベントハンドラの呼び出し
     */
    virtual void InvokeDragEventHandler(GuiEventHandlerType* pHandler) NN_NOEXCEPT = 0;

    /**
     * @brief   長押しドラッグか
     */
    bool NeedsLongPressDrag() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        return m_DragTrigger == DragTrigger::LongPress;
    }

    /**
     * @brief   ドラッグ中か
     */
    bool IsDragging() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK_P(m_MutexForDraggable);

        return m_DragState == DragState::Dragging;
    }

    /**
     * @brief   ドラッグの開始
     */
    virtual void BeginDrag(const Point2D& point) NN_NOEXCEPT
    {
        m_DragState      = DragState::Dragging;
        m_DragStartPoint = point;
        m_LastDragPoint  = point;
        InvokeDragEventHandler(&m_HandlerForDragEvent);
    }

    /**
     * @brief   ドラッグ中の移動
     */
    virtual void MoveDrag(const Point2D& point) NN_NOEXCEPT
    {
        m_LastDragPoint = point;
        InvokeDragEventHandler(&m_HandlerForDragMoveEvent);
    }

    /**
     * @brief   ドラッグの終了
     */
    virtual void EndDrag() NN_NOEXCEPT
    {
        m_DragState = DragState::Idle;
        InvokeDragEventHandler(&m_HandlerForDropEvent);
    }

private:
    /**
     * @brief   ドラッグ状態
     */
    enum class DragState
    {
        Idle,       //!< ドラッグしていない
        Dragging    //!< ドラッグ中
    };

private:
#if defined(S2D_GUI_ENABLE_LOCK)
    mutable ::nn::os::Mutex m_MutexForDraggable;
#endif  // if defined(S2D_GUI_ENABLE_LOCK)

    bool                m_IsDraggable;              //!< ドラッグ可否状態
    DragState           m_DragState;                //!< ドラッグ状態
    DragTrigger         m_DragTrigger;              //!< ドラッグ開始トリガ
    Rectangle           m_DragArea;                 //!< ドラッグ可能領域
    Point2D             m_DragStartPoint;           //!< ドラッグ開始位置
    Point2D             m_LastDragPoint;            //!< 前回のドラッグ位置

    GuiEventHandlerType m_HandlerForDragEvent;      //!< ドラッグ開始イベントハンドラ
    GuiEventHandlerType m_HandlerForDragMoveEvent;  //!< ドラッグ中の移動イベントハンドラ
    GuiEventHandlerType m_HandlerForDropEvent;      //!< ドラッグ終了イベントハンドラ
};

}}}  // nns::sgx::gui
