﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_Fs.h"

namespace
{

// 拡張ヒープ領域のサイズ
const size_t HeapMemorySize = 1024 * 1024;

// 拡張ヒープ領域
char g_HeapMemory[HeapMemorySize];

::nn::lmem::HeapHandle g_HeapHandle;
char* g_MountCache = nullptr;

bool g_IsInitialized = false;

void* AllocateForFs(size_t size) NN_NOEXCEPT
{
    return ::nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
};

void DeallocateForFs(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    ::nn::lmem::FreeToExpHeap(g_HeapHandle, p);
};

}  // anonymous

namespace nns { namespace sgx { namespace detail {

void InitializeFileSystem() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!g_IsInitialized);

    // FS 用の拡張ヒープを作成
    g_HeapHandle = ::nn::lmem::CreateExpHeap(
        g_HeapMemory,
        sizeof(g_HeapMemory),
        ::nn::lmem::CreationOption_NoOption);

    // アロケータを登録
    ::nn::fs::SetAllocator(AllocateForFs, DeallocateForFs);

    // RomFs をマウント
    size_t cacheSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::fs::QueryMountRomCacheSize(&cacheSize)
    );

    g_MountCache = reinterpret_cast<char*>(
        ::nn::lmem::AllocateFromExpHeap(g_HeapHandle, cacheSize));
    NN_ABORT_UNLESS_NOT_NULL(g_MountCache);

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::fs::MountRom(RomMountName, g_MountCache, cacheSize)
    );

    g_IsInitialized = true;
}

void FinalizeFileSystem() NN_NOEXCEPT
{
    if (!g_IsInitialized)
    {
        return;
    }

    g_IsInitialized = false;

    ::nn::fs::Unmount(nns::sgx::RomMountName);
    ::nn::lmem::FreeToExpHeap(g_HeapHandle, g_MountCache);
    ::nn::lmem::DestroyExpHeap(g_HeapHandle);
}

}}}  // nns::sgx::detail
