﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "SimpleGfx_Types.h"

namespace nns { namespace sgx {

// 描画に使用するシェーダのタイプ
enum class ShaderType
{
    None,       //!< シェーダ不使用 (割り当て解除)
    Simple,     //!< 頂点カラーを使用した単純な描画
    Texture     //!< テクスチャ描画
};

/**
 * @brief   OpenGL 用の 0.0 ～ 1.0 に正規化された色
 */
union NormalizedColor
{
    struct
    {
        GLfloat r;
        GLfloat g;
        GLfloat b;
        GLfloat a;
    };
    GLfloat v[4];

    /**
     * @brief   8 bit カラーを正規化して格納
     */
    void SetU8Color(const Color& color) NN_NOEXCEPT
    {
        r = ConvertFromU8(color.GetR());
        g = ConvertFromU8(color.GetG());
        b = ConvertFromU8(color.GetB());
        a = ConvertFromU8(color.GetA());
    }

    /**
     * @brief   8 bit カラーから正規化カラーを生成
     */
    static NormalizedColor FromU8Color(const Color& color) NN_NOEXCEPT
    {
        NormalizedColor newColor;
        newColor.SetU8Color(color);

        return newColor;
    }

private:
    static GLfloat ConvertFromU8(uint8_t u8color) NN_NOEXCEPT
    {
        return u8color / 255.0f;
    }
};

/**
 * @brief   コンパイル済みシェーダ ID の組
 */
struct ShaderPack
{
    GLuint vertexId;
    GLuint fragmentId;
    GLuint programId;
};

}}  // nns::sgx
