﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <algorithm>
#include <cmath>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_Constant.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>

#include "SimpleGfx.h"
#include "SimpleGfx_TypesInternal.h"
#include "SimpleGfx_PostEffect.h"
#include "detail/GraphicsHelper.h"
#include "detail/SimpleGfx_ApiInternal.h"
#include "detail/SimpleGfx_Font.h"
#include "detail/SimpleGfx_Fs.h"
#include "detail/SimpleGfx_Util.h"

// API 呼び出しを排他する
//#define NNS_SGX_ENABLE_LOCK

#if defined(NNS_SGX_ENABLE_LOCK)
#define SCOPED_LOCK     std::lock_guard<decltype(g_Mutex)> lock(g_Mutex)
#else
#define SCOPED_LOCK
#endif

// 拡張子の一致判定
// extension は文字列リテラルである必要がある
#define NNS_SGX_STRING_LITERAL_EQUAL(target, extension) \
    (nn::util::Strnlen(target, sizeof(extension)) == sizeof(extension) - 1 && \
     nn::util::Strnicmp(target, extension, sizeof(extension)) == 0)

namespace nns { namespace sgx {

namespace
{

// 頂点バッファの要素数のデフォルト値
const int DefaultVertexCountMax   = 32 * 1024;

// 多角形描画用の一時領域の頂点数
const int TemporaryVertexCountMax = 1024;

// 文字列描画用の一時領域サイズ
const int TemporaryStringBufferSize = 1024;

// 単純図形描画用頂点シェーダ
const char* VertexShaderSimple = R"(
#version 320 es
layout(location = 0) in highp vec3 a_Position;
layout(location = 1) in lowp  vec4 a_Color;
out lowp vec4 v_Color;

void main() {
    v_Color     = a_Color;
    gl_Position = vec4(a_Position, 1.0);
}
)";

// テクスチャ描画用頂点シェーダ
const char* VertexShaderTexture = R"(
#version 320 es
layout(location = 0) in highp   vec3 a_Position;
layout(location = 1) in lowp    vec4 a_Color;
layout(location = 2) in mediump vec2 a_TexCoord;
out mediump vec2 v_TexCoord;
out lowp    vec4 v_Color;

void main() {
    gl_Position = vec4(a_Position, 1.0);
    v_TexCoord  = a_TexCoord;
    v_Color     = a_Color;
}
)";

// 単純図形描画用フラグメントシェーダ
const char* FragmentShaderSimple = R"(
#version 320 es
layout(location = 0) out lowp vec4 o_FragColor;
in lowp vec4 v_Color;

void main() {
    o_FragColor = v_Color;
}
)";

// テクスチャ描画用フラグメントシェーダ
const char* FragmentShaderTexture = R"(
#version 320 es
layout(location = 0) out lowp vec4 o_FragColor;
in highp  vec2 v_TexCoord;
in lowp   vec4 v_Color;
uniform sampler2D s_Texture;

void main() {
    lowp vec4 tempColor = texture(s_Texture, v_TexCoord);
    o_FragColor = tempColor * v_Color;
}
)";

/**
 * @brief   シェーダの attribute の位置
 */
struct ShaderLocations
{
    GLuint position;
    GLuint vertexColor;
    GLuint textureCoord;
};

ShaderLocations g_ShaderLocations;

bool g_IsInitialized = false;
nn::os::Mutex g_Mutex(true);

// 描画設定
OriginPosition  g_OriginPosition = OriginPosition::TopLeft;
float           g_ImageOpacity   = 1.0f;
float           g_ImageRotation  = 0.0f;
FontContext     g_FontContext;

GLuint     g_ImageFrameBuffer;
ImageData* g_pRenderTargetImage = nullptr;

detail::GraphicsHelper g_GraphicsHelper;

ShaderPack g_SimpleShader;
ShaderPack g_TextureShader;

GLuint g_GlBuffers[3];
GLuint g_VertexBufferId;
GLuint g_VertexColorBufferId;
GLuint g_TexCoordBufferId;
GLuint g_VaoId;

int             g_VertexCountMax        = DefaultVertexCountMax;
int             g_NextVertexIndex       = 0;
char*           g_pVertexPointBuffer    = nullptr;
char*           g_pVertexColorBuffer    = nullptr;
char*           g_pVertexTexCoordBuffer = nullptr;
Point3D         g_TempPointBuffer[TemporaryVertexCountMax];
NormalizedColor g_TempColorBuffer[TemporaryVertexCountMax];
Color           g_TempColorBufferU8[TemporaryVertexCountMax];
Point2D         g_TempTexCoordBuffer[TemporaryVertexCountMax];

#if defined(NNS_SGX_ENABLE_DEBUG_LOG)
void DECL_SPEC DebugCallbackFunc(
    GLenum source,
    GLenum type,
    GLuint id,
    GLenum severity,
    GLsizei length,
    const GLchar* message,
    const void* userParam) NN_NOEXCEPT
{
    NN_UNUSED(length);
    NN_UNUSED(userParam);

    if (type != GL_DEBUG_TYPE_ERROR && type != GL_DEBUG_TYPE_PERFORMANCE)
    {
        return;
    }

    NN_LOG("GL Debug Callback:\n");
    NN_LOG("  source:       0x%08x\n", source);
    NN_LOG("  type:         0x%08x\n", type);
    NN_LOG("  id:           0x%08x\n", id);
    NN_LOG("  severity:     0x%08x\n", severity);
    NN_LOG("  message:      %s\n", message);

    NN_ASSERT(type != GL_DEBUG_TYPE_ERROR, "GL Error occurs.");
}
#endif  // if defined(NNS_SGX_ENABLE_DEBUG_LOG)

/**
 * @brief   グラフィックスの初期化
 */
void InitializeGraphics() NN_NOEXCEPT
{
    g_GraphicsHelper.Initialize();

#if defined(NNS_SGX_ENABLE_DEBUG_LOG)
    NN_LOG("GL_VERSION: %s\n", glGetString(GL_VERSION));
    NN_LOG("GL_SHADING_LANGUAGE_VERSION: %s\n", glGetString(GL_SHADING_LANGUAGE_VERSION));

    glDebugMessageCallback(DebugCallbackFunc, nullptr);
#endif  // if defined(NNS_SGX_ENABLE_DEBUG_LOG)

    // アルファブレンドの有効化
    glEnable(GL_BLEND);

    // ブレンドモードの初期化
    SetBlendMode(BlendMode::Normal);
}

/**
 * @brief   シェーダ―の初期化
 */
void InitializeShaders() NN_NOEXCEPT
{
    detail::CreateShader(&g_SimpleShader, &VertexShaderSimple, &FragmentShaderSimple);
    g_ShaderLocations.position    = detail::GetAttribLocationWithAssert(g_SimpleShader, "a_Position");
    g_ShaderLocations.vertexColor = detail::GetAttribLocationWithAssert(g_SimpleShader, "a_Color");

    detail::CreateShader(&g_TextureShader, &VertexShaderTexture, &FragmentShaderTexture);
    g_ShaderLocations.textureCoord = detail::GetAttribLocationWithAssert(g_TextureShader, "a_TexCoord");

    detail::RegisterShader(&g_SimpleShader, &g_TextureShader);
}

/**
 * @brief   シェーダの attributes を更新
 */
void UpdateShaderAttributes() NN_NOEXCEPT
{
    glBindVertexArray(g_VaoId);

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(Point3D) * g_VertexCountMax, g_pVertexPointBuffer, GL_STATIC_DRAW);
    glEnableVertexAttribArray(g_ShaderLocations.position);
    glVertexAttribPointer(g_ShaderLocations.position, NN_ARRAY_SIZE(Point3D::v), GL_FLOAT, GL_FALSE, 0, 0);

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(NormalizedColor) * g_VertexCountMax, g_pVertexColorBuffer, GL_STATIC_DRAW);
    glEnableVertexAttribArray(g_ShaderLocations.vertexColor);
    glVertexAttribPointer(g_ShaderLocations.vertexColor, NN_ARRAY_SIZE(NormalizedColor::v), GL_FLOAT, GL_FALSE, 0, 0);

    glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
    glBufferData(GL_ARRAY_BUFFER, sizeof(Point2D) * g_VertexCountMax, g_pVertexTexCoordBuffer, GL_STATIC_DRAW);
    glEnableVertexAttribArray(g_ShaderLocations.textureCoord);
    glVertexAttribPointer(g_ShaderLocations.textureCoord, NN_ARRAY_SIZE(Point2D::v), GL_FLOAT, GL_FALSE, 0, 0);
}

/**
 * @brief   頂点バッファを作成
 */
void InitializeVertexData() NN_NOEXCEPT
{
    glGenVertexArrays(1, &g_VaoId);
    glGenBuffers(NN_ARRAY_SIZE(g_GlBuffers), g_GlBuffers);

    g_VertexBufferId      = g_GlBuffers[0];
    g_VertexColorBufferId = g_GlBuffers[1];
    g_TexCoordBufferId    = g_GlBuffers[2];

    UpdateShaderAttributes();
}

/**
 * @brief   グラフィックス用オブジェクトの破棄
 */
void Destroy() NN_NOEXCEPT
{
    detail::DeleteShader(g_SimpleShader);
    detail::DeleteShader(g_TextureShader);

    glDeleteBuffers(NN_ARRAY_SIZE(g_GlBuffers), g_GlBuffers);
    glDeleteVertexArrays(1, &g_VaoId);
}

/**
 * @brief   グラフィックスの終了処理
 */
void FinalizeGraphics() NN_NOEXCEPT
{
    g_GraphicsHelper.Finalize();
}

/**
 * @brief   使用するシェーダの切り替え
 */
void ChangeShader(ShaderType type) NN_NOEXCEPT
{
    if (detail::ChangeShader(type))
    {
        UpdateShaderAttributes();
    }
}

/**
 * @brief   正規化した X 座標を取得
 */
NN_FORCEINLINE
GLfloat NormalizeX(float x) NN_NOEXCEPT
{
    auto size = detail::GetCanvasSize();
    GLfloat newX = static_cast<GLfloat>(x * 2.0f / size.width);
    if (g_OriginPosition == OriginPosition::TopLeft)
    {
        newX -= 1.0f;
    }

    return newX;
}

/**
 * @brief   正規化した Y 座標を取得
 */
NN_FORCEINLINE
GLfloat NormalizeY(float y) NN_NOEXCEPT
{
    auto size = detail::GetCanvasSize();
    GLfloat newY = static_cast<GLfloat>(y * 2.0f / size.height);
    if (g_OriginPosition == OriginPosition::TopLeft)
    {
        newY = 1.0f - newY;
    }

    return newY;
}

/**
 * @brief   2D 座標を正規化した 3D 座標に変換
 */
NN_FORCEINLINE
void Normalize(Point3D* pOutPoint, const Point2D& srcPoint) NN_NOEXCEPT
{
    pOutPoint->x = NormalizeX(srcPoint.x);
    pOutPoint->y = NormalizeY(srcPoint.y);
    pOutPoint->z = 0;
}

/**
 * @brief   描画対象テクスチャを変更済み状態にする
 */
NN_FORCEINLINE
void TouchRenderTarget() NN_NOEXCEPT
{
    if (g_pRenderTargetImage != nullptr)
    {
        g_pRenderTargetImage->isDirty = true;
    }
}

/**
 * @brief   円弧描画の共通処理
 */
void DrawArcCommon(
    float centerX,
    float centerY,
    float radius,
    float beginAngle,
    float endAngle,
    const Color& color,
    float lineWidth,
    bool isPie,
    GLenum primitiveType) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    if (radius < lineWidth * 0.1f)
    {
        // 小さすぎる半径は点で処理
        DrawPoint(centerX, centerY, color, lineWidth);
        return;
    }

    // 始点より終点の方が先であれば反転
    if (endAngle < beginAngle)
    {
        float tempAngle = beginAngle;
        beginAngle = endAngle;
        endAngle   = tempAngle;
    }

    float angle = std::fabsf(endAngle - beginAngle);

    int indexOffset = isPie ? 1 : 0;
    const int arcVertexCount = std::max(
        static_cast<int>(std::ceilf(32 * angle / nn::util::FloatPi)),
        2);
    const int vertexCount = arcVertexCount + indexOffset;

    float step = angle / (arcVertexCount - 1);
    float vx = radius * std::cosf(beginAngle);
    float vy = radius * std::sinf(beginAngle);

    if (isPie)
    {
        g_TempPointBuffer[0].x = NormalizeX(centerX);
        g_TempPointBuffer[0].y = NormalizeY(centerY);
    }

    for (int i = arcVertexCount - 1; i >= 0; i--)
    {
        g_TempPointBuffer[i + indexOffset].x = NormalizeX(centerX + vx);
        g_TempPointBuffer[i + indexOffset].y = NormalizeY(centerY + vy);

        // 角速度方向に進める
        float force = vx;
        vx -= vy * step;
        vy += force * step;

        // 半径修正
        force = radius / std::sqrtf(vx * vx + vy * vy);
        vx *= force;
        vy *= force;
    }

    // 色は統一
    for (int i = 0; i < vertexCount; i++)
    {
        g_TempColorBuffer[i].SetU8Color(color);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        vertexCount * sizeof(Point3D),
        g_TempPointBuffer
    );

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        vertexCount * sizeof(NormalizedColor),
        g_TempColorBuffer
    );

    glLineWidth(lineWidth);
    glDrawArrays(primitiveType, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

/**
 * @brief   円描画の共通処理
 */
void DrawCircleCommon(
    float centerX,
    float centerY,
    float radius,
    const Color& color,
    float lineWidth,
    GLenum primitiveType) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    if (radius < lineWidth * 0.1f)
    {
        // 小さすぎる半径は点で処理
        DrawPoint(centerX, centerY, color, lineWidth);
        return;
    }

    const int vertexCount = 56;

    float step = 2 * nn::util::FloatPi / vertexCount;
    float vx = radius;
    float vy = 0.0f;

    for (int i = vertexCount - 1; i >= 0; i--)
    {
        g_TempPointBuffer[i].x = NormalizeX(centerX + vx);
        g_TempPointBuffer[i].y = NormalizeY(centerY + vy);

        // 角速度方向に進める
        float force = vx;
        vx -= vy * step;
        vy += force * step;

        // 半径修正
        force = radius / std::sqrtf(vx * vx + vy * vy);
        vx *= force;
        vy *= force;
    }

    // 色は統一
    for (int i = 0; i < vertexCount; i++)
    {
        g_TempColorBuffer[i].SetU8Color(color);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        vertexCount * sizeof(Point3D),
        g_TempPointBuffer
    );

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        vertexCount * sizeof(NormalizedColor),
        g_TempColorBuffer
    );

    glLineWidth(lineWidth);
    glDrawArrays(primitiveType, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

/**
 * @brief   四角形に対して回転を適用する
 *
 * @detail  points は OpenGL 向けに正規化された座標
 */
void ApplyRotationForQuad(Point3D* points) NN_NOEXCEPT
{
    if (g_ImageRotation == 0.0f)
    {
        return;
    }

    NN_ASSERT_NOT_NULL(points);

    auto size = detail::GetCanvasSize();

    // 左上原点かつ正方形になるような座標系に無理矢理変換
    const float rateX = size.width  / size.height;
    const float rateY = size.height / size.width;
    Point3D tempPoints[4];
    for (int i = 0; i < NN_ARRAY_SIZE(tempPoints); i++)
    {
        tempPoints[i] = points[i];
        tempPoints[i].x += 1.0f;
        tempPoints[i].y  = (2.0f - (tempPoints[i].y + 1.0f)) * rateY;
    }

    // 矩形と見なして雑に中心を計算
    float centerX = std::fabsf(tempPoints[2].x - tempPoints[0].x) / 2.0f + tempPoints[0].x;
    float centerY = std::fabsf(tempPoints[2].y - tempPoints[0].y) / 2.0f + tempPoints[0].y;

    const float angle = g_ImageRotation;

    // 回転しつつ元の座標系に戻す関数
    const auto rotate = [&rateX, &centerX, &centerY, &angle](const Point3D& point) NN_NOEXCEPT
    {
        float tx = point.x - centerX;
        float ty = point.y - centerY;
        float x = tx * std::cosf(angle) - ty * std::sinf(angle);
        float y = tx * std::sinf(angle) + ty * std::cosf(angle);
        return Point3D
        {{
            centerX + x - 1.0f,
            1.0f - (centerY + y) * rateX,
            point.z
        }};
    };

#if 0
    NN_LOG("C: %.3f, %.3f\n", centerX, centerY);
    for (int i = 0; i < 4; i++)
    {
        auto r = rotate(tempPoints[i]);
        char log[200];
        auto len = nn::util::SNPrintf(
            log, sizeof(log),
            "[%d] %.3f, %.3f -> %.3f, %.3f ", i,
            tempPoints[i].x, tempPoints[i].y,
            r.x, r.y);
        NN_PUT(log, len);
        NN_LOG("\n");
    }
#endif

    // 各頂点を回転
    points[0] = rotate(tempPoints[0]);
    points[1] = rotate(tempPoints[1]);
    points[2] = rotate(tempPoints[2]);
    points[3] = rotate(tempPoints[3]);
}

/**
 * @brief   ピクセルフォーマットに応じた 1 ピクセルに必要なバイト数を取得
 */
int GetPixelBytes(PixelFormat format) NN_NOEXCEPT
{
    switch (format)
    {
    case PixelFormat::R8G8B8A8: return 4;
    case PixelFormat::U8:       return 1;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief   ピクセルフォーマットを OpenGL 用の値に変換
 */
GLint ConvertPixelFormatForGl(PixelFormat format) NN_NOEXCEPT
{
    switch (format)
    {
    case PixelFormat::R8G8B8A8: return GL_RGBA;
    case PixelFormat::U8:       return GL_RED;  // Red を swizzle で輝度に変換する
    default: NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief   テクスチャ生成の共通処理
 */
ResultCode CreateTextureCommon(
    ImageData* pOutImage,
    void* pImageOnMemory) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);
    NN_ASSERT_NOT_NULL(pImageOnMemory);

    SCOPED_LOCK;

    // テクスチャを生成
    glGenTextures(1, &(pOutImage->textureId));
    // TODO: エラー処理

    // テクスチャを割り当てる
    auto format = ConvertPixelFormatForGl(pOutImage->pixelFormat);
    glBindTexture(GL_TEXTURE_2D, pOutImage->textureId);
    glTexImage2D(
        GL_TEXTURE_2D,
        0,
        format,
        static_cast<GLsizei>(pOutImage->size.width),
        static_cast<GLsizei>(pOutImage->size.height),
        0,
        format,
        GL_UNSIGNED_BYTE,
        pImageOnMemory
    );

    if (format == GL_RED)
    {
        // 1 チャンネルの場合は輝度値として扱う
        GLint swizzleMask[] = { GL_ONE, GL_ONE, GL_ONE, GL_RED };
        glTexParameteriv(GL_TEXTURE_2D, GL_TEXTURE_SWIZZLE_RGBA, swizzleMask);
    }

    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

    pOutImage->pPixelData = reinterpret_cast<char*>(pImageOnMemory);
    pOutImage->isDirty    = false;

    return ResultCode::Success;
}

/**
 * @brief   単一の UTF-32 文字の処理
 */
void ProcessUtf32Char(float baseX, float baseY, int printedCount, uint32_t utf32) NN_NOEXCEPT
{
    if (!detail::IsValidSharedFontCharacter(utf32))
    {
        return;
    }

    const int vertexCount = 6;

    Size charSize;
    detail::GetSharedFontCharacterSize(&charSize, utf32);

    charSize.width  *= g_FontContext.scaleX;
    charSize.height *= g_FontContext.scaleY;

    // 座標は GL_TRIANGLES を想定して格納
    // (左上 -> 右上 -> 右下, 左上 -> 右下 -> 左下)

    float x = baseX + g_FontContext.offsetX;
    auto* pVertex = g_TempPointBuffer + printedCount * vertexCount;
    pVertex[0] = Point3D{ { NormalizeX(x), NormalizeY(baseY), 0 } };
    pVertex[1] = Point3D{ { NormalizeX(x + charSize.width), NormalizeY(baseY), 0 } };
    pVertex[2] = Point3D{ { NormalizeX(x + charSize.width), NormalizeY(baseY + charSize.height), 0 } };
    pVertex[3] = pVertex[0];
    pVertex[4] = pVertex[2];
    pVertex[5] = Point3D{ { NormalizeX(x), NormalizeY(baseY + charSize.height), 0 } };

    auto* pColor = g_TempColorBuffer + printedCount * vertexCount;
    pColor[5] = pColor[4] = pColor[3] = pColor[2] = pColor[1] = pColor[0] =
        NormalizedColor::FromU8Color(g_FontContext.color);

    Point2D texCoords[vertexCount];
    detail::GetSharedFontTextureCoord(texCoords, utf32);

    auto* pTexCoord = g_TempTexCoordBuffer + printedCount * vertexCount;
    pTexCoord[0] = texCoords[0];
    pTexCoord[1] = texCoords[1];
    pTexCoord[2] = texCoords[2];
    pTexCoord[3] = texCoords[0];
    pTexCoord[4] = texCoords[2];
    pTexCoord[5] = texCoords[3];

    g_FontContext.offsetX += charSize.width;
}

/**
 * @brief   UTF-32 文字列の描画
 */
void DrawUtf32Text(float x, float y, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    g_FontContext.offsetX = 0;

    auto pString = text;
    bool isNewLine = false;
    int count = 0;

    while (*pString != '\0')
    {
        if (*pString == '\n')
        {
            isNewLine = true;
            pString++;
            break;
        }
        else
        {
            ProcessUtf32Char(x, y, count, *pString);
            count++;
            pString++;
        }
    }

    if (count > 0)
    {
        ChangeShader(ShaderType::Texture);

        auto vertexCount = count * 6;

        glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
        glBufferSubData(
            GL_ARRAY_BUFFER,
            g_NextVertexIndex * sizeof(Point3D),
            vertexCount * sizeof(Point3D),
            g_TempPointBuffer
        );

        glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
        glBufferSubData(
            GL_ARRAY_BUFFER,
            g_NextVertexIndex * sizeof(NormalizedColor),
            vertexCount * sizeof(NormalizedColor),
            g_TempColorBuffer
        );

        glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
        glBufferSubData(
            GL_ARRAY_BUFFER,
            g_NextVertexIndex * sizeof(Point2D),
            vertexCount * sizeof(Point2D),
            g_TempTexCoordBuffer
        );

        auto textureId = detail::GetSharedFontTextureId();
        glActiveTexture(GL_TEXTURE0);
        glBindTexture(GL_TEXTURE_2D, textureId);
        glDrawArrays(GL_TRIANGLES, g_NextVertexIndex, vertexCount);

        g_NextVertexIndex += vertexCount;
        NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);
    }

    if (isNewLine)
    {
        float offsetY = detail::GetSharedFontLineHeight() * g_FontContext.scaleY;
        DrawUtf32Text(x, y + offsetY, pString);
    }

    TouchRenderTarget();
}

/**
 * @brief   UTF-32 文字列の描画サイズ取得
 */
void GetUtf32TextSize(Size* pOutSize, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(text);

    pOutSize->width  = 0;
    pOutSize->height = 0;

    auto pString = text;
    bool isNewLine = false;

    while (*pString != '\0')
    {
        if (*pString == '\n')
        {
            isNewLine = true;
            pString++;
            break;
        }
        else
        {
            Size charSize;
            detail::GetSharedFontCharacterSize(&charSize, *pString);

            charSize.width  *= g_FontContext.scaleX;
            charSize.height *= g_FontContext.scaleY;

            pOutSize->width += charSize.width;
            pOutSize->height = std::max(pOutSize->height, charSize.width);

            pString++;
        }
    }

    if (isNewLine)
    {
        Size plusSize;
        GetUtf32TextSize(&plusSize, pString);

        pOutSize->width   = std::max(pOutSize->width, plusSize.width);
        pOutSize->height += plusSize.height;
    }
}

}  // anonymous

const Color Colors::ConstTransparent   = Color(  0,   0,   0,   0);
const Color Colors::ConstBlack         = Color(  0,   0,   0, 255);
const Color Colors::ConstShadow        = Color( 32,  32,  32, 255);
const Color Colors::ConstSmoke         = Color( 64,  64,  64, 255);
const Color Colors::ConstDimGray       = Color(105, 105, 105, 255);
const Color Colors::ConstGray          = Color(128, 128, 128, 255);
const Color Colors::ConstDarkGray      = Color(169, 169, 169, 255);
const Color Colors::ConstSilver        = Color(192, 192, 192, 255);
const Color Colors::ConstLightGray     = Color(211, 211, 211, 255);
const Color Colors::ConstWhiteSmoke    = Color(245, 245, 245, 255);
const Color Colors::ConstWhite         = Color(255, 255, 255, 255);
const Color Colors::ConstRed           = Color(255,   0,   0, 255);
const Color Colors::ConstDarkRed       = Color(128,   0,   0, 255);
const Color Colors::ConstCrymson       = Color(220,  20,  60, 255);
const Color Colors::ConstNeonRed       = Color(255,  62,  52, 255);
const Color Colors::ConstLime          = Color(  0, 255,   0, 255);
const Color Colors::ConstGreen         = Color(  0, 128,   0, 255);
const Color Colors::ConstBlue          = Color(  0,   0, 255, 255);
const Color Colors::ConstYellow        = Color(255, 255,   0, 255);
const Color Colors::ConstMagenta       = Color(255,   0, 255, 255);
const Color Colors::ConstCyan          = Color(  0, 255, 255, 255);
const Color Colors::ConstGold          = Color(255, 215,   0, 255);
const Color Colors::ConstGoldenRod     = Color(218, 165,  32, 255);
const Color Colors::ConstDarkGoldenRod = Color(184, 134,  11, 255);
const Color Colors::ConstOrange        = Color(255, 165,   0, 255);
const Color Colors::ConstViolet        = Color(238, 130, 238, 255);
const Color Colors::ConstLimeGreen     = Color( 50, 205,  50, 255);
const Color Colors::ConstPaleGreen     = Color(152, 251, 152, 255);
const Color Colors::ConstMidNightBlue  = Color( 25,  25, 112, 255);
const Color Colors::ConstNavy          = Color(  0,   0, 128, 255);
const Color Colors::ConstMediumBlue    = Color(  0,   0, 205, 255);
const Color Colors::ConstRoyalBlue     = Color( 65, 105, 225, 255);
const Color Colors::ConstNeonBlue      = Color(  0, 185, 222, 255);

void FontContext::Initialize() NN_NOEXCEPT
{
    color   = Colors::White();
    offsetX = 0.0f;
    scaleX  = 1.0f;
    scaleY  = 1.0f;
}

ScopedFontContextSaver::ScopedFontContextSaver() NN_NOEXCEPT
    : m_Context()
{
    nns::sgx::SaveFontContext(&m_Context);
}

ScopedFontContextSaver::~ScopedFontContextSaver() NN_NOEXCEPT
{
    nns::sgx::RestoreFontContext(m_Context);
}

InitializationParameters InitializationParameters::CreateDefault() NN_NOEXCEPT
{
    InitializationParameters params = {};

    params.isInitializeFs = true;
    params.vertexCountMax = DefaultVertexCountMax;
    params.origin         = OriginPosition::TopLeft;
    params.canvasSize     = DefaultCanvasSize;
    params.baseFontSize   = detail::DefaultFontSize;

    return params;
}

void Initialize() NN_NOEXCEPT
{
    auto params = InitializationParameters::CreateDefault();
    Initialize(params);
}

void Initialize(const InitializationParameters& parameters) NN_NOEXCEPT
{
    SCOPED_LOCK;

    NN_ABORT_UNLESS(!g_IsInitialized, "Already initialized");

    ChangeShader(ShaderType::Simple);

    g_FontContext.Initialize();

    g_OriginPosition = parameters.origin;
    g_ImageOpacity   = 1.0f;
    detail::SetCanvasSize(parameters.canvasSize);

    g_pRenderTargetImage = nullptr;

    g_VertexCountMax        = parameters.vertexCountMax;
    g_NextVertexIndex       = 0;
    g_pVertexPointBuffer    = new char[g_VertexCountMax * sizeof(Point3D)];
    g_pVertexColorBuffer    = new char[g_VertexCountMax * sizeof(NormalizedColor)];
    g_pVertexTexCoordBuffer = new char[g_VertexCountMax * sizeof(Point2D)];
    NN_ASSERT_NOT_NULL(g_pVertexPointBuffer);
    NN_ASSERT_NOT_NULL(g_pVertexColorBuffer);
    NN_ASSERT_NOT_NULL(g_pVertexTexCoordBuffer);

    InitializeGraphics();
    InitializeShaders();
    InitializeVertexData();

    auto params = detail::FontInitializationParameters::CreateDefault();
    params.isDynamicMode = true;
    params.baseFontSize  = parameters.baseFontSize;
    detail::InitializeSharedFont(params);

    if (parameters.isInitializeFs)
    {
        detail::InitializeFileSystem();
    }

    g_IsInitialized = true;
}

void Finalize() NN_NOEXCEPT
{
    SCOPED_LOCK;

    NN_ABORT_UNLESS(g_IsInitialized, "Not initialized");

    Destroy();
    detail::FinalizeFileSystem();
    detail::FinalizeSharedFont();
    FinalizeGraphics();

    delete[] g_pVertexTexCoordBuffer;
    delete[] g_pVertexColorBuffer;
    delete[] g_pVertexPointBuffer;

    g_IsInitialized = false;
}

void GetCanvasSize(Size* pOutSize) NN_NOEXCEPT
{
    SCOPED_LOCK;

    NN_ASSERT_NOT_NULL(pOutSize);

    *pOutSize = detail::GetCanvasSize();
}

void BeginRender() NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_NextVertexIndex = 0;
    glClear(GL_COLOR_BUFFER_BIT | GL_STENCIL_BUFFER_BIT);

    // ブレンドモードの初期化
    SetBlendMode(BlendMode::Normal);
}

void EndRender() NN_NOEXCEPT
{
    SCOPED_LOCK;

    ResetRenderTarget();
    ChangeShader(ShaderType::None);
    g_GraphicsHelper.SwapBuffers();
}

void EndRenderNoSwap() NN_NOEXCEPT
{
    SCOPED_LOCK;

    ResetRenderTarget();
    ChangeShader(ShaderType::None);
}

void SwapBuffer() NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_GraphicsHelper.SwapBuffers();
}

void SetClearColor(const Color& color) NN_NOEXCEPT
{
    SCOPED_LOCK;

    auto normalColor = NormalizedColor::FromU8Color(color);
    glClearColor(normalColor.r, normalColor.g, normalColor.b, normalColor.a);
}

void SetBlendMode(BlendMode mode) NN_NOEXCEPT
{
    SCOPED_LOCK;

    switch (mode)
    {
    case BlendMode::Normal:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
        break;

    case BlendMode::Add:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE);
        break;

    case BlendMode::Subtract:
        glBlendEquation(GL_FUNC_REVERSE_SUBTRACT);
        glBlendFunc(GL_SRC_ALPHA, GL_ONE);
        break;

    case BlendMode::Multiple:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ZERO, GL_SRC_COLOR);
        break;

    case BlendMode::Screen:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE);
        break;

    case BlendMode::Xor:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
        break;

    case BlendMode::Inverse:
        glBlendEquation(GL_FUNC_ADD);
        glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ZERO);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void SetRenderTarget(ImageData* pTargetImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTargetImage);

    SCOPED_LOCK;

    if (g_pRenderTargetImage == pTargetImage)
    {
        return;
    }
    else if (g_pRenderTargetImage != nullptr)
    {
        // 別の画像に切り替える前に一旦元に戻す
        ResetRenderTarget();
    }

    // テクスチャへの描画用フレームバッファを生成
    glGenFramebuffers(1, &g_ImageFrameBuffer);
    glBindFramebuffer(GL_FRAMEBUFFER, g_ImageFrameBuffer);
    glFramebufferTexture(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, pTargetImage->textureId, 0);

    // 描画バッファのリストをセット
    GLenum drawBuffers[] = { GL_COLOR_ATTACHMENT0 };
    glDrawBuffers(NN_ARRAY_SIZE(drawBuffers), drawBuffers);

    g_pRenderTargetImage = pTargetImage;
}

void ResetRenderTarget() NN_NOEXCEPT
{
    SCOPED_LOCK;

    if (g_pRenderTargetImage == nullptr)
    {
        return;
    }

    // 標準のフレームバッファに戻す
    glBindFramebuffer(GL_FRAMEBUFFER, 0);
    glDeleteFramebuffers(1, &g_ImageFrameBuffer);

    g_pRenderTargetImage = nullptr;
}

void ApplyRenderArea(float x, float y) NN_NOEXCEPT
{
    auto size = detail::GetCanvasSize();
    ApplyRenderArea(x, y, size.width - x, size.height - y);
}

void ApplyRenderArea(float x, float y, float width, float height) NN_NOEXCEPT
{
    ApplyRenderArea({ { x, y, width, height } });
}

void ApplyRenderArea(const Rectangle& rect) NN_NOEXCEPT
{
    SCOPED_LOCK;
    detail::ApplyRenderArea(rect);
}

void RestoreRenderArea() NN_NOEXCEPT
{
    SCOPED_LOCK;
    detail::RestoreRenderArea();
}

void SetStencilEnabled(bool isEnabled) NN_NOEXCEPT
{
    SCOPED_LOCK;

    if (isEnabled)
    {
        glEnable(GL_STENCIL_TEST);
    }
    else
    {
        glDisable(GL_STENCIL_TEST);
    }
}

void SetStencilFunc(StencilFunc func, int reference, uint32_t mask) NN_NOEXCEPT
{
    const auto getGlFunc = [](StencilFunc func) NN_NOEXCEPT
    {
        switch (func)
        {
        case StencilFunc::Never:        return GL_NEVER;
        case StencilFunc::Less:         return GL_LESS;
        case StencilFunc::LessEqual:    return GL_LEQUAL;
        case StencilFunc::Greater:      return GL_GREATER;
        case StencilFunc::GreaterEqual: return GL_GEQUAL;
        case StencilFunc::Equal:        return GL_EQUAL;
        case StencilFunc::NotEqual:     return GL_NOTEQUAL;
        case StencilFunc::Always:       return GL_ALWAYS;
        default: NN_UNEXPECTED_DEFAULT;
        }
    };

    SCOPED_LOCK;

    glStencilFunc(getGlFunc(func), reference, mask);
}

void BeginRenderToStencil() NN_NOEXCEPT
{
    SCOPED_LOCK;

    // ステンシルバッファ以外には描画しない
    glColorMask(0, 0, 0, 0);
    glDepthMask(0);

    // 値はステンシルテストせずに描画
    glStencilOp(GL_KEEP, GL_REPLACE, GL_REPLACE);
    glStencilFunc(GL_ALWAYS, 1, ~0);
}

void EndRenderToStencil() NN_NOEXCEPT
{
    SCOPED_LOCK;

    // 通常描画
    glColorMask(1, 1, 1, 1);
    glDepthMask(1);
    glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
}

void ClearStencil() NN_NOEXCEPT
{
    SCOPED_LOCK;

    glClearStencil(0);
}

void DrawPoint(float x, float y, const Color& color, float size) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    const int vertexCount = 1;

    Point3D point = { { NormalizeX(x), NormalizeY(y), 0, } };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point3D), sizeof(point), &point);

    auto colorForGl = NormalizedColor::FromU8Color(color);
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(NormalizedColor), sizeof(colorForGl), &colorForGl);

    glLineWidth(size);
    glDrawArrays(GL_POINTS, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawLine(float x1, float y1, float x2, float y2, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawGradientLine(x1, y1, x2, y2, color, color, lineWidth);
}

void DrawLine(const Point2D& startPoint, const Point2D& endPoint, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawLine(startPoint.x, startPoint.y, endPoint.x, endPoint.y, color, lineWidth);
}

void DrawGradientLine(float x1, float y1, float x2, float y2, const Color& startColor, const Color& endColor, float lineWidth) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    const int vertexCount = 2;

    Point3D points[vertexCount] =
    {
        {{ NormalizeX(x1), NormalizeY(y1), 0 }},
        {{ NormalizeX(x2), NormalizeY(y2), 0 }}
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point3D), sizeof(points), points);

    NormalizedColor colors[vertexCount] =
    {
        NormalizedColor::FromU8Color(startColor),
        NormalizedColor::FromU8Color(endColor)
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(NormalizedColor), sizeof(colors), colors);

    glLineWidth(lineWidth);
    glDrawArrays(GL_LINES, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawGradientLine(const Point2D& startPoint, const Point2D& endPoint, const Color& startColor, const Color& endColor, float lineWidth) NN_NOEXCEPT
{
    DrawGradientLine(startPoint.x, startPoint.y, endPoint.x, endPoint.y, startColor, endColor, lineWidth);
}

void DrawRectangle(float x, float y, float width, float height, const Color& color, float lineWidth) NN_NOEXCEPT
{
    Point2D points[] =
    {
        {{ x,         y }},
        {{ x + width, y }},
        {{ x + width, y + height }},
        {{ x,         y + height }}
    };

    DrawPolygon(NN_ARRAY_SIZE(points), points, color, lineWidth);
}

void DrawRectangle(const Rectangle& rect, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawRectangle(rect.x, rect.y, rect.width, rect.height, color, lineWidth);
}

void DrawPolygon(int pointCount, const Point2D* points, const Color& color, float lineWidth) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(pointCount, TemporaryVertexCountMax);

    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    for (int i = 0; i < pointCount; i++)
    {
        Normalize(&g_TempPointBuffer[i], points[i]);
        g_TempColorBuffer[i].SetU8Color(color);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        pointCount * sizeof(Point3D),
        g_TempPointBuffer
    );

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        pointCount * sizeof(NormalizedColor),
        g_TempColorBuffer
    );

    glLineWidth(lineWidth);
    glDrawArrays(GL_LINE_LOOP, g_NextVertexIndex, pointCount);

    g_NextVertexIndex += pointCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawCircle(float centerX, float centerY, float radius, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawCircleCommon(centerX, centerY, radius, color, lineWidth, GL_LINE_LOOP);
}

void DrawCircle(const Point2D& center, float radius, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawCircle(center.x, center.y, radius, color, lineWidth);
}

void DrawArc(float centerX, float centerY, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawArcCommon(centerX, centerY, radius, beginAngle, endAngle, color, lineWidth, false, GL_LINE_STRIP);
}

void DrawArc(const Point2D& center, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawArc(center.x, center.y, radius, beginAngle, endAngle, color, lineWidth);
}

void DrawPie(float centerX, float centerY, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawArcCommon(centerX, centerY, radius, beginAngle, endAngle, color, lineWidth, true, GL_LINE_STRIP);
}

void DrawPie(const Point2D& center, float radius, float beginAngle, float endAngle, const Color& color, float lineWidth) NN_NOEXCEPT
{
    DrawPie(center.x, center.y, radius, beginAngle, endAngle, color, lineWidth);
}

void FillRectangle(float x, float y, float width, float height, const Color& color) NN_NOEXCEPT
{
    Point2D points[] =
    {
        {{ x,         y }},
        {{ x + width, y }},
        {{ x + width, y + height }},
        {{ x,         y + height }}
    };

    FillPolygon(NN_ARRAY_SIZE(points), points, color);
}

void FillRectangle(const Rectangle& rectangle, const Color& color) NN_NOEXCEPT
{
    FillRectangle(rectangle.x, rectangle.y, rectangle.width, rectangle.height, color);
}

void FillPolygon(int pointCount, const Point2D* points, const Color& color) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(pointCount, TemporaryVertexCountMax);

    SCOPED_LOCK;

    for (int i = 0; i < pointCount; i++)
    {
        g_TempColorBufferU8[i] = color;
    }

    FillGradientPolygon(pointCount, points, g_TempColorBufferU8);
}

void FillGradientPolygon(int pointCount, const Point2D* points, const Color* colors) NN_NOEXCEPT
{
    NN_ASSERT_LESS_EQUAL(pointCount, TemporaryVertexCountMax);

    SCOPED_LOCK;

    ChangeShader(ShaderType::Simple);

    for (int i = 0; i < pointCount; i++)
    {
        Normalize(&g_TempPointBuffer[i], points[i]);
        g_TempColorBuffer[i].SetU8Color(colors[i]);
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        pointCount * sizeof(Point3D),
        g_TempPointBuffer
    );

    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        pointCount * sizeof(NormalizedColor),
        g_TempColorBuffer
    );

    glDrawArrays(GL_TRIANGLE_FAN, g_NextVertexIndex, pointCount);

    g_NextVertexIndex += pointCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void FillCircle(float centerX, float centerY, float radius, const Color& color) NN_NOEXCEPT
{
    DrawCircleCommon(centerX, centerY, radius, color, 1.0f, GL_TRIANGLE_FAN);
}

void FillCircle(const Point2D& center, float radius, const Color& color) NN_NOEXCEPT
{
    FillCircle(center.x, center.y, radius, color);
}

void FillPie(float centerX, float centerY, float radius, float beginAngle, float endAngle, const Color& color) NN_NOEXCEPT
{
    DrawArcCommon(centerX, centerY, radius, beginAngle, endAngle, color, 1.0f, true, GL_TRIANGLE_FAN);
}

void FillPie(const Point2D& center, float radius, float beginAngle, float endAngle, const Color& color) NN_NOEXCEPT
{
    FillPie(center.x, center.y, radius, beginAngle, endAngle, color);
}

ResultCode LoadImage(ImageData* pOutImage, const char* imageFile) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);
    NN_ASSERT_NOT_NULL(imageFile);

    SCOPED_LOCK;

    auto* pExtension = std::strrchr(imageFile, '.');
    if (pExtension == nullptr)
    {
        // 拡張子がないのでファイル種別判定不能
        return ResultCode::DecodeFailed;
    }

    char* pPixelData;
    {
        ResultCode result;
        if (NNS_SGX_STRING_LITERAL_EQUAL(pExtension, ".bmp"))
        {
            result = detail::LoadBitmap(&pOutImage->size, reinterpret_cast<void**>(&pPixelData), imageFile);
            pOutImage->imageFormat = ImageFormat::Bitmap;
        }
        else if (NNS_SGX_STRING_LITERAL_EQUAL(pExtension, ".jpg") ||
                 NNS_SGX_STRING_LITERAL_EQUAL(pExtension, ".jpeg"))
        {
            result = detail::LoadJpeg(&pOutImage->size, reinterpret_cast<void**>(&pPixelData), imageFile);
            pOutImage->imageFormat = ImageFormat::Jpeg;
        }
        else
        {
            // 非対応の拡張子
            return ResultCode::DecodeFailed;
        }

        if (result != ResultCode::Success)
        {
            return result;
        }
    }

    pOutImage->pixelFormat = PixelFormat::R8G8B8A8;
    pOutImage->isFlippedV  = false;

    {
        auto result = CreateTextureCommon(pOutImage, pPixelData);
        if (result != ResultCode::Success)
        {
            delete[] pPixelData;
            pOutImage->pPixelData = nullptr;
        }

        return result;
    }
}

void CreateImage(
    ImageData* pOutImage,
    PixelFormat pixelFormat,
    int width,
    int height) NN_NOEXCEPT
{
    NN_ASSERT_GREATER(width,  0);
    NN_ASSERT_GREATER(height, 0);

    CreateImage(
        pOutImage,
        pixelFormat,
        { { static_cast<float>(width), static_cast<float>(height) } }
    );
}

void CreateImage(
    ImageData* pOutImage,
    PixelFormat pixelFormat,
    const Size& size) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);

    SCOPED_LOCK;

    pOutImage->size        = size;
    pOutImage->imageFormat = ImageFormat::Raw;
    pOutImage->pixelFormat = pixelFormat;
    pOutImage->isFlippedV  = false;

    // 空のテクスチャを作成
    size_t pixelDataSize = size.width * size.height * GetPixelBytes(pixelFormat);
    auto pPixelData = new char[pixelDataSize];
    NN_ABORT_UNLESS_NOT_NULL(pPixelData);

    std::memset(pPixelData, 0, pixelDataSize);

    auto result = CreateTextureCommon(pOutImage, pPixelData);

    // 失敗しないはず
    NN_ABORT_UNLESS_EQUAL(result, ResultCode::Success);
}

ResultCode CreateImageByMemory(
    ImageData* pOutImage,
    const char* pImageData,
    size_t imageSize,
    ImageFormat imageFormat) NN_NOEXCEPT
{
    return CreateImageByMemory(
        pOutImage,
        pImageData,
        imageSize,
        imageFormat,
        PixelFormat::R8G8B8A8);
}

ResultCode CreateImageByMemory(
    ImageData* pOutImage,
    const char* pImageData,
    size_t imageSize,
    ImageFormat imageFormat,
    PixelFormat pixelFormat) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);
    NN_ASSERT_NOT_NULL(pImageData);

    char* pPixelData;
    switch (imageFormat)
    {
    case ImageFormat::Bitmap:
        {
            auto result = detail::DecodeBitmap(
                &(pOutImage->size),
                reinterpret_cast<void**>(&pPixelData),
                pImageData,
                imageSize);
            if (result != ResultCode::Success)
            {
                return result;
            }
        }
        break;

    case ImageFormat::Jpeg:
        {
            auto result = detail::DecodeJpeg(
                &(pOutImage->size),
                reinterpret_cast<void**>(&pPixelData),
                pImageData,
                imageSize);
            if (result != ResultCode::Success)
            {
                return result;
            }
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    pOutImage->imageFormat = imageFormat;
    pOutImage->pixelFormat = pixelFormat;
    pOutImage->isFlippedV  = false;

    auto result = CreateTextureCommon(pOutImage, pPixelData);
    if (result != ResultCode::Success)
    {
        delete[] pPixelData;
        pOutImage->pPixelData = nullptr;
    }

    return result;
}

ResultCode UpdateImageByMemory(
    ImageData* pImage,
    const char* pImageFileData,
    size_t imageFileSize,
    ImageFormat imageFormat) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImage);
    NN_ASSERT_NOT_NULL(pImageFileData);

    SCOPED_LOCK;

    char* pPixelData;
    {
        Size size;
        ResultCode result;
        switch (imageFormat)
        {
        case ImageFormat::Raw:
            {
                size_t expectSize =
                    static_cast<size_t>(pImage->size.width) *
                    static_cast<size_t>(pImage->size.height) *
                    GetPixelBytes(pImage->pixelFormat);
                if (imageFileSize != expectSize)
                {
                    return ResultCode::DecodeFailed;
                }

                // Raw の場合はメモリ再確保不要
                std::memcpy(pImage->pPixelData, pImageFileData, expectSize);
                pImage->isDirty = true;

                return ResultCode::Success;
            }
            break;

        case ImageFormat::Bitmap:
            result = detail::DecodeBitmap(
                &size,
                reinterpret_cast<void**>(&pPixelData),
                pImageFileData,
                imageFileSize);
            break;

        case ImageFormat::Jpeg:
            result = detail::DecodeJpeg(
                &size,
                reinterpret_cast<void**>(&pPixelData),
                pImageFileData,
                imageFileSize);
            break;

        default:
            NN_UNEXPECTED_DEFAULT;
        }

        if (result != ResultCode::Success)
        {
            return result;
        }
        else if (static_cast<int>(size.width) != static_cast<int>(pImage->size.width) ||
            static_cast<int>(size.height) != static_cast<int>(pImage->size.height))
        {
            // サイズが異なる
            delete[] pPixelData;
            return ResultCode::DecodeFailed;
        }
    }

#if 0
    // テクスチャの更新
    glBindTexture(GL_TEXTURE_2D, pImage->textureId);
    glTexSubImage2D(
        GL_TEXTURE_2D,
        0,
        0,
        0,
        static_cast<GLsizei>(pImage->size.width),
        static_cast<GLsizei>(pImage->size.height),
        ConvertPixelFormatForGl(pImage->pixelFormat),
        GL_UNSIGNED_BYTE,
        pPixelData);
#endif

    if (pImage->pPixelData != nullptr)
    {
        delete[] pImage->pPixelData;
    }

    // 描画中以外は glTexSubImage2D を叩けないので、テクスチャの更新は描画時まで後回し
    pImage->pPixelData = reinterpret_cast<char*>(pPixelData);
    pImage->isDirty    = true;

    return ResultCode::Success;
}

void CaptureToImage(ImageData* pOutImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutImage);

    SCOPED_LOCK;

    auto size = detail::GetCanvasSize();

    pOutImage->size        = size;
    pOutImage->imageFormat = ImageFormat::Bitmap;
    pOutImage->pixelFormat = PixelFormat::R8G8B8A8;
    pOutImage->isFlippedV  = true;

    // フレームバッファからテクスチャを作成
    size_t pixelDataSize = size.width * size.height * GetPixelBytes(PixelFormat::R8G8B8A8);
    auto pPixelData = new char[pixelDataSize];
    NN_ABORT_UNLESS_NOT_NULL(pPixelData);

    glReadBuffer(GL_BACK);
    glReadPixels(
        0,
        0,
        static_cast<GLsizei>(size.width),
        static_cast<GLsizei>(size.height),
        ConvertPixelFormatForGl(pOutImage->pixelFormat),
        GL_UNSIGNED_BYTE,
        pPixelData
    );

    auto result = CreateTextureCommon(pOutImage, pPixelData);

    // 失敗しないはず
    NN_ABORT_UNLESS_EQUAL(result, ResultCode::Success);
}

void DestroyImage(ImageData* pImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImage);

    SCOPED_LOCK;

    if (!pImage->IsValid())
    {
        return;
    }

    glDeleteTextures(1, &(pImage->textureId));

    if (pImage->imageFormat == ImageFormat::Png)
    {
        //detail::util::StbFreeImage(pImage->pPixelData);
        NN_ABORT("Unsupported format");
    }
    else
    {
        delete[] pImage->pPixelData;
    }

    pImage->pPixelData = nullptr;
}

void DrawImage(const ImageData& image, float x, float y) NN_NOEXCEPT
{
    DrawImage(image, x, y, static_cast<float>(image.size.width), static_cast<float>(image.size.height));
}

void DrawImage(const ImageData& image, const Point2D& point) NN_NOEXCEPT
{
    DrawImage(image, point.x, point.y);
}

void DrawImage(const ImageData& image, float x, float y, float width, float height) NN_NOEXCEPT
{
    Rectangle destRect = { { x, y, width, height } };
    Rectangle srcRect  = { { 0, 0, static_cast<GLfloat>(image.size.width), static_cast<GLfloat>(image.size.height) } };
    DrawImage(image, destRect, srcRect);
}

void DrawImage(const ImageData& image, const Point2D& point, const Size& size) NN_NOEXCEPT
{
    DrawImage(image, point.x, point.y, size.width, size.height);
}

void DrawImageImpl(const ImageData& image, const Point3D points[4], const Quadrangle& texCoord) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Texture);

    const int vertexCount = 4;

    // 頂点座標
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point3D),
        vertexCount * sizeof(Point3D),
        points
    );

    // 頂点色
    auto vertexColor = NormalizedColor::FromU8Color(Colors::White());
    vertexColor.a *= g_ImageOpacity;
    NormalizedColor colors[vertexCount] =
    {
        vertexColor,
        vertexColor,
        vertexColor,
        vertexColor
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        vertexCount * sizeof(NormalizedColor),
        colors);

    // テクスチャ座標
    glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point2D),
        sizeof(texCoord.points),
        texCoord.points
    );

    glBindTexture(GL_TEXTURE_2D, image.textureId);

    // テクスチャの中身が変更されていれば再作成
    if (image.isDirty)
    {
        glTexSubImage2D(
            GL_TEXTURE_2D,
            0,
            0,
            0,
            static_cast<GLsizei>(image.size.width),
            static_cast<GLsizei>(image.size.height),
            ConvertPixelFormatForGl(image.pixelFormat),
            GL_UNSIGNED_BYTE,
            image.pPixelData
        );

        image.isDirty = false;
    }

    glDrawArrays(GL_TRIANGLE_FAN, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

void DrawImage(const ImageData& image, const Quadrangle& quad) NN_NOEXCEPT
{
    const int vertexCount = NN_ARRAY_SIZE(quad.points);

    // 頂点座標
    Point3D points[vertexCount] =
    {
        {{ NormalizeX(quad.points[0].x), NormalizeY(quad.points[0].y), 0 }},
        {{ NormalizeX(quad.points[1].x), NormalizeY(quad.points[1].y), 0 }},
        {{ NormalizeX(quad.points[2].x), NormalizeY(quad.points[2].y), 0 }},
        {{ NormalizeX(quad.points[3].x), NormalizeY(quad.points[3].y), 0 }}
    };
    ApplyRotationForQuad(points);

    // テクスチャ座標
    Quadrangle texCoord =
    {{
        { { 0.0f, 0.0f } },
        { { 1.0f, 0.0f } },
        { { 1.0f, 1.0f } },
        { { 0.0f, 1.0f } }
    }};

    if (image.isFlippedH)
    {
        auto tempPoint = texCoord.points[0];
        texCoord.points[0] = texCoord.points[1];
        texCoord.points[1] = tempPoint;
        tempPoint = texCoord.points[2];
        texCoord.points[2] = texCoord.points[3];
        texCoord.points[3] = tempPoint;
        //texCoord =
        //{{
        //    { { 0.0f, 1.0f } },
        //    { { 1.0f, 1.0f } },
        //    { { 0.0f, 0.0f } },
        //    { { 1.0f, 0.0f } }
        //}};
    }

    if (image.isFlippedV)
    {
        auto tempPoint = texCoord.points[0];
        texCoord.points[0] = texCoord.points[3];
        texCoord.points[2] = tempPoint;
        tempPoint = texCoord.points[1];
        texCoord.points[1] = texCoord.points[2];
        texCoord.points[3] = tempPoint;
        //texCoord =
        //{{
        //    { { 0.0f, 1.0f } },
        //    { { 1.0f, 1.0f } },
        //    { { 0.0f, 0.0f } },
        //    { { 1.0f, 0.0f } }
        //}};
    }

    DrawImageImpl(image, points, texCoord);
}

void DrawImage(const ImageData& image, const Point2D& destPoint, const Rectangle& srcRect) NN_NOEXCEPT
{
    Rectangle destRect = { { destPoint.x, destPoint.y, srcRect.width, srcRect.height } };
    DrawImage(image, destRect, srcRect);
}

void DrawImage(const ImageData& image, const Rectangle& destRect, const Rectangle& srcRect) NN_NOEXCEPT
{
    SCOPED_LOCK;

    ChangeShader(ShaderType::Texture);

    const int vertexCount = 4;

    // 頂点座標
    Point3D points[vertexCount] =
    {
        {{ NormalizeX(destRect.x),                  NormalizeY(destRect.y),                   0 }},
        {{ NormalizeX(destRect.x + destRect.width), NormalizeY(destRect.y),                   0 }},
        {{ NormalizeX(destRect.x + destRect.width), NormalizeY(destRect.y + destRect.height), 0 }},
        {{ NormalizeX(destRect.x),                  NormalizeY(destRect.y + destRect.height), 0 }}
    };
    ApplyRotationForQuad(points);
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexBufferId);
    glBufferSubData(GL_ARRAY_BUFFER, g_NextVertexIndex * sizeof(Point3D), sizeof(points), points);

    // 頂点色
    auto vertexColor = NormalizedColor::FromU8Color(Colors::White());
    vertexColor.a *= g_ImageOpacity;
    NormalizedColor colors[vertexCount] =
    {
        vertexColor,
        vertexColor,
        vertexColor,
        vertexColor
    };
    glBindBuffer(GL_ARRAY_BUFFER, g_VertexColorBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(NormalizedColor),
        vertexCount * sizeof(NormalizedColor),
        colors
    );

    // 描画サイズからテクスチャ座標を計算
    GLfloat x      = srcRect.x      / static_cast<GLfloat>(image.size.width);
    GLfloat y      = srcRect.y      / static_cast<GLfloat>(image.size.height);
    GLfloat width  = srcRect.width  / static_cast<GLfloat>(image.size.width);
    GLfloat height = srcRect.height / static_cast<GLfloat>(image.size.height);
    Point2D texCoords[vertexCount]
    {
        {{ x,         y }},
        {{ x + width, y }},
        {{ x + width, y + height }},
        {{ x,         y + height }}
    };

    if (image.isFlippedH)
    {
        auto tempCoord = texCoords[0];
        texCoords[0] = texCoords[1];
        texCoords[1] = tempCoord;
        tempCoord = texCoords[2];
        texCoords[2] = texCoords[3];
        texCoords[3] = tempCoord;
    }

    if (image.isFlippedV)
    {
        auto tempCoord = texCoords[0];
        texCoords[0] = texCoords[3];
        texCoords[3] = tempCoord;
        tempCoord = texCoords[1];
        texCoords[1] = texCoords[2];
        texCoords[2] = tempCoord;
    }

    glBindBuffer(GL_ARRAY_BUFFER, g_TexCoordBufferId);
    glBufferSubData(
        GL_ARRAY_BUFFER,
        g_NextVertexIndex * sizeof(Point2D),
        sizeof(texCoords),
        texCoords
    );

    glBindTexture(GL_TEXTURE_2D, image.textureId);

    if (image.isDirty)
    {
        glTexSubImage2D(
            GL_TEXTURE_2D,
            0,
            0,
            0,
            static_cast<GLsizei>(image.size.width),
            static_cast<GLsizei>(image.size.height),
            ConvertPixelFormatForGl(image.pixelFormat),
            GL_UNSIGNED_BYTE,
            image.pPixelData
        );
    }

    glDrawArrays(GL_TRIANGLE_FAN, g_NextVertexIndex, vertexCount);

    g_NextVertexIndex += vertexCount;
    NN_ASSERT_LESS(g_NextVertexIndex, g_VertexCountMax);

    TouchRenderTarget();
}

float GetImageOpacity() NN_NOEXCEPT
{
    SCOPED_LOCK;

    return g_ImageOpacity;
}

void SetImageOpacity(float opacity) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_ImageOpacity = std::min(std::max(opacity, 0.0f), 1.0f);
}

void SetImageRotation(float angle) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_ImageRotation = angle;
}

void SaveFontContext(FontContext* pOutContext) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutContext);
    SCOPED_LOCK;

    *pOutContext = g_FontContext;
}

void RestoreFontContext(const FontContext& context) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_FontContext = context;
}

void GetTextColor(Color* pOutColor) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutColor);
    SCOPED_LOCK;

    *pOutColor = g_FontContext.color;
}

void SetTextColor(const Color& color) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_FontContext.color = color;
}

void GetTextScale(float* pOutScaleX, float* pOutScaleY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutScaleX);
    NN_ASSERT_NOT_NULL(pOutScaleY);
    SCOPED_LOCK;

    *pOutScaleX = g_FontContext.scaleX;
    *pOutScaleY = g_FontContext.scaleY;
}

void GetTextSisze(float* pOutSizeX, float* pOutSizeY) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSizeX);
    NN_ASSERT_NOT_NULL(pOutSizeY);
    SCOPED_LOCK;

    int baseSize = detail::GetSharedFontBaseSize();
    *pOutSizeX = g_FontContext.scaleX * baseSize;
    *pOutSizeY = g_FontContext.scaleY * baseSize;
}

void SetTextScale(float scaleX, float scaleY) NN_NOEXCEPT
{
    SCOPED_LOCK;

    g_FontContext.scaleX = std::max(scaleX, 0.0f);
    g_FontContext.scaleY = std::max(scaleY, 0.0f);
}

void SetTextSize(float sizeX, float sizeY) NN_NOEXCEPT
{
    int baseSize = detail::GetSharedFontBaseSize();
    float scaleX = sizeX / baseSize;
    float scaleY = sizeY / baseSize;
    SetTextScale(scaleX, scaleY);
}

bool VConvertUtf8ToUtf32(uint32_t* pOutBuffer, int bufferLength, const char* text, std::va_list args) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutBuffer);
    NN_ASSERT_NOT_NULL(text);

    static char s_Buffer[TemporaryStringBufferSize];

    // 書式指定の展開
    auto length = std::vsnprintf(s_Buffer, sizeof(s_Buffer), text, args);
    if (length < 0)
    {
        // バッファサイズオーバー
        return false;
    }

    // UTF8 -> UTF32 変換
    {
        int lengthUtf32;
        auto result = nn::util::GetLengthOfConvertedStringUtf8ToUtf32(&lengthUtf32, s_Buffer);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf32 >= TemporaryStringBufferSize)
        {
            // 変換不能
            return false;
        }
    }

    auto result = nn::util::ConvertStringUtf8ToUtf32(pOutBuffer, TemporaryStringBufferSize, s_Buffer);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        return false;
    }

    return true;
}

void DrawText(float x, float y, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(format);

    SCOPED_LOCK;

    static uint32_t s_Utf32Buffer[TemporaryStringBufferSize];

    std::va_list args;
    va_start(args, format);
    auto isSuccess = VConvertUtf8ToUtf32(s_Utf32Buffer, TemporaryStringBufferSize, format, args);
    va_end(args);
    if (!isSuccess)
    {
        // 変換失敗
        return;
    }

    DrawUtf32Text(x, y, s_Utf32Buffer);
}

void DrawText(float x, float y, float width, float height, TextAlignment alignment, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(format);
    NN_ASSERT_MINMAX(
        static_cast<int>(alignment),
        static_cast<int>(TextAlignment::Left),
        static_cast<int>(TextAlignment::Right)
    );

    SCOPED_LOCK;

    static uint32_t s_Utf32Buffer[TemporaryStringBufferSize];

    std::va_list args;
    va_start(args, format);
    auto isSuccess = VConvertUtf8ToUtf32(s_Utf32Buffer, TemporaryStringBufferSize, format, args);
    va_end(args);
    if (!isSuccess)
    {
        // 変換失敗
        return;
    }

    DrawText(x, y, width, height, alignment, s_Utf32Buffer);
}

void DrawText(float x, float y, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    SCOPED_LOCK;

    DrawUtf32Text(x, y, text);
}

void DrawText(float x, float y, float width, float height, TextAlignment alignment, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);
    NN_ASSERT_MINMAX(
        static_cast<int>(alignment),
        static_cast<int>(TextAlignment::Left),
        static_cast<int>(TextAlignment::Right)
    );

    SCOPED_LOCK;

    // 描画サイズから位置を調整
    Size size;
    GetTextDrawSize(&size, text);

    float destX = x;
    switch (alignment)
    {
    case TextAlignment::Left:
        // 既に設定済み
        break;

    case TextAlignment::Center:
        destX += std::max((width - size.width) / 2.0f, 0.0f);
        break;

    case TextAlignment::Right:
        destX += std::max(width - size.width, 0.0f);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    float destY = y + std::max((height - size.height) / 2.0f, 0.0f);
    DrawUtf32Text(destX, destY, text);
}

void DrawText(float x, float y, const uint16_t* text, int textLength) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    SCOPED_LOCK;

    // UTF16 -> UTF8 変換
    static char s_Utf8Buffer[TemporaryStringBufferSize];

    {
        int lengthUtf8;
        auto result = nn::util::GetLengthOfConvertedStringUtf16NativeToUtf8(&lengthUtf8, text, textLength);
        if (result != nn::util::CharacterEncodingResult_Success ||
            lengthUtf8 >= TemporaryStringBufferSize)
        {
            // 変換不能
            return;
        }
    }

    auto result = nn::util::ConvertStringUtf16NativeToUtf8(s_Utf8Buffer, TemporaryStringBufferSize, text);
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        // 変換失敗
        return;
    }

    DrawText(x, y, s_Utf8Buffer);
}

void GetTextDrawSize(Size* pOutSize, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(format);

    SCOPED_LOCK;

    static uint32_t s_Utf32Buffer[TemporaryStringBufferSize];

    std::va_list args;
    va_start(args, format);
    auto success = VConvertUtf8ToUtf32(s_Utf32Buffer, TemporaryStringBufferSize, format, args);
    va_end(args);
    if (!success)
    {
        // 変換失敗
        pOutSize->width  = 0;
        pOutSize->height = 0;
        return;
    }

    GetTextDrawSize(pOutSize, s_Utf32Buffer);
}

void GetTextDrawSize(Size* pOutSize, const uint32_t* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutSize);
    NN_ASSERT_NOT_NULL(text);

    SCOPED_LOCK;

    GetUtf32TextSize(pOutSize, text);
}

}}  // nns::sgx
