﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_StringUtil.h>

#include "../sgx/SimpleGfx.h"
#include "../sgx/gui/SimpleGfx_GuiCommon.h"
#include "../util/StringTable.h"

#define ENABLE_DEBUG_LOG
#if defined(ENABLE_DEBUG_LOG)
#define PRINT_SCENE_NAME(log)   NN_LOG("[Scene] %s: %s\n", m_Name, log)
#else
#define PRINT_SCENE_NAME(log)   static_cast<void*>(0)
#endif  // if defined(ENABLE_DEBUG_LOG)

namespace nns { namespace hid { namespace scene {

/**
 * @brief   シーン名の最大長です。
 */
const int SceneNameLengthMax = 32;

/**
 * @brief   シーンの基本クラスです。
 */
class SceneBase
{
    NN_DISALLOW_COPY(SceneBase);
    NN_DISALLOW_MOVE(SceneBase);

public:
    /**
     * @brief   シーン名を指定せずにクラスを初期化します。
     */
    SceneBase() NN_NOEXCEPT
        : SceneBase("")
    {}

    /**
     * @brief   シーン名を指定してクラスを初期化します。
     */
    explicit SceneBase(const char* sceneName) NN_NOEXCEPT
        : m_Name()
        , m_IsAutoDeleteEnabled(false)
        , m_IsInitialized(false)
        , m_IsActive(false)
        , m_IsTerminated(false)
        , m_BackgroundColor()
        , m_MainContainer()
    {
        SetName(sceneName);

        nns::sgx::Size canvasSize;
        nns::sgx::GetCanvasSize(&canvasSize);
        m_MainContainer.SetSize(canvasSize);
        m_MainContainer.SetFocus();
    }

    virtual ~SceneBase() NN_NOEXCEPT {}

    /**
     * @brief   シーン名を取得します。
     */
    const char* GetName() const NN_NOEXCEPT
    {
        return m_Name;
    }

    /**
     * @brief   シーン名を設定します。
     */
    void SetName(const char* name) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(name);

        nn::util::Strlcpy(m_Name, name, sizeof(m_Name));
    }

    /**
     * @brief   シーン名の一致判定を行います。
     */
    bool IsNameMatched(const char* name) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(name);

        return nn::util::Strncmp(GetName(), name, SceneNameLengthMax) == 0 &&
            nn::util::Strnlen(GetName(), SceneNameLengthMax) == nn::util::Strnlen(name, SceneNameLengthMax);
    }

    /**
     * @brief   シーンの自動削除が有効かどうか判定します。
     */
    bool IsAutoDeleteEnabled() const NN_NOEXCEPT
    {
        return m_IsAutoDeleteEnabled;
    }

    /**
     * @brief   シーンの自動削除を有効にします。
     */
    void EnableAutoDelete() NN_NOEXCEPT
    {
        m_IsAutoDeleteEnabled = true;
    }

    /**
     * @brief   シーンが初期化済みかどうか判定します。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    /**
     * @brief   シーンがアクティブ状態かどうか判定します。
     */
    virtual bool IsActive() const NN_NOEXCEPT
    {
        return m_IsActive;
    }

    /**
     * @brief   シーンが終了しているか判定します。
     */
    virtual bool IsTerminated() const NN_NOEXCEPT
    {
        return m_IsTerminated;
    }

    /**
     * @brief   シーンの背景色を取得します。
     */
    const nns::sgx::Color& GetBackgroundColor() NN_NOEXCEPT
    {
        return m_BackgroundColor;
    }

    /**
     * @brief   シーンの背景色を設定します。
     */
    void SetBackgroundColor(const nns::sgx::Color& color) NN_NOEXCEPT
    {
        m_BackgroundColor = color;
    }

    /**
     * @brief   シーンを初期化します。
     */
    virtual void Initialize() NN_NOEXCEPT
    {
        m_IsInitialized   = true;
        m_IsActive        = false;
        m_IsTerminated    = false;
        m_BackgroundColor = nns::sgx::Color(0xF0, 0xF0, 0xF0, 0xFF);
        m_MainContainer.ClearChildren();
    }

    /**
     * @brief   シーンをアクティブ化します。
     */
    virtual void Activate() NN_NOEXCEPT
    {
        m_IsActive = true;
        PRINT_SCENE_NAME("Activated");
    }

    /**
     * @brief   シーンを非アクティブ化します。
     */
    virtual void Deactivate() NN_NOEXCEPT
    {
        m_IsActive = false;
        m_MainContainer.CancelCursorAnimation();

        PRINT_SCENE_NAME("Deactivated");
    }

    /**
     * @brief   シーンを終了します。
     */
    virtual void Terminate() NN_NOEXCEPT
    {
        m_IsTerminated = true;
        PRINT_SCENE_NAME("Terminated");
    }

    /**
     * @brief   シーンを更新します。
     */
    virtual void Update() NN_NOEXCEPT
    {
        m_MainContainer.Update();
        m_MainContainer.UpdateKeyInput();
    }

    /**
     * @brief   シーンを描画します。
     */
    void Render() NN_NOEXCEPT
    {
        RenderMain();
    }

    /**
     * @brief   子要素を追加します。
     */
    void AddChild(nns::sgx::gui::DisplayObject* pChild) NN_NOEXCEPT
    {
        m_MainContainer.AddChild(pChild);
    }

    /**
     * @brief   子要素を削除します。
     */
    void RemoveChild(const nns::sgx::gui::DisplayObject* pChild) NN_NOEXCEPT
    {
        m_MainContainer.RemoveChild(pChild);
    }

    /**
     * @brief   フォーカスを当てるオブジェクトを指定します。
     */
    void SetFocusedChild(nns::sgx::gui::DisplayObject* pChild) NN_NOEXCEPT
    {
        m_MainContainer.SetFocusedChild(pChild);
    }

protected:
    /**
     * @brief   シーン描画のメイン処理です。
     */
    virtual void RenderMain() NN_NOEXCEPT
    {
        m_MainContainer.Render();
    }

    /**
     * @brief   リソース文字列を取得します。
     */
    const char* GetResourceString(const char* key) NN_NOEXCEPT
    {
        return nns::hid::util::StringTable::GetInstance().Get(key);
    }

private:
    char    m_Name[SceneNameLengthMax + 1];             //!< シーン名
    bool    m_IsAutoDeleteEnabled;                      //!< 自動 delete フラグ
    bool    m_IsInitialized;                            //!< 初期化済みフラグ
    bool    m_IsActive;                                 //!< アクティブ状態
    bool    m_IsTerminated;                             //!< 終了済みフラグ

    nns::sgx::Color              m_BackgroundColor;      //!< シーンの背景色
    nns::sgx::gui::UiContainer   m_MainContainer;        //!< 画面全体をカバーするメインの UI コンテナ
};

}}}  // nns::hid::scene
