﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>
#include <vector>

#include <nn/nn_Macro.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Vibration.h>

namespace input {

/**
 * @brief   対応する NpadId の一覧です。
 */
const nn::hid::NpadIdType SupportedNpadIds[] =
{
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
    nn::hid::NpadId::No5,
    nn::hid::NpadId::No6,
    nn::hid::NpadId::No7,
    nn::hid::NpadId::No8,
    nn::hid::NpadId::Handheld
};

/**
 * @brief   対応する NpadId の数です。
 */
const int NpadIdCountMax = NN_ARRAY_SIZE(SupportedNpadIds);

/**
 * @brief   キーリピートの判定を開始するまでの遅延時間 (フレーム単位) です。
 */
const uint32_t DefaultRepeatWait = 30;

/**
 * @brief   キーリピートの間隔 (フレーム単位) です。
 */
const uint32_t DefaultRepeatInterval = 6;

/**
 * @brief   左コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadLeftJoy
{
    typedef nn::hid::NpadStyleJoyLeft Style;
    typedef nn::hid::NpadJoyLeftState State;
};

/**
 * @brief   右コン操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadRightJoy
{
    typedef nn::hid::NpadStyleJoyRight Style;
    typedef nn::hid::NpadJoyRightState State;
};

/**
 * @brief   JoyDual 操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadJoyDual
{
    typedef nn::hid::NpadStyleJoyDual Style;
    typedef nn::hid::NpadJoyDualState State;
};

/**
 * @brief   フルキー操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadFullKey
{
    typedef nn::hid::NpadStyleFullKey Style;
    typedef nn::hid::NpadFullKeyState State;
};

/**
 * @brief   携帯機操作に関する定義 (NpadController クラスのテンプレート引数として利用します)
 */
struct NpadHandheld
{
    typedef nn::hid::NpadStyleHandheld Style;
    typedef nn::hid::NpadHandheldState State;
};

/**
 * @brief   コントローラーに対する処理を扱うインターフェースクラスです。
 */
class INpadController
{
    NN_DISALLOW_COPY(INpadController);
    NN_DISALLOW_MOVE(INpadController);

public:
    INpadController() NN_NOEXCEPT {}
    virtual ~INpadController() NN_NOEXCEPT {}
    virtual void Initialize() NN_NOEXCEPT = 0;
    virtual void Update() NN_NOEXCEPT = 0;

    /**
     * @brief   入力無効状態の取得
     */
    static bool IsSilent() NN_NOEXCEPT
    {
        return g_IsSilent;
    }

    /**
     * @brief   入力無効状態の設定
     */
    static void SetSilent(bool isSilent) NN_NOEXCEPT
    {
        g_IsSilent = isSilent;
    }

    /**
     * @brief   NpadStyle の一致判定
     */
    virtual bool TestStyle(nn::hid::NpadStyleSet style) const NN_NOEXCEPT = 0;

    /**
     * @brief   NpadId を取得
     */
    virtual nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT = 0;

    /**
     * @brief   指定されたボタンの組み合わせが押されているか
     */
    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsPressed() const NN_NOEXCEPT
    {
        return IsPressed(ButtonT::Mask);
    }

    /**
     * @brief   指定されたボタンのいずれかが押されているか
     */
    virtual bool IsAnyPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsAnyPressed() const NN_NOEXCEPT
    {
        return IsAnyPressed(ButtonT::Mask);
    }

    /**
     * @brief   指定されたボタンの組み合わせが新たに押されたか
     */
    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsTriggered() const NN_NOEXCEPT
    {
        return IsTriggered(ButtonT::Mask);
    }

    /**
     * @brief   指定されたボタンのいずれかが新たに押されたか
     */
    virtual bool IsAnyTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsAnyTriggered() const NN_NOEXCEPT
    {
        return IsAnyTriggered(ButtonT::Mask);
    }

    /**
     * @brief   指定されたボタンの組み合わせが離されたか
     */
    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsReleased() const NN_NOEXCEPT
    {
        return IsReleased(ButtonT::Mask);
    }

    /**
     * @brief   指定されたボタンのいずれかが離されたか
     */
    virtual bool IsAnyReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsAnyReleased() const NN_NOEXCEPT
    {
        return IsAnyReleased(ButtonT::Mask);
    }

    /**
     * @brief   指定されたボタンがリピート入力されているか
     */
    virtual bool IsRepeated(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT = 0;

    template <typename ButtonT>
    bool IsRepeated() const NN_NOEXCEPT
    {
        return IsRepeated(ButtonT::Mask);
    }

    /**
     * @brief   左スティックの入力状態
     */
    virtual nn::hid::AnalogStickState GetLeftAnalogStickState() const NN_NOEXCEPT = 0;

    /**
     * @brief   右スティックの入力状態
     */
    virtual nn::hid::AnalogStickState GetRightAnalogStickState() const NN_NOEXCEPT = 0;

    /**
     * @brief   何かしらの操作があったか
     */
    virtual bool IsControlled() const NN_NOEXCEPT = 0;

    /**
     * @brief   指定したボタン押し (押下 → 離す) をエミュレーション
     */
    virtual void PerformPress(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT = 0;

private:
    static bool                 g_IsSilent;
};

/**
 * @brief   操作形態ごとの処理を記述した、 INpadController の派生クラスです。
 */
template <typename Npad>
class NpadController : public INpadController
{
    NN_DISALLOW_COPY(NpadController);
    NN_DISALLOW_MOVE(NpadController);

public:
    explicit NpadController(nn::hid::NpadIdType npadId) NN_NOEXCEPT
        : m_NpadId(npadId)
        , m_IsDummy(false)
        , m_CurrentButtons()
        , m_PreviousButtons()
        , m_EmulateButtons()
        , m_AnalogStickL()
        , m_AnalogStickR()
        , m_RepeatInfo()
        , m_RepeatWait(DefaultRepeatWait)
        , m_RepeatInterval(DefaultRepeatInterval)
        , m_EmulateDelay(0)
    {}

    void SetDummy(bool isDummy) NN_NOEXCEPT
    {
        m_IsDummy = isDummy;
    }

    virtual void Initialize() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual bool TestStyle(nn::hid::NpadStyleSet style) const NN_NOEXCEPT NN_OVERRIDE
    {
        return style == Npad::Style::Mask;
    }

    virtual nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_NpadId;
    }

    virtual bool IsPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return !IsSilent() && (m_CurrentButtons & buttons) == buttons;
    }

    virtual bool IsAnyPressed(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return !IsSilent() && (m_CurrentButtons & buttons).IsAnyOn();
    }

    virtual bool IsTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return !IsSilent() &&
            (m_CurrentButtons & buttons) == buttons &&
            (m_PreviousButtons & buttons) != buttons;
    }

    virtual bool IsAnyTriggered(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        // 何も押されていない状態から何かが押されたらトリガー
        return !IsSilent() &&
            (m_CurrentButtons & buttons).IsAnyOn() &&
            (m_PreviousButtons & buttons).IsAllOff();
    }

    virtual bool IsReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        return !IsSilent() &&
            (m_CurrentButtons & buttons) != buttons &&
            (m_PreviousButtons & buttons) == buttons;
    }

    virtual bool IsAnyReleased(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        auto curr = m_CurrentButtons & buttons;
        auto prev = m_PreviousButtons & buttons;

        // 押されているボタンが 0 になったとき
        return !IsSilent() &&
            curr.IsAllOff() &&
            curr.CountPopulation() < prev.CountPopulation();
    }

    virtual bool IsRepeated(nn::hid::NpadButtonSet buttons) const NN_NOEXCEPT NN_OVERRIDE
    {
        if (IsSilent() || buttons.IsAllOff())
        {
            return false;
        }

        for (int i = 0; i < NN_ARRAY_SIZE(m_RepeatInfo); i++)
        {
            if (!buttons.Test(i))
            {
                continue;
            }

            const auto& repeat = m_RepeatInfo[i];
            if (repeat.waitCount == 1)
            {
                // 押下直後
                return true;
            }

            if (repeat.waitCount < m_RepeatWait)
            {
                // 初回リピート待ち
                return false;
            }

            if (repeat.intervalCount == 0)
            {
                // 2 回目以降のリピート
                return true;
            }
        }

        return false;
    }

    virtual nn::hid::AnalogStickState GetLeftAnalogStickState() const NN_NOEXCEPT NN_OVERRIDE
    {
        return IsSilent() ? nn::hid::AnalogStickState() : m_AnalogStickL;
    }

    virtual nn::hid::AnalogStickState GetRightAnalogStickState() const NN_NOEXCEPT NN_OVERRIDE
    {
        return IsSilent() ? nn::hid::AnalogStickState() : m_AnalogStickR;
    }

    virtual bool IsControlled() const NN_NOEXCEPT NN_OVERRIDE
    {
        return !IsSilent() &&
            (m_CurrentButtons.IsAnyOn() || m_CurrentButtons != m_PreviousButtons);
    }

    virtual void PerformPress(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT NN_OVERRIDE
    {
        m_EmulateButtons = buttons;
        m_EmulateDelay   = 3;
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        UpdateButtons();
    }

private:
    struct RepeatInfo
    {
        uint32_t    waitCount;
        uint32_t    intervalCount;
    };

private:
    /**
     * @brief   リピート判定の更新
     */
    void UpdateRepeat() NN_NOEXCEPT
    {
        for (int i = 0; i < NN_ARRAY_SIZE(m_RepeatInfo); i++)
        {
            auto& repeat = m_RepeatInfo[i];
            if (!m_CurrentButtons.Test(i))
            {
                repeat.waitCount     = 0;
                repeat.intervalCount = 0;
                continue;
            }

            if (repeat.waitCount < m_RepeatWait)
            {
                // 初回リピート開始待ち
                repeat.waitCount++;
                repeat.intervalCount = 0;
            }
            else
            {
                // リピート中
                repeat.intervalCount = (repeat.intervalCount + 1) % m_RepeatInterval;
            }

        }
    }

    /**
     * @brief   ボタン押下状態の更新
     */
    void UpdateButtons() NN_NOEXCEPT
    {
        m_PreviousButtons = m_CurrentButtons;

        typename Npad::State state;
        if (m_IsDummy)
        {
            state = typename Npad::State();
        }
        else
        {
            nn::hid::GetNpadState(&state, m_NpadId);
        }

        if (m_EmulateDelay > 1)
        {
            m_CurrentButtons = m_EmulateButtons;
            m_EmulateDelay--;
        }
        else if (m_EmulateDelay == 1)
        {
            m_CurrentButtons = nn::hid::NpadButtonSet();
            m_EmulateDelay = 0;
        }
        else
        {
            m_CurrentButtons = state.buttons;
        }

        m_AnalogStickL = state.analogStickL;
        m_AnalogStickR = state.analogStickR;

        UpdateRepeat();
    }

private:
    const nn::hid::NpadIdType   m_NpadId;           //!< Npad ID
    bool                        m_IsDummy;          //!< ダミーデバイスか
    nn::hid::NpadButtonSet      m_CurrentButtons;   //!< 現在のボタン情報
    nn::hid::NpadButtonSet      m_PreviousButtons;  //!< 前回のボタン情報
    nn::hid::NpadButtonSet      m_EmulateButtons;   //!< 押下エミュレーション用のボタン情報
    nn::hid::AnalogStickState   m_AnalogStickL;     //!< 左スティック
    nn::hid::AnalogStickState   m_AnalogStickR;     //!< 右スティック

    RepeatInfo                  m_RepeatInfo[sizeof(nn::hid::NpadButtonSet) * 8];
    uint32_t                    m_RepeatWait;
    uint32_t                    m_RepeatInterval;
    uint32_t                    m_EmulateDelay;
};

/**
 * @brief   接続されているコントローラーを管理するクラスです。
 */
class NpadManager final
{
    NN_DISALLOW_COPY(NpadManager);
    NN_DISALLOW_MOVE(NpadManager);

public:
    static NpadManager* GetInstance() NN_NOEXCEPT
    {
        static NpadManager s_Manager;
        return &s_Manager;
    }

    /**
     * @brief   接続可能なコントローラーの台数を返します。
     */
    static int GetPadCountMax() NN_NOEXCEPT
    {
        return NpadIdCountMax;
    }

    /**
     * @brief   指定した NpadId に対応する NpadController を取得します。
     */
    INpadController* GetNpadController(const nn::hid::NpadIdType& id) NN_NOEXCEPT;

    /**
     * @brief   どの NpadId にも属さないダミー入力用 NpadController を取得します。
     */
    INpadController* GetDummyController() NN_NOEXCEPT;

    /**
     * @brief   接続されている NpadController の一覧を取得します。
     */
    std::vector<INpadController*> GetNpadControllers() NN_NOEXCEPT;

    /**
     * @brief   指定された NpadId にコントローラーが接続されているか判定します。
     */
    bool IsConnected(const nn::hid::NpadIdType& id) NN_NOEXCEPT;

    /**
     * @brief   最後に操作されたコントローラーの NpadId を取得します。
     */
    nn::hid::NpadIdType GetLastController() NN_NOEXCEPT
    {
        return m_LastControlledNpadId;
    }

    /**
     * @brief   コントローラー情報を更新します。
     */
    void Update() NN_NOEXCEPT;

private:
    /**
     * @brief   NpadController を格納できる十分なサイズ
     */
    static const size_t NpadControllerStorageSize = 1024;

    struct ControllerStorageType
    {
        char storage[NpadControllerStorageSize];
        nn::hid::NpadStyleSet style;

        bool IsAvailable() const NN_NOEXCEPT
        {
            return style.IsAnyOn();
        }
    };

private:
    NpadManager() NN_NOEXCEPT
        : m_Mutex(true)
        , m_NpadControllersStorage()
        , m_LastControlledNpadId()
    {
        Initialize();
    }

    ~NpadManager() NN_NOEXCEPT
    {
        Finalize();
    }

    /**
     * @brief   Npad の初期化処理を行います。
     */
    void Initialize() NN_NOEXCEPT;

    /**
     * @brief   Npad の終了処理を行います。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   最終操作コントローラーを更新します。
     */
    void UpdateLastControlledId() NN_NOEXCEPT;

private:
    mutable nn::os::Mutex       m_Mutex;

    ControllerStorageType       m_NpadControllersStorage[NpadIdCountMax + 1];
    nn::hid::NpadIdType         m_LastControlledNpadId;
};

/**
 * @brief   接続されている NpadController の一覧を取得するヘルパー関数です。
 */
NN_FORCEINLINE
std::vector<INpadController*> GetNpadControllers() NN_NOEXCEPT
{
    return NpadManager::GetInstance()->GetNpadControllers();
}

NN_FORCEINLINE
bool IsAnyTriggered(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT
{
    auto pControllers = GetNpadControllers();
    for (auto pController : pControllers)
    {
        if (pController->IsAnyTriggered(buttons))
        {
            return true;
        }
    }

    if (NpadManager::GetInstance()->GetDummyController()->IsAnyTriggered(buttons))
    {
        return true;
    }

    return false;
}

/**
 * @brief   接続されている NpadController のうち、いずれかのキー入力があるかどうかを判定するヘルパー関数です。
 */
template <typename Button>
bool IsAnyTriggered() NN_NOEXCEPT
{
    auto pControllers = GetNpadControllers();
    for (auto pController : pControllers)
    {
        if (pController->IsTriggered<Button>())
        {
            return true;
        }
    }

    if (NpadManager::GetInstance()->GetDummyController()->IsTriggered<Button>())
    {
        return true;
    }

    return false;
}

/**
 * @brief   接続されている NpadController のうち、いずれかのキー入力があるかどうかを判定するヘルパー関数です。
 */
template <typename Button>
bool IsAnyRepeated() NN_NOEXCEPT
{
    auto pControllers = GetNpadControllers();
    for (auto pController : pControllers)
    {
        if (pController->IsRepeated<Button>())
        {
            return true;
        }
    }

    if (NpadManager::GetInstance()->GetDummyController()->IsReleased<Button>())
    {
        return true;
    }

    return false;
}

/**
 * @brief   最後に操作されたコントローラーに対して、指定したボタンの押下をエミュレートします。
 *
 * @details 最後に操作されたコントローラーが切断されている場合は、最初に見つかったコントローラーを対象とします。
 */
void PerformPressLastController(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT;

/**
 * @brief   ダミーコントローラーに対して、指定したボタンの押下をエミュレートします。
 */
void PerformPressDummyController(nn::hid::NpadButtonSet buttons) NN_NOEXCEPT;

}  // input
