﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "layout_Base.h"

namespace nns { namespace hidfw { namespace layout {

    void BaseItem::SetDefault() NN_NOEXCEPT
    {
        m_FrameCount = 0;
        m_AnimationCount = 0;

        m_MainColor = nn::util::Color4u8::White();
        m_SubColor = nn::util::Color4u8::Gray();
        m_TextColor = nn::util::Color4u8::Blue();
        m_BorderColor = nn::util::Color4u8::Black();
        m_EffectColor = nn::util::Color4u8::Green();

        SetPos(0.f, 0.f);
        SetSize(200.f, 50.f);
        m_Scale = nn::util::MakeFloat2(1.f, 1.f);
        m_Text = "Item";
        m_Func = nullptr;
        m_Param = nullptr;

        m_ChooseMode.Reset();
        m_ChooseMode = ChoiceMode::Controller::Mask | ChoiceMode::Touch::Mask;
        m_Option.Reset();
        m_Option |= (LayoutOption::Visible::Mask | LayoutOption::ForciblyFocus::Mask | LayoutOption::OnOffMode::Mask);
        m_State.Reset();
        m_State |= LayoutState::Enable::Mask;
        m_OldState.Reset();
        m_OldState = m_State;
    }

    void BaseItem::Setup(float x, float y, float w, float h, LayoutFunction func, const char* text, ...) NN_NOEXCEPT
    {
        SetPos(x, y);
        SetSize(w, h);
        SetFunc(func);

        std::va_list list;
        va_start(list, text);
        SetText(text, list);
        va_end(list);

        m_Scale = nn::util::MakeFloat2(1.f, 1.f);
    }

    void BaseItem::SetText(const char* text, ...) NN_NOEXCEPT
    {
        std::va_list list;
        char* pText = nullptr;
        va_start(list, text);
        vasprintf(&pText, text, list);
        m_Text = pText;
        va_end(list);
        free(pText);
    }

    void BaseItem::Cancel() NN_NOEXCEPT
    {
        m_State.Set(LayoutState::Selected::Index, false);
        m_State.Set(LayoutState::Canceled::Index, true);
    }

    void BaseItem::Select(bool callFunc) NN_NOEXCEPT
    {
        if (m_State.Test<LayoutState::Enable>())
        {
            m_Option.Set(LayoutOption::CallFunction::Index, callFunc);
            m_State.Set(LayoutState::Canceled::Index, false);
            m_State.Set(LayoutState::Selected::Index, true);
        }
    }

    void BaseItem::Hover(bool enable) NN_NOEXCEPT
    {
        m_State.Set(LayoutState::Hover::Index, enable);
    }

    //==============================================================================================================================================

    int BaseItemSet::m_MoveForciblyFocusButtonIndex = 0;

    bool BaseItemSet::m_IsForciblyFocusChanced = false;

    BaseItemSet* BaseItemSet::m_MoveForciblyFocusButtonSet = nullptr;

    BaseItemSet::BaseItemSet() NN_NOEXCEPT
    {
        m_ItemList.clear();
        m_Action.Reset();
        m_SelectedId = 0;
        m_FutureSelectId = 0;
        m_FutureFocusId = 0;
        m_Orientation = eOrientation_Horizontal;
        m_pParentButtonSet = nullptr;
        m_pPrevButtonSet = m_pNextButtonSet = nullptr;

        m_SelectedIds.clear();

        m_Option = (LayoutOption::Visible::Mask | LayoutOption::OnOffMode::Mask);
        m_State = LayoutState::Enable::Mask;

        m_SelectButton.Reset();
        m_SelectButton = (nn::hid::NpadButton::A::Mask);
        m_CancelButton.Reset();
        m_CancelButton = (nn::hid::NpadButton::B::Mask);

        m_UpButton.Reset();
        m_UpButton = (nn::hid::NpadButton::Up::Mask | nn::hid::NpadButton::StickLUp::Mask | nn::hid::NpadButton::StickRUp::Mask);
        m_DownButton.Reset();
        m_DownButton = (nn::hid::NpadButton::Down::Mask | nn::hid::NpadButton::StickLDown::Mask | nn::hid::NpadButton::StickRDown::Mask);
        m_LeftButton.Reset();
        m_LeftButton = (nn::hid::NpadButton::Left::Mask | nn::hid::NpadButton::StickLLeft::Mask | nn::hid::NpadButton::StickRLeft::Mask);
        m_RightButton.Reset();
        m_RightButton = (nn::hid::NpadButton::Right::Mask | nn::hid::NpadButton::StickLRight::Mask | nn::hid::NpadButton::StickRRight::Mask);
    }

    BaseItemSet::~BaseItemSet() NN_NOEXCEPT
    {

    }

    void BaseItemSet::UpdateInput() NN_NOEXCEPT
    {
        /*
         *  入力処理の優先順位は下記の通りです
         *
         *  0. 最優先：関数呼び出し・及び優先指定有り (Priority のフラグが建っている場合)
         *
         *  1. 　優先：タッチパネル
         *
         *  2. 　標準：コントローラ入力
         *
         *  優先順位が低い入力情報は無視されます
         */

        // 優先設定がされていない場合、タッチ入力が存在するか確認を行います
        if (!m_Action.Test<Action::Priority>())
        {
            UpdateTouch();
        }

        if (!m_Action.Test<Action::Touch>() && !m_Action.Test<Action::Priority>())
        {
            UpdateController();
        }

        if (m_Action.Test<Action::Priority>())
        {
            // 優先設定がある場合タッチのタッチとコントローラのフラグは下ろす
            m_Action.Set<Action::Touch>(false);
            m_Action.Set<Action::Gamepad>(false);
        }
        else if (m_Action.Test<Action::Touch>())
        {
            // タッチ入力を検知した場合、コントローラのフラグを下ろす
            m_Action.Set<Action::Gamepad>(false);
        }
    }

    void BaseItemSet::UpdateTouch() NN_NOEXCEPT
    {
        if (gTouch.IsEnable(nn::util::MakeFloat2(0, 0), nn::util::MakeFloat2(1280, 720)))
        {
            for (
                std::vector<BaseItem*>::iterator it = m_ItemList.begin();
                it != m_ItemList.end();
                ++it
                )
            {
                if (((*it) != nullptr))
                {
                    if ((*it)->GetChooseMode().Test(nns::hidfw::layout::BaseItem::ChoiceMode::Touch::Index))
                    {
                        if (gTouch.IsEnable((*it)->GetPos(), (*it)->GetSize()))
                        {
                            auto index = static_cast<int>(std::distance(m_ItemList.begin(), it));

                            if (m_State.Test<LayoutState::OnFocus>())
                            {
                                m_FutureSelectId = index;
                                m_Action.Set<Action::Touch>();
                                if (gTouch.IsTap((*it)->GetPos(), (*it)->GetSize()))
                                {
                                    m_Action.Set<Action::Select>();
                                    break;
                                }
                            }
                            else if ((*it)->IsForciblyFocus())
                            {
                                /*
                                *フォーカスが合っていないアイテムリストがタッチされた場合
                                * 許可されているのであれば移動を試みます
                                */
                                m_MoveForciblyFocusButtonSet = this;                            // 移動先(自分)
                                m_IsForciblyFocusChanced = true;                                // 強引な(Next、Backでない)フォーカス移動が生じた
                                m_MoveForciblyFocusButtonIndex = index;   // フォーカス後のアイテムリストインデックス
                                m_Action.Set<Action::Touch>();                                  // タッチ操作された
                                break;
                            }
                        }
                    }
                }
            }
        }
    }

    void BaseItemSet::UpdateController() NN_NOEXCEPT
    {
        for (
            std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
            it != gController.GetConnectedControllerList().end();
            ++it
            )
        {
            // 項目の選択・移動予約が既になされている場合は処理を中断
            if (m_Action.Test<Action::Gamepad>())
            {
                break;
            }
            /*
            *  操作プレイヤーの優先順位は プレイヤー番号が若い順です
            *  入力値の優先順位は キャンセル → 選択 → 項目移動
            *  上記の優先度で入力のあったアクションを実行予約する
            */
            if ((*it)->IsTrigger(nn::hid::NpadButton::B::Mask))
            {
                m_Action.Set<Action::Cancel>();
            }
            else if ((*it)->IsTrigger(nn::hid::NpadButton::A::Mask))
            {
                m_Action.Set<Action::Select>();
            }
            else
            {
                // 選択移動
                bool isNextButton = (*it)->IsRepeat((m_Orientation == eOrientation_Horizontal) ? m_LeftButton : m_UpButton);
                bool isBackButton = !(*it)->IsHold(((m_Orientation == eOrientation_Horizontal) ? m_LeftButton : m_UpButton)) && (*it)->IsRepeat((m_Orientation == eOrientation_Horizontal) ? m_RightButton : m_DownButton);
                bool isMoveButtonSet = ((*it)->GetButtonSet() & ((m_Orientation == eOrientation_Horizontal) ? (m_LeftButton | m_RightButton) : (m_UpButton | m_DownButton))).IsAllOff();

                if (isNextButton)
                {
                    m_Action.Set<Action::Back>(true);
                }
                else if (isBackButton)
                {
                    m_Action.Set<Action::Next>(true);
                }
                else if (isMoveButtonSet)
                {
                    isNextButton = (*it)->IsRepeat((m_Orientation == eOrientation_Horizontal) ? m_UpButton : m_LeftButton);
                    isBackButton = !(*it)->IsHold(((m_Orientation == eOrientation_Horizontal) ? m_UpButton : m_LeftButton)) && (*it)->IsRepeat((m_Orientation == eOrientation_Horizontal) ? m_DownButton : m_RightButton);

                    if (isNextButton)
                    {
                        if (m_pPrevButtonSet != nullptr)
                        {
                            m_pChangeItemSet = m_pPrevButtonSet;
                            m_Action.Set<Action::ChangeItemSet>(true);
                        }
                    }
                    else if (isBackButton)
                    {
                        if (m_pNextButtonSet != nullptr)
                        {
                            m_pChangeItemSet = m_pNextButtonSet;
                            m_Action.Set<Action::ChangeItemSet>(true);
                        }
                    }
                }
            }

            // 上下左右もしくは A B ボタンが押下げられている場合、ボタン入力があったと判定します
            if ((*it)->IsHold(m_UpButton | m_DownButton | m_LeftButton | m_RightButton) |
                (*it)->IsHold(nn::hid::NpadButton::A::Mask | nn::hid::NpadButton::B::Mask))
            {
                m_Action.Set<Action::Gamepad>(true);
            }
            if (m_Option.Test<LayoutOption::ThroughSelect>() || m_ItemList.at(m_SelectedId)->IsThroughSelect())
            {
                m_Action.Set<Action::Select>(false);
            }
            if (m_Option.Test<LayoutOption::ThroughCancel>() || m_ItemList.at(m_SelectedId)->IsThroughCancel())
            {
                m_Action.Set<Action::Cancel>(false);
            }
            if (m_Option.Test<LayoutOption::ThroughChoose>() || m_ItemList.at(m_SelectedId)->IsThroughChoose())
            {
                m_Action.Set<Action::Back>(false);
                m_Action.Set<Action::Next>(false);
                m_Action.Set<Action::ChangeItemSet>(false);
                m_Action.Set<Action::ChangeItemSet>(false);
            }
        }
    }

    void BaseItemSet::ProcessActionCommand() NN_NOEXCEPT
    {
        // NoneAction が有効の場合、動作予約を完全に無視します
        if (!m_Action.Test<Action::NoneAction>())
        {
            if (m_State.Test<LayoutState::OnFocus>())
            {
                // フォーカスが合っているアイテムセットを処理中にフォーカス移動の通知を受け取った場合
                if (m_IsForciblyFocusChanced)
                {
                    // 他のアイテムリストへ強制移動が発生した
                    m_pChangeItemSet = m_MoveForciblyFocusButtonSet;
                    m_FutureSelectId = m_MoveForciblyFocusButtonIndex;
                    m_Action.Set<Action::Priority>(true);           // 優先して処理します
                    m_Action.Set<Action::ChangeItemSet>(true);      // アイテムセットが変更されます
                    m_Action.Set<Action::ChangeFocusItem>(true);    // フォーカスが変更されます

                    m_Action.Set<Action::Next>(false);              // 項目移動を無視します
                    m_Action.Set<Action::Back>(false);
                    m_Action.Set<Action::Select>(false);            // 項目選択を無視します
                    m_Action.Set<Action::Cancel>(false);
                }
            }

            // フォーカスが合っていない、表示されていないアイテムリストは基本的に操作を受け付けません
            // ただじ優先フラグが建っている場合は通過します
            if (
                (m_State.Test<LayoutState::OnFocus>() && m_Option.Test<LayoutOption::Visible>()) ||
                 m_Action.Test<Action::Priority>()
                )
            {
                // ====================================================
                // コマンド処理
                // ====================================================
                if (m_Action.Test<Action::Next>() && !m_Option.Test<LayoutOption::ThroughChoose>())
                {
                    auto nextIndex = (m_FutureSelectId + 1) % GetListSize();
                    for (;
                        nextIndex != m_FutureSelectId;
                        nextIndex = (nextIndex + 1) % GetListSize()
                        )
                    {
                        if (m_ItemList.at(nextIndex)->GetChooseMode().Test<BaseItem::ChoiceMode::Controller>())
                        {
                            break;
                        }
                    }
                    m_FutureSelectId = nextIndex;
                }
                else if (m_Action.Test<Action::Back>() && !m_Option.Test<LayoutOption::ThroughChoose>())
                {
                    auto backIndex = (m_FutureSelectId + GetListSize() - 1) % GetListSize();
                    for ( ; backIndex != m_FutureSelectId; backIndex = (backIndex + GetListSize() - 1) % GetListSize() )
                    {
                        if (m_ItemList.at(backIndex)->GetChooseMode().Test<BaseItem::ChoiceMode::Controller>())
                        {
                            break;
                        }
                    }
                    m_FutureSelectId = backIndex;
                }
                else if (m_Action.Test<Action::Select>() &&  !m_Option.Test<LayoutOption::ThroughSelect>())
                {
                    NN_ASSERT_NOT_NULL(m_ItemList.at(m_SelectedId));
                    NN_ASSERT_NOT_NULL(m_ItemList.at(m_FutureSelectId));

                    if (m_ItemList.at(m_FutureSelectId)->IsMultiMode())
                    {
                        m_ItemList.at(m_FutureSelectId)->GetLayoutState().Test<LayoutState::Selected>() ? m_ItemList.at(m_FutureSelectId)->Cancel() : m_ItemList.at(m_FutureSelectId)->Select(m_Action.Test<Action::CallFunc>());
                    }
                    else
                    {
                        for (const auto& itr : m_SelectedIds)
                        {
                            if (!m_ItemList.at(itr)->IsMultiMode())
                            {
                                m_ItemList.at(itr)->Cancel();
                            }
                        }
                        m_ItemList.at(m_FutureSelectId)->Select(m_Action.Test<Action::CallFunc>());
                    }
                    m_SelectedId = m_FutureSelectId;
                }
                else if (m_Action.Test<Action::Cancel>() && !m_Option.Test<LayoutOption::ThroughCancel>())
                {
                    NN_ASSERT_NOT_NULL(m_ItemList.at(m_SelectedId));
                    NN_ASSERT_NOT_NULL(m_ItemList.at(m_FutureSelectId));

                    if (m_ItemList.at(m_FutureSelectId)->IsMultiMode())
                    {
                        m_ItemList.at(m_FutureSelectId)->Cancel();
                    }
                    else
                    {
                        for (const auto& itr : m_SelectedIds)
                        {
                            if (!m_ItemList.at(itr)->IsMultiMode())
                            {
                                m_ItemList.at(itr)->Cancel();
                            }
                        }
                    }
                }
                else if (
                    m_Action.Test<Action::ChangeItemSet>() &&
                    m_pChangeItemSet != nullptr
                    )
                {
                    if (m_IsForciblyFocusChanced)
                    {
                        m_IsForciblyFocusChanced = false;
                        m_pChangeItemSet->FocusItem(m_MoveForciblyFocusButtonIndex);
                    }
                    else
                    {
                        auto selectedItem = m_ItemList.at(std::min(m_FutureSelectId, static_cast<int>(m_ItemList.size()) - 1));
                        std::map<float, int> list;

                        nn::util::Float2 center = selectedItem->GetPos();
                        center.x += (selectedItem->GetSize()).x / 2.f;
                        center.y += (selectedItem->GetSize()).y / 2.f;

                        int index = 0;
                        for (auto itr : m_pChangeItemSet->GetItemSet())
                        {
                            if (!itr->GetChooseMode().Test<BaseItem::ChoiceMode::None>())
                            {
                                list.insert(std::make_pair(
                                    std::sqrtf(
                                        std::pow(center.x - (itr->GetPos().x + itr->GetSize().x / 2), 2) +
                                        std::pow(center.y - (itr->GetPos().y + itr->GetSize().y / 2), 2)),
                                    index
                                ));
                            }
                            ++index;
                        }
                        m_pChangeItemSet->FocusItem(list.size() > 0 ? list.begin()->second : 0);
                    }
                    m_pChangeItemSet->Focus(true);
                    this->Focus(false);
                    m_pChangeItemSet->ProcessActionCommand();
                }
            }
        }
        if (m_Action.Test<Action::ChangeFocusItem>())
        {
            m_FutureSelectId = m_FutureFocusId;
        }
    } // NOLINT(impl/function_size)

    void BaseItemSet::Update() NN_NOEXCEPT
    {

        m_pChangeItemSet = nullptr;

        // リストに何も登録されていない場合は一切合財の処理を行わず終了する
        if (GetListSize() <= 0)
        {
            m_Action.Reset();
            return;
        }

        // ====================================================
        // 入力
        // ====================================================

        UpdateInput();

        ProcessActionCommand();

        m_Action.Reset();
        m_Action.Set<Action::CallFunc>();

        // ====================================================
        // ボタンの更新及びテキストの描画処理
        // ====================================================
        for (std::vector<BaseItem*>::iterator it = m_ItemList.begin(); it < m_ItemList.end(); ++it)
        {
            auto index = static_cast<int>(std::distance(m_ItemList.begin(), it));
            (*it)->Hover((index == m_FutureSelectId) && m_State.Test<LayoutState::OnFocus>());
            (*it)->Update();
            if (m_Option.Test<LayoutOption::Visible>())
            {
                if ((*it) != nullptr)
                {
                    (*it)->PrintText();
                }
            }
        }

        m_SelectedIds.clear();
        for (size_t i = 0; i < m_ItemList.size(); ++i)
        {
            if (m_ItemList.at(i)->IsSelected())
            {
                m_SelectedIds.push_back(i);
            }
        }
    }

    void BaseItemSet::Draw() NN_NOEXCEPT
    {
        if (m_Option.Test<LayoutOption::Visible>())
        {
            for (std::vector<BaseItem*>::iterator it = m_ItemList.begin(); it < m_ItemList.end(); ++it)
            {
                if ((*it) != nullptr)
                {
                    (*it)->Draw();
                }
            }
        }
    }

    bool BaseItemSet::Select(int index, bool callFunc) NN_NOEXCEPT
    {
        if (index < GetListSize())
        {
            m_Action.Set<Action::Select>();
            m_Action.Set<Action::Priority>();
            m_Action.Set<Action::CallFunc>(callFunc);
            m_FutureSelectId = index;
            return true;
        }
        return false;
    }

    int BaseItemSet::GetFocusItemIndex() NN_NOEXCEPT
    {
        return m_FutureSelectId;
    }

    int BaseItemSet::GetSelectedItemIndex() NN_NOEXCEPT
    {
        return m_SelectedId;
    }

    bool BaseItemSet::GetSelectedItemIndex(int* pOutIndex) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutIndex);
        auto result = m_SelectedIds.size() > 0;
        if (result)
        {
            *pOutIndex = m_SelectedIds.front();
        }
        return result;
    }

    const std::vector<int>& BaseItemSet::GetSelectedItemIndices() const NN_NOEXCEPT
    {
        return m_SelectedIds;
    }

    void BaseItemSet::FocusItem(int index)
    {
        m_Action.Set(Action::ChangeFocusItem::Index, true);
        if (GetListSize() > 0)
        {
            m_FutureFocusId = index < GetListSize() ? index : GetListSize() - 1;
        }
    }

    void BaseItemSet::Focus(bool enable) NN_NOEXCEPT
    {
        m_State.Set(LayoutState::OnFocus::Index, enable);
        bool isChangeFocus = m_Action.Test<Action::ChangeFocusItem>();
        m_Action.Reset();
        m_Action.Set(Action::NoneAction::Index, true);
        if (isChangeFocus)
        {
            m_Action.Set(Action::ChangeFocusItem::Index, true);
        }
    }

    bool BaseItemSet::IsFocus() NN_NOEXCEPT
    {
        return m_State.Test<LayoutState::OnFocus>();
    }
}}}
