﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief       タッチパネル操作のための API の宣言
 */

#pragma once

#include <cmath>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_Gesture.h>

#include <nn/util/util_VectorApi.h>

namespace nns { namespace hidfw { namespace hid {

    /**
     * @brief       タッチパネル関連の設定、状態取得等を行うためのクラスです
     */
    class Touch
    {
        NN_DISALLOW_COPY(Touch);
        NN_DISALLOW_MOVE(Touch);
    public:
        //----------------------------------------------------------------
        //! @brief              タッチスクリーンの初期化を行います
        //----------------------------------------------------------------
        void Initialize() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              タッチスクリーンとジェスチャーの更新
        //----------------------------------------------------------------
        void Update() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief              インスタンスを取得します
        //----------------------------------------------------------------
        static Touch& GetInstance() NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  (TODO) 指定された範囲内で指が触れている常態か取得
        //! @param[in] pos          検索領域の左上スクリーン座標
        //! @param[in] size         検索領域のサイズ
        //! @return                 指がスクリーンに触れている場合 true
        //----------------------------------------------------------------
        bool IsEnable(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  指定された範囲内でタッチされたか取得
        //! @param[in] pos          検索領域の左上スクリーン座標
        //! @param[in] size         検索領域のサイズ
        //! @return                 タッチしている場合 true
        //----------------------------------------------------------------
        bool IsTouch(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  指定された範囲内でタップされたか取得
        //! @param[in] pos          検索領域の左上スクリーン座標
        //! @param[in] size         検索領域のサイズ
        //! @return                 タップしている場合 true
        //----------------------------------------------------------------
        bool IsTap(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  指定された範囲内でブレスされたか取得
        //! @param[in] pos          検索領域の左上スクリーン座標
        //! @param[in] size         検索領域のサイズ
        //! @return                 プレスしている場合 true
        //----------------------------------------------------------------
        bool IsPress(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  指定された範囲内のジェスチャーステータスを取得します
        //! @param[out] outState    ジェスチャーのステータス
        //! @param[in] pos          検索領域の左上スクリーン座標
        //! @param[in] size         検索領域のサイズ
        //! @return                 取得に成功した場合 true が返ります
        //----------------------------------------------------------------
        bool GetGestureState(nn::hid::GestureType* outType, nn::hid::GestureState* outState, const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  直近のジェスチャーステータスを取得します
        //! @param[out] outState    ジェスチャーのステータス
        //! @return                 取得に成功した場合 true が返ります
        //----------------------------------------------------------------
        bool GetGestureState(nn::hid::GestureState* outState) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  指定された範囲内のジェスチャーステータスを取得します
        //! @param[in] stateCount   ステータスの最大取得数
        //! @param[in] pos          検索領域の左上スクリーン座標
        //! @param[in] size         検索領域のサイズ
        //! @return                 取得したステータスの個数
        //----------------------------------------------------------------
        int GetGestureStates(nn::hid::GestureState* outState, int stateCount, const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

        //----------------------------------------------------------------
        //! @brief                  特定の点に一番近い位置のタッチ情報を取得します
        //! @param[out] outState    タッチステータス
        //! @param[in] nearPos      探索するスクリーン座標
        //! @return                 取得に成功した場合 true が返ります
        //----------------------------------------------------------------
        bool GetTouchState(nn::hid::TouchState* outState, const nn::util::Float2& nearPos) NN_NOEXCEPT;

    private:

        Touch() NN_NOEXCEPT;
        ~Touch() NN_NOEXCEPT;

    private:
        nn::hid::TouchScreenState<nn::hid::TouchStateCountMax>  m_TouchState[nn::hid::TouchScreenStateCountMax];    //!< 最新のタッチ情報
        int                                                     m_TouchStateCount;                                  //!< 取得できたタッチ情報の個数
        nn::hid::TouchScreenState<nn::hid::TouchStateCountMax>  m_oldTouchState;                                    //!< 一つ前のフレームのタッチ情報

        nn::hid::GestureState                                   m_GestureState[nn::hid::GestureStateCountMax];      //!< 最新のジェスチャー情報
        int                                                     m_GestureCount;                                     //!< 取得できたジェスチャー情報の個数
        nn::hid::GestureState                                   m_oldGestureState;                                  //!< 一つ前のフレームのジェスチャー情報
    };

}}}
