﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

nn::Result MountSdCard() NN_NOEXCEPT
{
    NN_LOG("Mount SD card as \"sd:/\"\n");

    auto result = nn::fs::MountSdCardForDebug("sd");
    if ( nn::fs::ResultSdCardAccessFailed::Includes(result) )
    {
        NN_LOG("SD card not found.");
    }

    return result;
}

nn::Result CreateLogFile() NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType directoryEntryType;

    auto result = nn::fs::GetEntryType(&directoryEntryType, "sd:/FuelGaugeDumper.txt");
    if ( nn::fs::ResultPathNotFound().Includes(result) )
    {
        NN_LOG("Create \"sd:/FuelGaugeDumper.txt\"\n");

        result = nn::fs::CreateFile("sd:/FuelGaugeDumper.txt", 0);
        if ( nn::fs::ResultPathNotFound::Includes(result) )
        {
            ;
        }
        else if ( nn::fs::ResultPathAlreadyExists::Includes(result)
            || nn::fs::ResultTargetLocked::Includes(result) )
        {
            ;
        }
        else if( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            NN_ABORT("Usable space not enough.\n");
        }
    }

    return result;
}

nn::Result OpenLogFile(nn::fs::FileHandle* pFileHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pFileHandle);
    NN_LOG("Open \"sd:/FuelGaugeDumper.txt\"\n");

    auto result = nn::fs::OpenFile(pFileHandle, "sd:/FuelGaugeDumper.txt", nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend);
    if ( nn::fs::ResultPathNotFound::Includes(result) )
    {
        ;
    }
    else if ( nn::fs::ResultTargetLocked::Includes(result) )
    {
        ;
    }

    return result;
}

void CloseLogFile(nn::fs::FileHandle* pFileHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pFileHandle);
    NN_LOG("Close \"sd:/FuelGaugeDumper.txt\"\n");
    nn::fs::CloseFile(*pFileHandle);
}

void UnmountSdCard() NN_NOEXCEPT
{
    NN_LOG("Unmount \"sd:/\"\n");
    nn::fs::Unmount("sd");
}

nn::Result AppendLog(nn::fs::FileHandle* pFileHandle, void* pWriteBuffer, size_t length) NN_NOEXCEPT
{
    NN_SDK_ASSERT(pFileHandle);
    NN_SDK_ASSERT(pWriteBuffer);
    NN_LOG("Write to \"sd:/FuelGaugeDumper.txt\"\n");

    int64_t fileSize = 0;
    (void)nn::fs::GetFileSize(&fileSize, *pFileHandle);

    auto result = nn::fs::WriteFile(*pFileHandle, fileSize, pWriteBuffer, length, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    if ( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
    {
        NN_ABORT("Usable space not enough.\n");
    }

    return result;
}
