﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/rid.h>
#include <nn/fs/fs_Debug.h>
#include <nn/rid/rid_Result.h>
#include <nn/util/util_ScopeExit.h>
#include "../DevQuestMenu_DebugSettings.h"
#include "../DevQuestMenu_Log.h"
#include "../DevQuestMenu_RapidJsonParser.h"
#include "../DevQuestMenu_RidBuffer.h"
#include "../DevQuestMenu_RidFileIo.h"
#include "DevQuestMenu_ApplicationUpdateModeScene.h"

namespace nn { namespace devquestmenu {

    namespace
    {
        //! ダウンロード状況のラベル
        static const std::map<nn::rid::ApplicationUpdateProgress::State, std::string> UpdateStateLabel =
        {
            { nn::rid::ApplicationUpdateProgress::State::DoNothing, "DoNothing" },
            { nn::rid::ApplicationUpdateProgress::State::Downloading, "Downloading" },
            { nn::rid::ApplicationUpdateProgress::State::Completed, "Completed" },
            { nn::rid::ApplicationUpdateProgress::State::Failed, "Failed" },
            { nn::rid::ApplicationUpdateProgress::State::Cancelled, "Cancelled" },
        };

        // サンプル用 ApplicationAsset を作成する
        void CreateSampleApplicationAsset(rid::ApplicationAsset outAssetList[], int assetListCount) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(assetListCount >= 4);

            outAssetList[0].index = 1;
            outAssetList[0].demoApplicationId = { 0x0100b15003a28000 };
            outAssetList[0].aocApplicationId = { 0x0100069000078000 };

            outAssetList[1].index = 4;
            outAssetList[1].demoApplicationId = { 0x0100b15003a28001 };
            outAssetList[1].aocApplicationId = { 0x0100069000078000 };

            outAssetList[2].index = 10;
            outAssetList[2].demoApplicationId = { 0x0100b15003a28002 };
            outAssetList[2].aocApplicationId = { 0x0100069000078000 };

            outAssetList[3].index = 30;
            outAssetList[3].demoApplicationId = { 0x0100b15003a28030 };
            outAssetList[3].aocApplicationId = { 0x0100069000078000 };
        }
    }

    ApplicationUpdateModeScene::ApplicationUpdateModeScene(RootSurfaceContext* pRootSurface) NN_NOEXCEPT
        : ModeSceneCommon("Application Update Mode", pRootSurface)
    {
        InitializeView();

        //! アプリケーション更新スレッドの生成
        static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[16 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_UpdateExecutionThread, [](void * p)
        {
            auto self = reinterpret_cast<ApplicationUpdateModeScene*>(p);

            const int MaxAssetListLength = 128;
            std::unique_ptr<rid::ApplicationAsset[]> pAssetList(new rid::ApplicationAsset[MaxAssetListLength]);
            int assetListCount;

            if (ReadUpdateApplicationListAndCreateAssetListForDebug(&assetListCount, pAssetList.get(), MaxAssetListLength))
            {
            }
            else
            {
                //! SD カードが挿入されていない場合、サンプルアセットを利用する
                QUESTMENU_LOG("Non SD Card. DevQuestMenu use a sample application asset.\n");
                assetListCount = 4;
                CreateSampleApplicationAsset(pAssetList.get(), assetListCount);
            }

            NN_ABORT_UNLESS(assetListCount != 0);
            NN_ABORT_UNLESS_NOT_NULL(pAssetList.get());

            rid::DownloadResult resultList[128 + 1];
            int resultCount;

            self->m_ApplicationUpdateResult = self->m_ApplicationUpdater.Execute(&resultCount, resultList, 128 + 1, pAssetList.get(), assetListCount, true);

            QUESTMENU_LOG("Application Update Result:\n");
            for (int i = 0; i < resultCount; i++)
            {
                if (resultList[i].result.IsSuccess())
                {
                    QUESTMENU_LOG("ApplicationId: %llx, Result: Success\n", resultList[i].applicationId);
                }
                else
                {
                    QUESTMENU_LOG("ApplicationId: %llx, Result: Failed(0x%08x)\n", resultList[i].applicationId, resultList[i].result.GetInnerValueForDebug());
                }
            }

            self->m_IsApplicationUpdateDone = true;
        }
        , this, s_Stack, sizeof(s_Stack), os::DefaultThreadPriority));

        //! アプリケーション更新中フラグを立てる
        m_pRootSurface->SetApplicationUpdating(true);

        //! アプリケーション更新開始
        nn::os::StartThread(&m_UpdateExecutionThread);
    };

    ApplicationUpdateModeScene::~ApplicationUpdateModeScene() NN_NOEXCEPT
    {
        //! 実行スレッドの破棄
        nn::os::WaitThread(&m_UpdateExecutionThread);
        nn::os::DestroyThread(&m_UpdateExecutionThread);
    }

    void ApplicationUpdateModeScene::OnLoop() NN_NOEXCEPT
    {
        rid::ApplicationUpdateProgress progress = m_ApplicationUpdater.GetProgress();
        m_ApplicationUpdateProgress.get()->setValue(static_cast<double>(progress.done) / static_cast<double>(progress.total));
        QUESTMENU_LOG("%s: %d / %d\n", UpdateStateLabel.at(progress.state).c_str(), static_cast<int>(progress.done), static_cast<int>(progress.total));

        //! アプリケーション更新中に試遊時間になった場合
        if (m_pRootSurface->GetDisplayModeTime().IsDuringDisplayModeTime())
        {
            m_ApplicationUpdater.Cancel();
        }
    };

    ModeType ApplicationUpdateModeScene::DetermineNextMode(const DisplayModeTime& displayModeTime) NN_NOEXCEPT
    {
        NN_UNUSED(displayModeTime);

        if (!m_IsApplicationUpdateDone)
        {
            return ModeType_NonChange;
        }
        else
        {
            //! 要コミット時は本体再起動する
            if (m_ApplicationUpdater.IsCommitRequired())
            {
                m_pRootSurface->SaveSettingsAndReboot(); //! 本体再起動
            }

            if (m_ApplicationUpdateResult.IsFailure())
            {
                QUESTMENU_LOG("Failed as result 0x%08x\n", m_ApplicationUpdateResult.GetInnerValueForDebug());
            }

            return ModeType_PrepareMode;
        }
    };

    ModeType ApplicationUpdateModeScene::GetSuccessNextMode() NN_NOEXCEPT
    {
        return ModeType_PrepareMode;
    };

    void ApplicationUpdateModeScene::InitializeView() NN_NOEXCEPT
    {
        //! 更新状態ラベル
        m_ApplicationUpdateStatus = MakeLabel("Application Update Start", m_StatusLabelSpec);

        //! 進捗バー
        std::unique_ptr<glv::Slider> slider(new glv::Slider(m_ProgressSliderRect));
        slider.get()->disable(glv::Property::FocusToTop | glv::Property::HitTest);
        *this << slider.get();
        m_ApplicationUpdateProgress = std::move(slider);
    };
}}
