﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <functional>
#include <nn/fs.h>
#include <nn/nn_BitTypes.h>
#include <nn/fs/fs_AddOnContent.h>
#include <nn/oe/oe_PowerStateControlApi.private.h>
#include <nn/time/time_ApiForMenu.h>
#include "DevQuestMenu_DebugSettings.h"
#include "DevQuestMenu_Log.h"
#include "DevQuestMenu_RapidJsonParser.h"
#include "DevQuestMenu_RidBuffer.h"
#include "DevQuestMenu_RidSaveDataIo.h"
#include "DevQuestMenu_RootSurface.h"
#include "ModeScene/DevQuestMenu_ApplicationUpdateModeScene.h"
#include "ModeScene/DevQuestMenu_DisplayModeScene.h"
#include "ModeScene/DevQuestMenu_MenuUpdateModeScene.h"
#include "ModeScene/DevQuestMenu_PrepareModeScene.h"
#include "ModeScene/DevQuestMenu_SleepModeScene.h"
#include "ModeScene/DevQuestMenu_SystemUpdateModeScene.h"
#include "ModeScene/DevQuestMenu_UpdateCheckModeScene.h"

namespace
{
    //! 本体設定セーブデータ生成
    const std::string CreateJsonSettingsSaveData(
        int8_t startHour, int8_t startMinute, int8_t endHour, int8_t endMinute, bool isMenuUpdateRequired, bool isApplicationUpdating, bool isMenuUpdateSkipped) NN_NOEXCEPT
    {
        auto toString = [](bool input)->std::string
        {
            return input ? "true" : "false";
        };

        std::string settingsSaveData =
            "{\"display_mode_start_hour\":\"" + std::to_string(startHour) + "\"," +
            "\"display_mode_start_minute\":\"" + std::to_string(startMinute) + "\"," +
            "\"display_mode_end_hour\":\"" + std::to_string(endHour) + "\"," +
            "\"display_mode_end_minute\":\"" + std::to_string(endMinute) + "\"," +
            "\"is_menu_update_required\":\"" + toString(isMenuUpdateRequired) + "\"," +
            "\"is_application_updating\":\"" + toString(isApplicationUpdating) + "\"," +
            "\"is_menu_update_skipped\":\"" + toString(isMenuUpdateSkipped) + "\"}";

        return settingsSaveData;
    }

    const std::string CreateJsonSettingsSaveData(
        const nn::devquestmenu::DisplayModeTime& displayModeTime, bool isMenuUpdateRequired, bool isApplicationUpdating, bool isMenuUpdateSkipped) NN_NOEXCEPT
    {
        return CreateJsonSettingsSaveData(
            displayModeTime.startTime.hour, displayModeTime.startTime.minute, displayModeTime.endTime.hour, displayModeTime.endTime.minute, isMenuUpdateRequired, isApplicationUpdating, isMenuUpdateSkipped);
    }
}

namespace nn { namespace devquestmenu {

    using namespace glv;

    /**
     * @brief       DevQuestMenu のエントリポイントです
     */
    void DevQuestMenuMain() NN_NOEXCEPT
    {
        //! GLV の共通カラースタイル定義
        glv::Style::standard().color.set(glv::StyleColor::BlackOnWhite);
        glv::Style::standard().color.fore.set(0.5);

        //! ユーザサーフェイスコンテキストの作成
        const int width = glv::glutGet(GLUT_SCREEN_WIDTH);
        const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

        RootSurfaceContext* context = new RootSurfaceContext(width, height);

        // メインループコールバックを登録.
        ApplicationFrameworkRegisterLoopCallback(context);

        // メインループ.
        glv::Application::run();

        // GLV依存のユーザコンテキストの解放.
        delete context;

        // エンジンコンテキストの解放処理タイミングを明示的に呼び出すこともできます.
        // 呼び出さない場合では静的インスタンスのデストラクションのタイミングで呼ばれます.
        glv::Application::quit();
    }

    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   width       幅
     * @param[in]   height      高
     */
    RootSurfaceContext::RootSurfaceContext(float width, float height) NN_NOEXCEPT
        : glv::Window(width, height, "DevQuestMenu"),
        glv::GLV(),
        m_DebugTab(),
        m_ElapsedTick(),
        m_DisplayWidth(width),
        m_DisplayHeight(height),
        m_DisplayGridWidth(width / 16),
        m_DisplayGridHeight(height / 18),
        m_DisplayModeTime(0, 0, 22, 0),
        m_IsMenuUpdateRequired(false),
        m_IsApplicationUpdating(false),
        m_IsMenuUpdateSkipped(false),
        m_DebugButtonRect(m_DisplayGridWidth * 13, m_DisplayGridHeight, m_DisplayGridWidth * 3, m_DisplayGridHeight * 3),
        m_MenuUpdateSkipButtonRect(m_DisplayGridWidth * 10, m_DisplayGridHeight * 14, m_DisplayGridWidth * 3, m_DisplayGridHeight * 2),
        m_HeaderSpec(glv::Place::TL, 0.0f, 0.0f, 25.f),
        m_CurrentTimeSpec(glv::Place::TL, 0, m_DisplayGridHeight * 6, 25.f),
        m_HeaderLabel(),
        m_StartTimeUnit(),
        m_EndTimeUnit()
    {
        //! シーンメニュー
        m_pModeScene = CreateScene<UpdateCheckModeScene>();

        //! デバッグメニュー
        InitializeDebugMenu(m_DisplayGridWidth, m_DisplayGridHeight);

        //! 設定読み込み
        InitializeMenuSettings();

        //! 描画
        this->setGLV(*this);
    }

    /**
     * @brief       デストラクタです。
     */
    RootSurfaceContext::~RootSurfaceContext() NN_NOEXCEPT
    {
    }

    /**
     * @brief       ランタイムエンジンにアタッチされた際に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     *
     * @see         glv::ApplicationLoopCallback::OnLoopAttached()
     */
    void RootSurfaceContext::OnLoopAttached(glv::ApplicationLoopContext& context) NN_NOEXCEPT
    {
        glv::ApplicationLoopCallback::OnLoopAttached(context);
    }

    /**
     * @brief       ランタイムエンジンからデタッチされた際に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     *
     * @see         glv::ApplicationLoopCallback::OnLoopDetached()
     */
    void RootSurfaceContext::OnLoopDetached(glv::ApplicationLoopContext& context) NN_NOEXCEPT
    {
        glv::ApplicationLoopCallback::OnLoopDetached(context);
    }

    /**
     * @brief       glv シーンレンダラ前に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     *
     * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
     */
    const glv::RequiredRestoration RootSurfaceContext::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(events);
        NN_UNUSED(context);

        // 戻り値は固定
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    /**
     * @brief       glv シーンレンダラ後に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     *
     * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
     */
    const glv::RequiredRestoration RootSurfaceContext::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(events);
        NN_UNUSED(context);

        // ループ毎の後処理
        HandleLoopPost(context, events);

        // HID イベント処理
        HandleHidEvent(context, events);

        // 戻り値は固定
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    bool RootSurfaceContext::onEvent(glv::Event::t events, glv::GLV& context) NN_NOEXCEPT
    {
        NN_UNUSED(events);
        NN_UNUSED(context);
        return false;
    }

    void RootSurfaceContext::onDraw(glv::GLV& context) NN_NOEXCEPT
    {
        glv::GLV::onDraw(context);
    }

    /**
     * @brief       デバッグメニューを配置します
     *
     */
    void RootSurfaceContext::InitializeDebugMenu(float gridWidth, float gridHeight) NN_NOEXCEPT
    {
        std::unique_ptr<glv::ViewContainer> pDebugTab(
            new glv::ViewContainer(glv::Rect(0, 0, gridWidth * 11, gridHeight * 18)));

        //! 強制モード遷移ボタン
        glv::Rect debugButtonRect = { gridWidth * 3, gridHeight, gridWidth * 2, gridHeight * 3 };
        m_TransitionButton = MakeLabelButton(pDebugTab.get(), "Force Mode\nChange", [&]
        {
            ChangeMode(m_pModeScene.get()->GetSuccessNextMode());
        }
        , debugButtonRect);

        //! メニュー更新モードスキップボタン
        glv::Rect menuUpdateSkipButtonRect = { gridWidth * 3, gridHeight * 14, gridWidth * 2, gridHeight * 3 };
        std::string initializeFlag = m_IsMenuUpdateSkipped ? "true" : "false";
        m_MenuUpdateSkipButton = MakeLabelButton("Menu Update\nSkip: " + initializeFlag, [this]
        {
            bool isMenuUpdateSkipped = IsMenuUpdateSkipped() ^ true;
            SetMenuUpdateSkipped(isMenuUpdateSkipped);
            std::string flag = isMenuUpdateSkipped ? "true" : "false";
            std::string caption = "Menu Update\nSkip:" + flag;
            GetMenuUpdateSkipButton()->ChangeCaption(caption.c_str());
        }
        , menuUpdateSkipButtonRect);
        *pDebugTab.get() << m_MenuUpdateSkipButton.get();

        //! 本体更新時間の調整ボタン
        glv::Rect startRect = { 0, gridHeight * 8.f, gridWidth * 3.f, gridHeight };
        m_StartTimeUnit = MakeTimeScheduleUnit(pDebugTab.get(), &(m_DisplayModeTime.startTime), "DisplayModeStart:", startRect);

        glv::Rect endRect = { 0, gridHeight * 13.f, gridWidth * 3.f, gridHeight };
        m_EndTimeUnit = MakeTimeScheduleUnit(pDebugTab.get(), &(m_DisplayModeTime.endTime), "DisplayModeEnd:", endRect);

        UpdateCurrentTime();

        //! ヘッダ
        m_HeaderLabel = MakeLabel(pDebugTab.get(), m_pModeScene.get()->GetCaption(), m_HeaderSpec);

        //! 各シーンごとのデバッグボタンを登録する
        m_pModeScene.get()->RegisterDebugButton(pDebugTab.get());

        *this << pDebugTab.get();
        m_DebugTab = std::move(pDebugTab);

        //! 初期フォーカスをセットする
        setFocus(m_MenuUpdateSkipButton.get());
    }

    void RootSurfaceContext::InitializeMenuSettings() NN_NOEXCEPT
    {
        const char* saveDataName = "ridSettingsSaveData.json";

        //! デフォルト設定の関数
        auto setDefaultSettingsSaveData = [](const char* saveDataName)->void
        {
            std::string defaultSettings = CreateJsonSettingsSaveData(0, 0, 22, 0, false, false, false);
            RidSaveDataIo::Write((nn::Bit8*)defaultSettings.c_str(), RidBuffer::BufferSize, saveDataName);
            NN_ABORT_UNLESS(RidSaveDataIo::Read(RidBuffer::s_Buffer, RidBuffer::BufferSize, saveDataName));
        };

        //! 設定セーブデータ読み込み
        if (!RidSaveDataIo::Read(RidBuffer::s_Buffer, RidBuffer::BufferSize, saveDataName))
        {
            setDefaultSettingsSaveData(saveDataName);
        }

        nne::rapidjson::Document document;
        document.Parse((const char*)RidBuffer::s_Buffer);

        //! 設定セーブデータが Parse できない場合、デフォルト設定を書きこみ直す
        if (document.HasParseError())
        {
            setDefaultSettingsSaveData(saveDataName);
            document.Parse((const char*)RidBuffer::s_Buffer);
            NN_ABORT_UNLESS(document.HasParseError());
        }
        std::string settingsJson;
        RapidJsonParser::CreatePrettyJsonText(&settingsJson, document);
        QUESTMENU_LOG("menu settings: %s\n", settingsJson.c_str());

        //! メンバ変数の初期化
        auto getIntValue = [](const char* key, const nne::rapidjson::Document& document)->int8_t
        {
            std::string value;
            RapidJsonParser::GetStringValue(&value, key, document);
            return std::stoi(value);
        };

        auto getBoolValue = [](const char* key, const nne::rapidjson::Document& document)->bool
        {
            std::string value;
            RapidJsonParser::GetStringValue(&value, key, document);
            return value == "true" ? true : false;
        };

        m_DisplayModeTime.startTime.hour = getIntValue("display_mode_start_hour", document);
        m_DisplayModeTime.startTime.minute = getIntValue("display_mode_start_minute", document);
        m_DisplayModeTime.endTime.hour = getIntValue("display_mode_end_hour", document);
        m_DisplayModeTime.endTime.minute = getIntValue("display_mode_end_minute", document);
        m_IsMenuUpdateRequired = getBoolValue("is_menu_update_required", document);
        m_IsApplicationUpdating = getBoolValue("is_application_updating", document);
        m_IsMenuUpdateSkipped = getBoolValue("is_menu_update_skipped", document);

        // SD カードに設定ファイルがあれば試遊モード時間を書き換える(簡易 QuestSettings 機能)
        if (ReadDisplayModeSettingsForDebug(&m_DisplayModeTime.startTime, &m_DisplayModeTime.endTime))
        {
            QUESTMENU_LOG("Display mode time settings was read from SD.\n", settingsJson.c_str());
        }
    }

    /**
     * @brief   試遊台メニューの設定を保存します
     *
     */
    void RootSurfaceContext::SaveSettings() NN_NOEXCEPT
    {
        std::string settingsSaveData = CreateJsonSettingsSaveData(m_DisplayModeTime, m_IsMenuUpdateRequired, m_IsApplicationUpdating, m_IsMenuUpdateSkipped);
        RidSaveDataIo::Write((nn::Bit8*) settingsSaveData.c_str(), RidBuffer::BufferSize, "ridSettingsSaveData.json");
    }

    /**
     * @brief   試遊台メニューの設定を保存して本体再起動します
     *
     */
    void RootSurfaceContext::SaveSettingsAndReboot() NN_NOEXCEPT
    {
        SaveSettings();
        nn::rid::Reboot();
    }

    /**
     * @brief       時間設定用の UI を作成します
     *
     */
    std::unique_ptr<TimeScheduleUnit> RootSurfaceContext::MakeTimeScheduleUnit(glv::ViewContainer* pContainer, nn::time::CalendarTime* pTime, const char* capture, const glv::Rect& r) NN_NOEXCEPT
    {
        std::unique_ptr<TimeScheduleUnit> pUnit(new TimeScheduleUnit());

        //! キャプチャ
        pUnit.get()->captureLabel = MakeLabel(pContainer, capture, glv::Label::Spec(glv::Place::TL, r.left(), r.top(), 25.f));

        //! 時刻
        pUnit.get()->hourLabel = MakeLabel(
            pContainer, NnTimeWrapper::ConvertToStringHour(*pTime), glv::Label::Spec(glv::Place::TL, r.left(), r.top() + r.height() * 2, 25.f));

        pUnit.get()->minuteLabel = MakeLabel(
            pContainer, ":" + NnTimeWrapper::ConvertToStringMinute(*pTime), glv::Label::Spec(glv::Place::TL, r.left() + r.width() / 2, r.top() + r.height() * 2, 25.f));

        auto* p = pUnit.get();

        //! Time Up ボタン
        pUnit.get()->hourUp = MakeLabelButton(pContainer, "Up", [&,p,pContainer,pTime,r]
        {
            NnTimeWrapper::AddHour(pTime, 1);
            p->hourLabel = MakeLabel(
                pContainer, NnTimeWrapper::ConvertToStringHour(*pTime), glv::Label::Spec(glv::Place::TL, r.left(), r.top() + r.height() * 2, 25.f));
        }
        , Rect(r.left(), r.top() + r.height(), r.width() / 2.f, r.height()));

        pUnit.get()->minuteUp = MakeLabelButton(pContainer, "Up", [&,p,pContainer,pTime,r]
        {
            NnTimeWrapper::AddMinute(pTime, 1);
            p->minuteLabel = MakeLabel(
                pContainer, ":" + NnTimeWrapper::ConvertToStringMinute(*pTime), glv::Label::Spec(glv::Place::TL, r.left() + r.width() / 2, r.top() + r.height() * 2, 25.f));
        }
        , Rect(r.left() + r.width() / 2.f, r.top() + r.height(), r.width() / 2.f, r.height()));

        //! Time Down ボタン
        pUnit.get()->hourDown = MakeLabelButton(pContainer, "Down", [&,p,pContainer,pTime,r]
        {
            NnTimeWrapper::AddHour(pTime, -1);
            p->hourLabel = MakeLabel(
                pContainer, NnTimeWrapper::ConvertToStringHour(*pTime), glv::Label::Spec(glv::Place::TL, r.left(), r.top() + r.height() * 2, 25.f));
        }
        , Rect(r.left(), r.top() + r.height() * 3, r.width() / 2.f, r.height()));

        pUnit.get()->minuteDown = MakeLabelButton(pContainer, "Down", [&,p,pContainer,pTime,r]
        {
            NnTimeWrapper::AddMinute(pTime, -1);
            p->minuteLabel = MakeLabel(
                pContainer, ":" + NnTimeWrapper::ConvertToStringMinute(*pTime), glv::Label::Spec(glv::Place::TL, r.left() + r.width() / 2, r.top() + r.height() * 2, 25.f));
        }
        , Rect(r.left() + r.width() / 2.f, r.top() + r.height() * 3, r.width() / 2.f, r.height()));

        return pUnit;
    }

    /**
     * @brief       ラベルを作成します
     *
     * @param[in]   label       表示
     * @param[in]   spec        位置
     */
    std::unique_ptr<glv::Label> RootSurfaceContext::MakeLabel(const std::string& label, const glv::Label::Spec& spec) NN_NOEXCEPT
    {
        std::unique_ptr<glv::Label> object(new glv::Label(label, spec));
        *this << object.get();
        return object;
    }

    /**
     * @brief       ラベルを作成します
     *
     * @param[in]   label       表示
     * @param[in]   spec        位置
     */
    std::unique_ptr<glv::Label> RootSurfaceContext::MakeLabel(glv::ViewContainer* pContainer, const std::string& label, const glv::Label::Spec& spec) NN_NOEXCEPT
    {
        std::unique_ptr<glv::Label> object(new glv::Label(label, spec));
        *pContainer << object.get();
        return object;
    }

    /**
     * @brief       ボタンを作成します
     *
     * @param[in]   label       表示
     * @param[in]   callback    実行される関数
     * @param[in]   r           位置・大きさ
     */
    std::unique_ptr<LabelButton> RootSurfaceContext::MakeLabelButton(const std::string& label, std::function< void() > callback, const glv::Rect& r) NN_NOEXCEPT
    {
        std::unique_ptr<LabelButton> object(new LabelButton(label.c_str(), callback, r));
        *this << object.get();
        return object;
    }

    /**
     * @brief       ボタンを作成します
     *
     * @param[in]   label       表示
     * @param[in]   callback    実行される関数
     * @param[in]   r           位置・大きさ
     */
    std::unique_ptr<LabelButton> RootSurfaceContext::MakeLabelButton(glv::ViewContainer* pContainer, const std::string& label, std::function< void() > callback, const glv::Rect& r) NN_NOEXCEPT
    {
        std::unique_ptr<LabelButton> object(new LabelButton(label.c_str(), callback, r));
        *pContainer << object.get();
        return object;
    }

    /**
     * @brief       現在時刻を更新します
     *
     */
    void RootSurfaceContext::UpdateCurrentTime() NN_NOEXCEPT
    {
        //! 時間ラベル
        nn::time::CalendarTime currentTime = NnTimeWrapper::GetCurrentTime();
        m_CurrentTimeLabel = MakeLabel(
            "Current Time: (UTC)\n" + NnTimeWrapper::ConvertToStringHour(currentTime) + ":" + NnTimeWrapper::ConvertToStringMinute(currentTime), m_CurrentTimeSpec);
    }

    /**
     * @brief       対象の動作モードのシーンを作成します
     *
     * @param[in]   width       画面の横幅
     * @param[in]   height      画面の縦幅
     */
    template<typename T>
    std::unique_ptr<ModeSceneCommon> RootSurfaceContext::CreateScene() NN_NOEXCEPT
    {
        std::unique_ptr<ModeSceneCommon> pScene(new T(this));
        *this << pScene.get();
        return pScene;
    }

    /**
     * @brief       対象の動作モードに遷移させます
     *
     * @param[in]   targetMode  遷移先の動作モード
     */
    void RootSurfaceContext::ChangeMode(ModeType targetType) NN_NOEXCEPT
    {
        switch (targetType)
        {
        case ModeType_DisplayMode:
            m_pModeScene = CreateScene<DisplayModeScene>();
            break;
        case ModeType_SystemUpdateMode:
            m_pModeScene = CreateScene<SystemUpdateModeScene>();
            break;
        case ModeType_MenuUpdateMode:

            if (!m_IsMenuUpdateSkipped)      //! メニュー更新を無視するかどうか
            {
                m_pModeScene = CreateScene<MenuUpdateModeScene>();
            }
            else
            {
                m_IsMenuUpdateRequired = false;     //! 要メニュー更新フラグを降ろす
                SaveSettings();                     //! セーブデータ保存
                m_pModeScene = CreateScene<ApplicationUpdateModeScene>();   //! Application 更新モードへ
            }

            break;
        case ModeType_ApplicationUpdateMode:
            m_pModeScene = CreateScene<ApplicationUpdateModeScene>();
            break;
        case ModeType_SleepMode:
            m_pModeScene = CreateScene<SleepModeScene>();
            break;
        case ModeType_PrepareMode:
            m_pModeScene = CreateScene<PrepareModeScene>();
            break;
        case ModeType_UpdateCheckMode:
            m_pModeScene = CreateScene<UpdateCheckModeScene>();
            break;
        case ModeType_NonChange:
            //! DoNothing
            return;
        default:
            NN_UNEXPECTED_DEFAULT;
            break;
        }

        InitializeDebugMenu(m_DisplayGridWidth, m_DisplayGridHeight);
    }

    /**
     * @brief       ループ毎の後処理を行います。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     */
    void RootSurfaceContext::HandleLoopPost(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);

        //! 1 秒ごとにループ処理を実行する
        static nn::os::Tick s_NextTick = nn::os::GetSystemTick();
        static const nn::os::Tick OneSecondTick = nn::os::ConvertToTick(nn::TimeSpan::FromSeconds(1));

        if (nn::os::GetSystemTick() < s_NextTick)
        {
            return;
        }
        s_NextTick += OneSecondTick;

        //! 現在時刻の更新
        UpdateCurrentTime();

        //! モードごとの定期処理
        m_pModeScene.get()->OnLoop();

        //! モード遷移判定
        ModeType type = m_pModeScene.get()->DetermineNextMode(m_DisplayModeTime);
        if (type != ModeType_NonChange)
        {
            ChangeMode(type);
        }
    }

    /**
     * @brief       HID イベントを処理します。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     */
    void RootSurfaceContext::HandleHidEvent(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }
}}
