﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv_resources.h>
#include <glv/detail/FsUtilities.h>
#include <glv/detail/glv_GfxTextApi.h>
#include <nn/fs.h>
#include <nn/account/account_Api.h>
#include <nn/bpc/bpc_WakeupConfig.h>
#include <nn/diag/diag_AbortObserverSystemApi.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/npns/npns_ApiSystem.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/psm/psm_Api.h>
#include <nn/rid.h>
#include <nn/util/util_Arithmetic.h>
#include <nn/util/util_CharacterEncoding.h>

#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
// For declearation of 'nv::InitializeGraphics' and 'nv::SetGraphicsAllocator'
#include <nv/nv_MemoryManagement.h>
#endif

#if NN_GFX_IS_TARGET_NVN
#include <nvn/nvn.h>
#include <nvn/nvn_FuncPtrInline.h>
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
// For declearation of 'glslcSetAllocator'
#include <nvnTool/nvnTool_GlslcInterface.h>
#endif
#endif

#include "DevQuestMenu_MenuApplicationId.h"
#include "DevQuestMenu_RootSurface.h"

namespace {

    const size_t HeapByteSize = 64 * 1024 * 1024;
    const size_t MallocBytesSize = 32 * 1024 * 1024;
    const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX ) || NN_GFX_IS_TARGET_NVN

    /**
     * @brief 共通アロケータ
     *
     */
    static void* Allocate(size_t size, size_t alignment, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    /**
     * @brief 共通デアロケータ
     *
     */
    static void Deallocate(void* addr, void* userPtr)
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    /**
     * @brief 共通リアロケータ
     *
     */
    static void* Reallocate(void* addr, size_t newSize, void* userPtr)
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif
#endif

    /**
     * @brief ペリフェラルセットアップ
     *
     */
    static void SetupPeripherals() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
#if defined( NN_BUILD_CONFIG_SPEC_NX )
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
        nv::InitializeGraphics(::malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
#endif
#if NN_GFX_IS_TARGET_NVN
        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
#endif
#endif
    }

    static const glv::InitialConfiguration LocalGfxConfiguration = glv::InitialConfiguration(512 * 1024, 1 * 1024 * 1024, 8);
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration(
        glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary
    );

extern "C" void nninitStartup() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::SetMemoryHeapSize(HeapByteSize));
    uintptr_t address;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateMemoryBlock(&address, MallocBytesSize));
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MallocBytesSize);
}

extern "C" void nnMain() NN_NOEXCEPT
{
    // ペリフェラルを初期化
    SetupPeripherals();

    // アボート時に Fatal 画面を出すために、システムプロセス用 AbortObserver を登録する
    nn::diag::InitializeSystemProcessAbortObserver();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
    nn::ns::Initialize();
    nn::psm::Initialize();  //! 電源ライブラリ
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForMenu()); //! 時間ライブラリ
    nn::bpc::InitializeWakeupConfig();
    nn::account::Initialize();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::npns::InitializeForSystem());
    nn::rid::Initialize(nn::devquestmenu::DevQuestMenuId);

    // ROM マウント
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&mountRomCacheBufferSize));
    std::unique_ptr<char[]> mountRomCacheBuffer(new char[mountRomCacheBufferSize]);
    NN_ABORT_UNLESS_NOT_NULL(mountRomCacheBuffer);
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::fs::MountRom("Contents", mountRomCacheBuffer.get(), mountRomCacheBufferSize)
    );

    // リソースプロバイダ及びリソースの初期化
    glv::Resources::GetSharedResourceContext().Initialize("Contents");

    // GLVランタイムエンジンの初期化及びアプリケーションフレームワークの初期化
    glv::ApplicationFrameworkInitialize(LocalHidConfiguration);

    // DevMenuエントリポイント呼び出し
    nn::devquestmenu::DevQuestMenuMain();

    nn::npns::FinalizeForSystem();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());
    nn::psm::Finalize();

    // フォントロード完了したのでマウント解除
    nn::fs::Unmount("Contents");
}
}
