﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "base/ui_MenuItemBase.h"
#include "base/ui_MenuItemAttributeTraceFocusState.h"
#include "base/ui_MenuItemBoxBase.h"
#include "base/ui_MenuItemStyleProperty.h"
#include "../panel/panel_PanelText.h"

namespace ui{

    class MenuItemScrollBoxStyle
        : public
            base::MenuItemStylePropertySizeStyle<base::StateNone,
            base::MenuItemStylePropertyAlignment<base::StateNone,
            base::MenuItemStylePropertyBackgroundStyle<base::StateNone,
            base::MenuItemStylePropertyMargin<base::StateNone,
            base::MenuItemStylePropertyHorizontalScrollbarStyle<base::StateNone,
            base::MenuItemStylePropertyVerticalScrollbarStyle<base::StateNone,
            base::MenuItemStylePropertyArrowButtonScrollDistance<base::StateNone,
            base::MenuItemStylePropertyAnalogStickScrollDistanceMax<base::StateNone,
            base::MenuItemStyleBase<MenuItemScrollBoxStyle>
            >>>>>>>>
    {};

    class MenuItemScrollBox
        : public  base::MenuItemAttributeTraceFocusState
                < base::MenuItemBase<IMenuItem, MenuItemScrollBox>
                >
    {
    private:
        struct InnerLayout
        {
            Position m_ItemPosition = {};
            Size m_Size = {};
        };

    public:
        explicit MenuItemScrollBox(const MenuItemScrollBoxStyle& style) NN_NOEXCEPT;

        virtual int GetChildCount() const NN_NOEXCEPT UI_FINAL
        {
            return m_pInnerItem ? 1 : 0;
        }

        virtual std::shared_ptr<IMenuItem> GetChild(int index) NN_NOEXCEPT UI_FINAL
        {
            if(index != 0)
            {
                return nullptr;
            }

            return m_pInnerItem;
        }

        virtual bool IsFocusAcceptable() const NN_NOEXCEPT UI_FINAL
        {
            return m_IsFocusAcceptable;
        }

        virtual bool IsTouchAcceptable() const NN_NOEXCEPT UI_FINAL
        {
            return true;
        }

        virtual void NotifyFocusedItemChanged(const MenuFocusedItemChangedContext& context) NN_NOEXCEPT UI_FINAL;

    public:
        virtual MenuButtonResult HandleUpButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleDownButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleLeftButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleRightButtonPressed(const MenuButtonHandleContext& context) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleRightAnalogStickInput(const MenuButtonHandleContext&, const nn::util::Vector3f& value) NN_NOEXCEPT UI_FINAL;

        virtual MenuButtonResult HandleTouchDown(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleTouchMove(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleTouchUp(const MenuButtonHandleContext& context, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL;
        virtual MenuButtonResult HandleTouchCanceled() NN_NOEXCEPT UI_FINAL;
    private:
        bool PassFocusToInnerItemImpl(const MenuButtonHandleContext& context) NN_NOEXCEPT;

    public:
        virtual void UpdateLayoutRecursively(const MenuLayoutUpdateContext& context) NN_NOEXCEPT UI_FINAL;
    private:
        static InnerLayout CalculateInnerLayoutImpl(
            const std::shared_ptr<const IMenuItem>& pInnerItem,
            const MenuItemScrollBoxStyle& style,
            const MenuLayoutUpdateContext& context
        ) NN_NOEXCEPT;

        static void CalculateViewAreaSizeImpl(
            Size* pOutOuterSize,
            Size* pOutViewAreaSize,
            const Size& innerLayoutSize,
            const MenuItemScrollBoxStyle& style
        ) NN_NOEXCEPT;

        static Position CalculateInnerPositionToFocusTarget(
            const Position& prevPosition,
            const Size& viewSize,
            const std::shared_ptr<const IMenuItem>& pSelf,
            const std::shared_ptr<const IMenuItem>& pTarget
        ) NN_NOEXCEPT;

        static Position CalculateInnerPositionImpl(
            const Position& prevPosition,
            const InnerLayout& innerLayout,
            const Size& viewSize,
            bool isHScrolling,
            bool isVScrolling,
            const MenuItemScrollBoxStyle& style,
            const MenuLayoutUpdateContext& context
        ) NN_NOEXCEPT;


    public:
        virtual std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT UI_FINAL
        {
            return m_pPanel;
        }

        virtual void UpdatePanel() NN_NOEXCEPT UI_FINAL;
    private:
        void UpdateHScrollbarPanelImpl() NN_NOEXCEPT;
        void UpdateVScrollbarPanelImpl() NN_NOEXCEPT;

    public:
        void RemoveInnerItem() NN_NOEXCEPT
        {
            if(m_pInnerItem)
            {
                m_pInnerItem->OnParentItemChanged(nullptr, 0);
            }
            m_pInnerItem.reset();
            m_pViewAreaPanel->ClearChildren();
        }

        std::shared_ptr<ConcreteType> SetInnerItem(const std::shared_ptr<IMenuItem>& item) NN_NOEXCEPT
        {
            // 以前のものを除外
            RemoveInnerItem();

            // 新しいものを設定
            if(item)
            {
                m_pInnerItem = item;
                item->OnParentItemChanged(this->shared_from_this(), 0);
                m_pViewAreaPanel->AddChild(item->GetPanel());
            }

            return this->shared_from_this();
        }

        std::shared_ptr<ConcreteType> SetPanelName(const std::string& value) NN_NOEXCEPT
        {
            m_pPanel->SetPanelName(value);
            return this->shared_from_this();
        }

    private:
        MenuItemScrollBoxStyle m_Style = {};
        std::shared_ptr<panel::PanelContainer> m_pPanel;
        std::shared_ptr<panel::PanelContainer> m_pViewAreaPanel;
        std::shared_ptr<panel::PanelText> m_pHorizontalBar;
        std::shared_ptr<panel::PanelText> m_pHorizontalKnob;
        std::shared_ptr<panel::PanelText> m_pVerticalBar;
        std::shared_ptr<panel::PanelText> m_pVerticalKnob;

        std::shared_ptr<IMenuItem> m_pInnerItem;

        // タッチ状態
        struct
        {
            bool m_IsHeld = false;
            Position m_PrevPosition = {};
        } m_TouchState;

        // 入力によるスクロール量
        // UpdateLayout で使用され、クリアされる。
        Position m_RequestedInnerPositionUpdate = {};

        // フォーカスの移動による表示位置の変更のためのターゲット
        // 移動が必要ない場合、 nullptr。
        // UpdateLayout で使用され、クリアされる。
        std::weak_ptr<const IMenuItem> m_pFocusTargetItem = {};

        // このアイテム自身がフォーカスを得るか。
        // Layout に依存するため UpdateLayout 中で変更する。
        //
        // 画面に表示されているアイテムしかフォーカスを受け取ることはできないため
        // InnerItem にフォーカスを受け取るものが含まれていない場合や、
        // フォーカスを受け取るものが内に存在しない場合ボタン操作で画面スクロールできなくなる。
        // この状況を回避するため、 InnerItem がフォーカスを受け取らない場合、このアイテム自身がフォーカスを受け取れるようにする。
        bool m_IsFocusAcceptable = true; // 初回の UpdateLayout 以前は LayoutSize == 0 なので InnerItem はフォーカスを受け取らない。

        // UpdateLayout() で更新する値。
        struct
        {
            Size m_ViewSize = {};
            InnerLayout m_InnerLayout = {};
            Position m_InnerPosition = {};
            bool m_IsHorizontallyScrolling = false;
            bool m_IsVerticallyScrolling = false;
        } m_Layout;
    };

}
