﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <functional>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "../ui_Macro.h"
#include "../ui_IMenuItem.h"

namespace ui{ namespace base{

    // タッチ操作に対してユーザーが設定したコールバックを呼び出す。
    template<typename TBaseType>
    class MenuItemAttributeTouchCallback
        : public TBaseType
    {
    public:
        typedef typename TBaseType::ConcreteType ConcreteType;

    public:
        virtual MenuButtonResult HandleTouched(const MenuButtonHandleContext&, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL
        {
            if(this->IsTouchAcceptable() && m_OnTouchedCallbackFunction)
            {
                m_OnTouchedCallbackFunction(position);
                return MenuButtonResult::GetHandled();
            }
            return MenuButtonResult::GetNotHandled();
        }

        virtual MenuButtonResult HandleTouchDown(const MenuButtonHandleContext&, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL
        {
            if(this->IsTouchAcceptable() && m_OnTouchDownCallbackFunction)
            {
                m_OnTouchDownCallbackFunction(position);
                return MenuButtonResult::GetHandled();
            }
            return MenuButtonResult::GetNotHandled();
        }

        virtual MenuButtonResult HandleTouchUp(const MenuButtonHandleContext&, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL
        {
            if(this->IsTouchAcceptable() && m_OnTouchUpCallbackFunction)
            {
                m_OnTouchUpCallbackFunction(position);
                return MenuButtonResult::GetHandled();
            }
            return MenuButtonResult::GetNotHandled();
        }

        virtual MenuButtonResult HandleTouchMove(const MenuButtonHandleContext&, const nn::util::Vector3f& position) NN_NOEXCEPT UI_FINAL
        {
            if(this->IsTouchAcceptable() && m_OnTouchMoveCallbackFunction)
            {
                m_OnTouchMoveCallbackFunction(position);
                return MenuButtonResult::GetHandled();
            }
            return MenuButtonResult::GetNotHandled();
        }

        virtual MenuButtonResult HandleTouchCanceled() NN_NOEXCEPT UI_FINAL
        {
            if(this->IsTouchAcceptable() && m_OnTouchCanceledCallbackFunction)
            {
                m_OnTouchCanceledCallbackFunction();
                return MenuButtonResult::GetHandled();
            }
            return MenuButtonResult::GetNotHandled();
        }

        std::shared_ptr<ConcreteType> SetTouchedCallbackFunction(const std::function<void (const nn::util::Vector3f&)>& value) NN_NOEXCEPT
        {
            m_OnTouchedCallbackFunction = value;
            return this->shared_from_this();
        }

        std::shared_ptr<ConcreteType> SetTouchHandlingCallbackFunctions(
            const std::function<void (const nn::util::Vector3f&)>& onTouchDown,
            const std::function<void (const nn::util::Vector3f&)>& onTouchUp,
            const std::function<void (const nn::util::Vector3f&)>& onTouchMove,
            const std::function<void ()>& onTouchCanceled
        ) NN_NOEXCEPT
        {
            m_OnTouchDownCallbackFunction = onTouchDown;
            m_OnTouchUpCallbackFunction = onTouchUp;
            m_OnTouchMoveCallbackFunction = onTouchMove;
            m_OnTouchCanceledCallbackFunction = onTouchCanceled;
            return this->shared_from_this();
        }

    protected:
        bool HasAnyTouchCallbackFunction() const NN_NOEXCEPT
        {
            return m_OnTouchedCallbackFunction
                || m_OnTouchDownCallbackFunction
                || m_OnTouchUpCallbackFunction
                || m_OnTouchMoveCallbackFunction
                || m_OnTouchCanceledCallbackFunction;
        }

    private:
        std::function<void (const nn::util::Vector3f&)> m_OnTouchedCallbackFunction;

        std::function<void (const nn::util::Vector3f&)> m_OnTouchDownCallbackFunction;
        std::function<void (const nn::util::Vector3f&)> m_OnTouchUpCallbackFunction;
        std::function<void (const nn::util::Vector3f&)> m_OnTouchMoveCallbackFunction;
        std::function<void ()> m_OnTouchCanceledCallbackFunction;

    };

}}
