﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

namespace ui{
    typedef uint64_t ItemStateFilterType;
}

namespace ui{ namespace base{

    enum ItemStateFlag : uint64_t
    {
        ItemStateFlag_None = 0llu,
        ItemStateFlag_All  = ~ItemStateFlag_None,

        // Enabled
        ItemStateFlag_Enabled   = 1 << 0,
        ItemStateFlag_Disabled  = 1 << 1,
        ItemStateFlag_EnablityMask = ItemStateFlag_Enabled | ItemStateFlag_Disabled,

        // Focus
        ItemStateFlag_OutFocus  = 1 << 2,
        ItemStateFlag_InFocus   = 1 << 3,
        ItemStateFlag_FocusMask = ItemStateFlag_InFocus | ItemStateFlag_OutFocus,

        // Selected
        ItemStateFlag_NotSelected  = 1 << 4,
        ItemStateFlag_Selected     = 1 << 5,
        ItemStateFlag_SelectionMask = ItemStateFlag_Selected | ItemStateFlag_NotSelected,
    };

    typedef uint64_t ItemStateFlagType;

    // static const int TState::StateCount;
    // static ItemStateFlagType TState::GetStateFlags(int state);
    // static State TState::MakeStateValue(...);
    template<typename T, typename TState>
    class ItemStatedValue
    {
    public:
        static const uint64_t StateCount = TState::StateCount;
        typedef TState StateType;

    public:
        template<typename ResultType>
        ResultType SetValue(ItemStateFilterType filter, const T& value, ResultType result) NN_NOEXCEPT
        {
            for(uint64_t state = 0; state < StateCount; state++)
            {
                auto flags = TState::GetStateFlags(state);
                if((filter & flags) == flags)
                {
                    m_Value[state] = value;
                }
            }
            return result;
        }

        template<typename ...Args>
        T GetValue(Args... args) const NN_NOEXCEPT
        {
            auto state = TState::MakeStateIndex(args...);
            if(state < 0 || state >= StateCount)
            {
                return {};
            }

            return m_Value[state];
        }

    private:
        T m_Value[StateCount] = {};
    };

    struct StateNone
    {
        static const uint64_t StateCount = 1;

        static ItemStateFlagType GetStateFlags(uint64_t) NN_NOEXCEPT
        {
            return ItemStateFlag_None;
        }

        static uint64_t MakeStateIndex() NN_NOEXCEPT
        {
            return 0;
        }
    };

    namespace detail
    {
        template<typename BaseType, ItemStateFlagType FlagT, ItemStateFlagType FlagF>
        struct BinaryStateVariation
        {
            static const int StateCount = 2 * BaseType::StateCount;

            static ItemStateFlagType GetStateFlags(uint64_t index) NN_NOEXCEPT
            {
                ItemStateFlagType flags = BaseType::GetStateFlags(index % BaseType::StateCount);
                switch(index / BaseType::StateCount)
                {
                case 0:
                    flags |= FlagF;
                    break;
                case 1:
                    flags |= FlagT;
                    break;
                default: NN_UNEXPECTED_DEFAULT;
                }
                return flags;
            }

            template<typename... Args>
            static uint64_t MakeStateIndex(bool value, Args... args) NN_NOEXCEPT
            {
                return (value ? BaseType::StateCount : 0) + BaseType::MakeStateIndex(args...);
            }

            static uint64_t MakeStateIndex(bool value) NN_NOEXCEPT
            {
                return (value ? BaseType::StateCount : 0);
            }

        };

        template<typename BaseType>
        struct StateVariationEnablity
            : BinaryStateVariation<BaseType, ItemStateFlag_Enabled, ItemStateFlag_Disabled>
        {};

        template<typename BaseType>
        struct StateVariationFocus
            : BinaryStateVariation<BaseType, ItemStateFlag_InFocus, ItemStateFlag_OutFocus>
        {};

        template<typename BaseType>
        struct StateVariationSelection
            : BinaryStateVariation<BaseType, ItemStateFlag_Selected, ItemStateFlag_NotSelected>
        {};
    }

    struct StateEnablity
        : detail::StateVariationEnablity<StateNone>
    {};

    struct StateEnablityFocus
        : detail::StateVariationEnablity
        < detail::StateVariationFocus<StateNone>
        >
    {};

    struct StateEnablityFocusSelection
        : detail::StateVariationEnablity
        < detail::StateVariationFocus
        < detail::StateVariationSelection
        < StateNone
        >>>
    {};

}}
