﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NotificationSlot.h"

#include "../../Config.h"

namespace scene{ namespace overlay{

    namespace{

        // postCategory の通知で preCategory の通知を上書きできるかを返す。
        bool CheckIsCategoryOverwritable(NotificationCategory postCategory, NotificationCategory preCategory) NN_NOEXCEPT
        {
            return true;
        }

        // postPriority の通知で prePriority の通知を上書きできるかを返す。
        // priority は値が小さい方が高優先度。
        bool CheckIsPriorityOverwritable(NotificationPriorityType postPriority, NotificationPriorityType prePriority) NN_NOEXCEPT
        {
            return postPriority <= prePriority;
        }

    }

    NotificationSlot::NotificationSlot() NN_NOEXCEPT
    {
        m_pPanelContainer = std::make_shared<panel::PanelContainer>();
        m_pPanelContainer->SetVisibility(panel::PanelVisibility::Transparent);
        m_pPanelContainer->SetSize(NotificationWidth, NotificationHeight);

        nn::os::InitializeTimerEvent(&m_ExpiringEvent, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_ExpiringEventHolder, &m_ExpiringEvent);
        m_IsExpiringEventLinked = false;
    }

    NotificationSlot::~NotificationSlot() NN_NOEXCEPT
    {
        ResetNotification();
        nn::os::FinalizeMultiWaitHolder(&m_ExpiringEventHolder);
        nn::os::FinalizeTimerEvent(&m_ExpiringEvent);
    }

    void NotificationSlot::ResetNotification() NN_NOEXCEPT
    {
        m_pNotification.reset();

        if(m_IsExpiringEventLinked)
        {
            nn::os::UnlinkMultiWaitHolder(&m_ExpiringEventHolder);
            nn::os::StopTimerEvent(&m_ExpiringEvent);
            nn::os::ClearTimerEvent(&m_ExpiringEvent);
            m_IsExpiringEventLinked = false;
        }
    }

    bool NotificationSlot::HasNotification() const NN_NOEXCEPT
    {
        return m_pNotification != nullptr;
    }

    void NotificationSlot::SetNotificationSlotIndex(int value) NN_NOEXCEPT
    {
        m_pPanelContainer->SetPosition(nn::util::Vector3f(0, static_cast<float>(NotificationHeight * value), 0));
    }

    bool NotificationSlot::SetNotification(
        const std::shared_ptr<INotification>& pNotification,
        nn::os::Tick startTick,
        nn::os::MultiWaitType* pMultiWait
        ) NN_NOEXCEPT
    {
        if(m_pNotification && pNotification)
        {
            auto preCategory = m_pNotification->GetCategory();
            auto postCategory = pNotification->GetCategory();
            if(!CheckIsCategoryOverwritable(postCategory, preCategory))
            {
                NN_DEVOVL_DISP_LOG_NMSG("new notification can not overwrite old one(category)\n");
                return false;
            }
            if(postCategory == preCategory)
            {
                auto prePriority = m_pNotification->GetPriority();
                auto postPriority = pNotification->GetPriority();
                if(!CheckIsPriorityOverwritable(postPriority, prePriority))
                {
                    NN_DEVOVL_DISP_LOG_NMSG("new notification can not overwrite old one(priority)\n");
                    return false;
                }
            }
        }

        ResetNotification();

        m_pNotification = pNotification;

        if(pNotification)
        {
            nn::TimeSpan time = (pNotification->GetTickToLive() - (nn::os::GetSystemTick() - startTick)).ToTimeSpan();
            if(time < 0)
            {
                time = 0; // 最低 1 フレームは表示する
            }
            nn::os::ClearTimerEvent(&m_ExpiringEvent); // 既にクリア済な気はするが、一応。
            nn::os::LinkMultiWaitHolder(pMultiWait, &m_ExpiringEventHolder);
            nn::os::StartOneShotTimerEvent(&m_ExpiringEvent, time);
            m_IsExpiringEventLinked = true;
        }

        return true;
    }

    void NotificationSlot::CheckExpired(nn::os::Tick currentTick) NN_NOEXCEPT
    {
        if(nn::os::TryWaitTimerEvent(&m_ExpiringEvent))
        {
            ResetNotification();
        }
    }

    void NotificationSlot::UpdatePanel() NN_NOEXCEPT
    {
        std::vector<std::shared_ptr<panel::IPanel>> list;

        if(auto p = m_pNotification)
        {
            list.push_back(p->GetPanel());
        }

        m_pPanelContainer->SetChildren(list);
    }

    std::shared_ptr<panel::IPanel> NotificationSlot::GetPanel() NN_NOEXCEPT
    {
        return m_pPanelContainer;
    }

}}
