﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "debug_PowerConsumptionMeter.h"

#include <nn/pcm/pcm.h>
#include <../../common-headers/core/nverror.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>

extern "C" int NvOsDrvOpen(const char *pathname);
extern "C" void NvOsDrvClose(int fd);
extern "C" NvError NvOsDrvIoctl(int fd, unsigned int iocode, void *arg, size_t);

namespace scene{ namespace debug{

    namespace {
        void EnsurePcmInitialized() NN_NOEXCEPT
        {
            static bool s_IsInitialized = false;
            if (!s_IsInitialized)
            {
                nn::pcm::Initialize();
                s_IsInitialized = true;
            }
        }

        bool CheckPcmSupport() NN_NOEXCEPT
        {
            return nn::pcm::IsSupported(nn::pcm::MeasuringPoint_Cpu);
        }

        int GetGpuCurrentPower() NN_NOEXCEPT
        {
            EnsurePcmInitialized();
            return nn::pcm::ReadCurrentPower(nn::pcm::MeasuringPoint_Gpu);
        }

        int GetCpuCurrentPower() NN_NOEXCEPT
        {
            EnsurePcmInitialized();
            return nn::pcm::ReadCurrentPower(nn::pcm::MeasuringPoint_Cpu);
        }

        int GetDdrCurrentPower() NN_NOEXCEPT
        {
            EnsurePcmInitialized();
            return nn::pcm::ReadCurrentPower(nn::pcm::MeasuringPoint_Ddr);
        }

        int GetVsysApCurrentPower() NN_NOEXCEPT
        {
            EnsurePcmInitialized();
            return nn::pcm::ReadCurrentPower(nn::pcm::MeasuringPoint_VsysAp);
        }

        int GetGpuLoad(int gpuCtrlFd) NN_NOEXCEPT
        {
            struct nvgpu_gpu_get_load_args gpuLoadArgs = { 0 };
            NvOsDrvIoctl(gpuCtrlFd, NVGPU_GPU_IOCTL_GET_LOAD, &gpuLoadArgs, sizeof(gpuLoadArgs));
            return gpuLoadArgs.busy_time;
        }

        void GetMemoryBandwidth(scene::debug::mcbwInfo& info, mc_stats& mc_stats) NN_NOEXCEPT
        {
            int i;
            uint64_t kbps[McbmEntryCount] = { 0 };

            const char *mcbwClientList[McbmEntryCount]{
                "readtally",
                "writetally",
            };

            for (i = 0; i < McbmEntryCount; i++)
                info.kbps[i] = 0;

            if (!info.initialize)
            {
                mc_stats.remove_all_cli();
                for (int i = 0; i < McbmEntryCount; i++) {
                    mc_stats.add_client(mcbwClientList[i], i);
                }
                info.initialize = true;
            }
            else
            {
                mc_stats.mc_stat_stop(&(kbps[0]), &(kbps[1]));
                for (i = 0; i < McbmEntryCount; i++)
                    info.kbps[i] = kbps[i];
            }

            mc_stats.mc_stat_start();
        }
    }

    PowerConsumptionMeter::PowerConsumptionMeter() NN_NOEXCEPT
    {
        m_pContainer = std::make_shared<panel::PanelContainer>();
        m_pPanelText = std::make_shared<panel::PanelText>();

        // サイズは debug_ActivityPowerConsumptionMeter.h にて (300,200) で固定
        m_pContainer->SetPanelName("powmtr");
        m_pContainer->SetVisibility(panel::PanelVisibility::Visible);

        m_pPanelText->SetVisibility(panel::PanelVisibility::Visible);
        m_pPanelText->SetColor({ 0, 0, 0, 0.5f });
        m_Graph->SetPosition(0, 0);
        m_pPanelText->SetSize(300, 24 * 5); // 描画したい行数分だけ
        m_pPanelText->SetTextColor({ 1, 1, 1, 1 });
        m_pPanelText->SetTextPosition(4, 0); // PanelText 内のローカル座標
        m_pPanelText->SetTextSize(13);  // FontSize

        int lineWidth = 2;
        // TORIAEZU: 固定値で設定
        m_Graph->SetPosition(0, 0);
        m_Graph->SetSize(300, 200);
        m_Graph->SetBackgroundColor(nn::util::Color4f(0, 0, 0, 0.5f));
        m_Graph->SetScrollSpeed(2);

        // Gpu
        m_Graph->SetValueRange(SeriesIndex_Gpu, 0, 4000);
        m_Graph->SetEntryLineColor(SeriesIndex_Gpu, nn::util::Color4f(0, 1, 0, 1));
        m_Graph->SetEntryLineWidth(SeriesIndex_Gpu, lineWidth);

        // Cpu
        m_Graph->SetEntryLineColor(SeriesIndex_Cpu, nn::util::Color4f(1, 0, 0, 1));
        m_Graph->SetValueRange(SeriesIndex_Cpu, 0, 4000);
        m_Graph->SetEntryLineWidth(SeriesIndex_Cpu, lineWidth);

        // Ddr
        m_Graph->SetEntryLineColor(SeriesIndex_Ddr, nn::util::Color4f(0, 0, 1, 1));
        m_Graph->SetValueRange(SeriesIndex_Ddr, 0, 4000);
        m_Graph->SetEntryLineWidth(SeriesIndex_Ddr, lineWidth);

        m_GpuCtrlFd = NvOsDrvOpen("/dev/nvhost-ctrl-gpu");

        m_pContainer->AddChild(m_Graph->GetPanel());
        m_pContainer->AddChild(m_pPanelText);

        m_FrameIndex = 0;
        m_ProductPower = m_BatteryLife = m_AvgGpuLoad = 0.0f;
        m_McbwInfo.initialize = false;
    }

    PowerConsumptionMeter::~PowerConsumptionMeter() NN_NOEXCEPT
    {
        m_McStats.remove_all_cli();
        NvOsDrvClose(m_GpuCtrlFd);
    }

    void PowerConsumptionMeter::SetPosition(int x, int y) NN_NOEXCEPT
    {
        m_pContainer->SetPosition(x, y);
    }

    void PowerConsumptionMeter::SetSize(int w, int h) NN_NOEXCEPT
    {
        m_pContainer->SetSize(w, h);
    }

    void PowerConsumptionMeter::Update() NN_NOEXCEPT
    {
        char buffer[256] = "????";

        int currentVsysAp = GetVsysApCurrentPower();

        if (m_FrameIndex % 10 == 0)
        {
            // 0.00003 * power^2 + 0.6793 * power + 26.391
            m_ProductPower = 0.00003 * currentVsysAp * currentVsysAp + 0.6793 * currentVsysAp + 26.391f;
            float capacity = (8.5f * m_ProductPower / 1000.0f * m_ProductPower / 1000.0f - 217.5f * m_ProductPower / 1000.0f + 5173.0f) * 3.6f / 1000.0f;
            float correctedCapacity = (1.0f - 0.0053f * (m_ProductPower / 1000.0f) * (m_ProductPower / 1000.0f) + 0.0718f * m_ProductPower / 1000.0f - 0.1238f) * capacity;
            correctedCapacity = correctedCapacity * 0.1f + correctedCapacity * 0.9f;
            m_BatteryLife = correctedCapacity / m_ProductPower * 1000.0f;

            int gpuLoad = GetGpuLoad(m_GpuCtrlFd);
            m_AvgGpuLoad = std::max(m_AvgGpuLoad * 0.9f + gpuLoad * 0.1f, static_cast<float>(gpuLoad));
            GetMemoryBandwidth(m_McbwInfo, m_McStats);

            if (CheckPcmSupport())
            {
                nn::util::SNPrintf(buffer, sizeof(buffer), "Power Consumption : %.2f mw.\nEstimate Battery Life : %.2f hour\nPower : CPU (Red), GPU (Green), DDR (Blue)\nGpu busy : %.2f %\nBandwidth Read %.2f GB/s, Write %.2f GB/s\n", m_ProductPower, m_BatteryLife, m_AvgGpuLoad * 0.1f, m_McbwInfo.kbps[0] / 1000000.0f, m_McbwInfo.kbps[1] / 1000000.0f);
            }
            else
            {
                nn::util::SNPrintf(buffer, sizeof(buffer), "Gpu busy : %.2f %\nBandwidth Read %.2f GB/s, Write %.2f GB/s\nPower value is available only on SDEV!\n", m_AvgGpuLoad * 0.1f, m_McbwInfo.kbps[0] / 1000000.0f, m_McbwInfo.kbps[1] / 1000000.0f);
            }
            m_pPanelText->SetText(buffer);
        }

        m_Graph->PushEntryValue(SeriesIndex_Gpu, GetGpuCurrentPower() / 1.5f);
        m_Graph->PushEntryValue(SeriesIndex_Cpu, GetCpuCurrentPower() / 1.5f);
        m_Graph->PushEntryValue(SeriesIndex_Ddr, GetDdrCurrentPower() / 1.5f);
        m_Graph->Update();

        m_FrameIndex++;
    }

    std::shared_ptr<panel::IPanel> PowerConsumptionMeter::GetPanel() NN_NOEXCEPT
    {
        return m_pContainer;
    }

}}
