﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os.h>
#include "debug_LogStorageLm.h"
#include "../../panel/panel_PanelContainer.h"
#include "../../panel/panel_PanelText.h"
#include "../../panel/panel_PanelRectangle.h"
#include "debug_ActivityLogViewer.h"

namespace scene{ namespace debug{

    namespace detail
    {
        class LogViewerRenderer
        {
        public:
            LogViewerRenderer() NN_NOEXCEPT;

            void Update(
                const panel::PanelRectangle& rect,
                const std::shared_ptr<ILogStorage>& pLogStorage,
                uint64_t begIdx,
                uint64_t endIdx,
                uint64_t storageBegIdx,
                uint64_t storageEndIdx,
                float fontSize,
                int lineHeight,
                bool isFitToBottom,
                bool isScrollbarVisible
            ) NN_NOEXCEPT;

            std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT;

        private:
            void UpdateLinePanelListImpl(
                std::vector<std::shared_ptr<panel::IPanel>>& outList,
                const panel::PanelRectangle& rect,
                const std::shared_ptr<ILogStorage>& pLogStorage,
                uint64_t begIdx,
                uint64_t endIdx,
                float fontSize,
                int lineHeight,
                bool isFitToBottom
            ) NN_NOEXCEPT;

            std::shared_ptr<panel::IPanel> AcquireLinePanelImpl(
                uint64_t lineIndex,
                const std::shared_ptr<ILogStorage>& pLogStorage,
                int lineWidth,
                int lineHeight,
                float fontSize
            ) NN_NOEXCEPT;

            void UpdateScrollbarImpl(
                std::vector<std::shared_ptr<panel::IPanel>>& outList,
                const panel::PanelRectangle& rect,
                uint64_t begIdx,
                uint64_t endIdx,
                uint64_t storageBegIdx,
                uint64_t storageEndIdx
            ) NN_NOEXCEPT;

        private:
            std::shared_ptr<panel::PanelContainer> m_pRootPanel;
            std::shared_ptr<panel::PanelContainer> m_pLinePanelContainer;

            std::vector<std::shared_ptr<panel::IPanel>> m_LinePanelList;
            uint64_t m_LineIndexBase = 0;

            std::shared_ptr<panel::PanelText> m_pScrollbarBackground;
            std::shared_ptr<panel::PanelText> m_pScrollbarKnob;

            nn::util::Color4f m_BackgroundColor = {0, 0, 0, .5f};
            nn::util::Color4f m_LogFontColor = {1, 1, 1, 1};

            int m_ScrollbarWidth = 16;
            nn::util::Color4f m_ScrollbarBackgroundColor = {0, 0, 0, .5f};
            nn::util::Color4f m_ScrollbarKnobColor = {.2f, .2f, 1, .75f};
        };
    }

    class LogViewerImpl
    {
    public:
        LogViewerImpl() NN_NOEXCEPT;
        ~LogViewerImpl() NN_NOEXCEPT;

        bool IsLogStorageAttached() NN_NOEXCEPT;
        void AttachLogStorage(const std::shared_ptr<ILogStorage>& pLogStorage) NN_NOEXCEPT;
        void DetachLogStorage() NN_NOEXCEPT;

        void SetPosition(int x, int y) NN_NOEXCEPT;
        void SetSize(int w, int h) NN_NOEXCEPT;

        void SetTextSize(float size) NN_NOEXCEPT;

        float GetCharacterWidth(const char* character, size_t length) NN_NOEXCEPT;
        int GetLineHeight() NN_NOEXCEPT;

        void ScrollUp(int scrollLineCount) NN_NOEXCEPT;
        void ScrollDown(int scrollLineCount) NN_NOEXCEPT;
        void MoveToBottom() NN_NOEXCEPT;

        void SetScrollbarVisibility(bool value) NN_NOEXCEPT;

        void Update() NN_NOEXCEPT;
        std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT;

    private:
        float GetCharacterWidthImpl(const char* character, size_t length) NN_NOEXCEPT;
        int GetLineHeightImpl() NN_NOEXCEPT;
        int GetRenderedLineCountMaxImpl() NN_NOEXCEPT;
        void UpdateAutoLineFeedImpl() NN_NOEXCEPT;

    private:
        nn::os::Mutex         m_Mutex;
        panel::PanelRectangle m_Rectangle = {};
        float                 m_TextSize  = 1;

        std::weak_ptr<ILogStorage> m_pLogStorage             = {};
        bool                       m_IsBoundToMostRecentLine = true;
        uint64_t                   m_LineIndex               = 0;

        bool m_IsScrollbarVisible = false;

        detail::LogViewerRenderer m_Renderer = {};
    };


    class LogViewer
    {
        NN_DISALLOW_COPY(LogViewer);
        NN_DISALLOW_MOVE(LogViewer);
    public:
        LogViewer() NN_NOEXCEPT
        {
        }

        ~LogViewer() NN_NOEXCEPT
        {
            m_pImpl.reset(); // AutoLineFeed の都合で先に Viewer を削除する
            m_pStorageLm.reset();
        }

        void Update() NN_NOEXCEPT
        {
            if(m_pStorageLm)
            {
                m_pStorageLm->Update();
            }

            if(m_pImpl)
            {
                m_pImpl->Update();
            }
        }

        std::shared_ptr<panel::IPanel> GetPanel() NN_NOEXCEPT
        {
            if(m_pImpl)
            {
                return m_pImpl->GetPanel();
            }
            return nullptr;
        }

        void SetActivity(const ActivityLogViewer& value) NN_NOEXCEPT
        {
            // LogStorage
            if(!m_pStorageLm)
            {
                m_pStorageLm = std::make_shared<LogStorageLm>();
                m_pStorageLm->Activate();
            }

            // LogViewer
            if(value.isBackground)
            {
                m_pImpl.reset();
            }
            else if(!m_pImpl)
            {
                m_pImpl = std::make_shared<LogViewerImpl>();
            }

            // Storage のアタッチ
            if(m_pImpl && m_pStorageLm)
            {
                if(!m_pImpl->IsLogStorageAttached())
                {
                    m_pImpl->AttachLogStorage(m_pStorageLm);
                }
            }

            // パラメータ設定
            if(m_pImpl)
            {
                m_pImpl->SetPosition(value.common.positionX, value.common.positionY);
                m_pImpl->SetSize(value.width, value.height);
                m_pImpl->SetTextSize(value.fontSize);
            }

            m_Activity = value;
        }

        void ScrollUp(int n) NN_NOEXCEPT
        {
            if(m_pImpl)
            {
                m_pImpl->ScrollUp(n);
            }
        }

        void ScrollDown(int n) NN_NOEXCEPT
        {
            if(m_pImpl)
            {
                m_pImpl->ScrollDown(n);
            }
        }

        void MoveToBottom() NN_NOEXCEPT
        {
            if(m_pImpl)
            {
                m_pImpl->MoveToBottom();
            }
        }

        void SetScrollbarVisibility(bool value) NN_NOEXCEPT
        {
            if(m_pImpl)
            {
                m_pImpl->SetScrollbarVisibility(value);
            }
        }

        int GetLineHeight() NN_NOEXCEPT
        {
            if(m_pImpl)
            {
                return m_pImpl->GetLineHeight();
            }
            return 1;
        }


    private:
        std::shared_ptr<LogViewerImpl> m_pImpl;
        std::shared_ptr<LogStorageLm> m_pStorageLm;
        ActivityLogViewer m_Activity = {};



    };

}}
