﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include "debug_LayoutEditor.h"

namespace scene{ namespace debug{

    class LayoutEditorController
    {
    public:
        enum Anchor
        {
            Anchor_None = 0,
            Anchor_TopLeft,
            Anchor_TopRight,
            Anchor_BottomLeft,
            Anchor_BottomRight,
        };

    public:
        explicit LayoutEditorController(const std::shared_ptr<LayoutEditor>& pEditor, LayoutEditor::ModuleHandle handle) NN_NOEXCEPT
            : m_pEditor(pEditor)
            , m_Handle(handle)
        {
            EstimatePositionAnchor();
        }

        ~LayoutEditorController() NN_NOEXCEPT
        {
            Deactivate();
        }

        ui::Size GetCanvasSize() const NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                return m_pEditor->GetCanvasSize();
            }
            return {};
        }

        ui::Size GetSize() const NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                return m_pEditor->GetModuleSize(m_Handle);
            }
            return {};
        }

        ui::Position GetPosition() const NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                return m_pEditor->GetModulePosition(m_Handle);
            }
            return {};
        }

        // 位置を指定します。
        // アンカーは解除されます。
        void SetPosition(const ui::Position& pos) NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                m_pEditor->SetModulePosition(m_Handle, pos);
            }
            m_PositionAnchor = Anchor_None;
        }

        // アンカーを設定します。
        // RefreshPositionToAnchor を呼び出すことで位置が移動します。
        void SetPositionAnchor(Anchor anchor) NN_NOEXCEPT
        {
            m_PositionAnchor = anchor;
        }

        // アンカーを推定します。
        // x, y 方向それぞれ独立に以下の優先順位で調べます。
        //   x: 左端が一致→右端が一致→アンカーなし
        //   y: 上端が一致→下端が一致→アンカーなし
        void EstimatePositionAnchor() NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                auto pos = m_pEditor->GetModulePosition(m_Handle);
                auto size = m_pEditor->GetModuleSize(m_Handle);
                auto csize = m_pEditor->GetCanvasSize();

                // x
                bool isLeft  = (pos.x == 0);
                bool isRight = (pos.x == csize.width - size.width);

                // y
                bool isTop    = (pos.y == 0);
                bool isBottom = (pos.y == csize.height - size.height);

                if(isTop && isLeft)
                {
                    m_PositionAnchor = Anchor_TopLeft;
                }
                else if(isTop && isRight)
                {
                    m_PositionAnchor = Anchor_TopRight;
                }
                else if(isBottom && isLeft)
                {
                    m_PositionAnchor = Anchor_BottomLeft;
                }
                else if(isBottom && isRight)
                {
                    m_PositionAnchor = Anchor_BottomRight;
                }
                else
                {
                    m_PositionAnchor = Anchor_None;
                }
            }
            else
            {
                m_PositionAnchor = Anchor_None;
            }
        }

        // アンカーで指定された位置に Position を変更します。
        // アンカーが解除されていた場合何もしません。
        void RefreshPositionToAnchor() NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                if(m_PositionAnchor == Anchor_TopLeft)
                {
                    m_pEditor->SetModulePosition(m_Handle, {0, 0});
                }
                else if(m_PositionAnchor == Anchor_TopRight)
                {
                    auto size = m_pEditor->GetModuleSize(m_Handle);
                    auto csize = m_pEditor->GetCanvasSize();
                    m_pEditor->SetModulePosition(m_Handle, {csize.width - size.width, 0});
                }
                else if(m_PositionAnchor == Anchor_BottomLeft)
                {
                    auto size = m_pEditor->GetModuleSize(m_Handle);
                    auto csize = m_pEditor->GetCanvasSize();
                    m_pEditor->SetModulePosition(m_Handle, {0, csize.height - size.height});
                }
                else if(m_PositionAnchor == Anchor_BottomRight)
                {
                    auto size = m_pEditor->GetModuleSize(m_Handle);
                    auto csize = m_pEditor->GetCanvasSize();
                    m_pEditor->SetModulePosition(m_Handle, {csize.width - size.width, csize.height - size.height});
                }
            }
        }

        void Move(const ui::Position& dpos) NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                m_pEditor->MoveModule(m_Handle, dpos);
            }
        }

        void Activate() NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                m_pEditor->SetModuleActivity(m_Handle, true);
            }
        }

        void Deactivate() NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                m_pEditor->SetModuleActivity(m_Handle, false);
            }
        }

        ui::Size UpdateLayout() NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                return m_pEditor->UpdateLayout();
            }
            return {};
        }

        void UpdatePanel() NN_NOEXCEPT
        {
            if(m_pEditor)
            {
                m_pEditor->UpdateInternalPanel();
            }
        }

    private:
        std::shared_ptr<LayoutEditor> m_pEditor;
        LayoutEditor::ModuleHandle m_Handle = {};
        Anchor m_PositionAnchor;
    };

}}
