﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "panel_PanelContainer.h"

#include <algorithm>
#include <nn/nn_Assert.h>

namespace panel {

    PanelContainer::PanelContainer()
        : PanelBase(PanelType_Container)
    {
    }

    PanelContainer::~PanelContainer()
    {
        ClearChildren();
    }

    std::shared_ptr<IPanelContainer> PanelContainer::CastToContainer() NN_NOEXCEPT
    {
        return this->shared_from_this();
    }

    int PanelContainer::GetChildrenCount() const NN_NOEXCEPT
    {
        return static_cast<int>(m_Children.size());
    }

    int PanelContainer::GetChildIndex(const std::shared_ptr<IPanel>& child) const NN_NOEXCEPT
    {
        for(size_t i = 0; i < m_Children.size(); i++)
        {
            if(m_Children[i] == child)
            {
                return static_cast<int>(i);
            }
        }
        return -1;
    }

    std::shared_ptr<IPanel> PanelContainer::GetChild(int index) NN_NOEXCEPT
    {
        if(index < 0 || index >= static_cast<int>(m_Children.size()))
        {
            return nullptr;
        }
        return m_Children[index];
    }

    void PanelContainer::ClearChildren() NN_NOEXCEPT
    {
        NN_DEVOVL_PANEL_LOG_PANEL("p%llu:ClearChildren()\n", GetPanelIdentifier());
        for(auto& p : m_Children)
        {
            p->NotifyParentChanged(nullptr);
        }
        m_Children.clear();
        InvalidateSelfImpl(); // 子のあった場所の描き直しが必要
    }

    void PanelContainer::SetChildren(const std::vector<std::shared_ptr<IPanel>>& list) NN_NOEXCEPT
    {
        // 完全に一致する場合無視
        if(m_Children == list)
        {
            return;
        }

        NN_DEVOVL_PANEL_LOG_PANEL("p%llu:SetChildren(n=%llu)\n", GetPanelIdentifier(), list.size());

        std::vector<std::shared_ptr<IPanel>> curChildren;
        curChildren.swap(m_Children);

        std::vector<std::shared_ptr<IPanel>> newChildren;
        newChildren.reserve(list.size());

        size_t keptCount = 0;
        size_t addedCount = 0;
        size_t removedCount = 0;
        for(size_t srcIdx = 0; srcIdx < list.size(); srcIdx++)
        {
            auto p = list[srcIdx];
            newChildren.push_back(p); // 新しいリストに追加

            auto curItr = std::find(curChildren.begin(), curChildren.end(), p);
            if(curItr != curChildren.end())
            {
                // 古いリスト上で nullptr にする
                curItr->reset();
                keptCount++;
            }
            else
            {
                // 親が変わったことを通知
                p->NotifyParentChanged(this->shared_from_this());
                addedCount++;
            }
        }

        // 削除された子に親が変わったことを通知
        for(auto& c : curChildren)
        {
            if(c)
            {
                c->NotifyParentChanged(nullptr);
                removedCount++;
            }
        }

        NN_ASSERT_EQUAL(newChildren.size(), list.size());
        NN_ASSERT_EQUAL(keptCount + removedCount, curChildren.size());
        NN_ASSERT_EQUAL(keptCount + addedCount, list.size());

        m_Children.swap(newChildren);

        InvalidateSelfImpl(); // 子の順番が変わっているかもしれない
    }

    void PanelContainer::AddChild(const std::shared_ptr<IPanel>& child) NN_NOEXCEPT
    {
        if(child == nullptr)
        {
            return;
        }

        NN_DEVOVL_PANEL_LOG_PANEL("p%llu:AddChild(id=%llu)\n", GetPanelIdentifier(), child->GetPanelIdentifier());
        NN_ASSERT_EQUAL(nullptr, child->GetParent());
        m_Children.push_back(child);
        child->NotifyParentChanged(this->shared_from_this());
        InvalidateSelfImpl(); // 増やす時には要らないかもしれない
    }

    void PanelContainer::RemoveChild(const std::shared_ptr<IPanel>& child) NN_NOEXCEPT
    {
        if(child == nullptr)
        {
            return;
        }

        if(child->GetParent() != this->shared_from_this())
        {
            return;
        }

        NN_DEVOVL_PANEL_LOG_PANEL("p%llu:RemoveChild(id=%llu)\n", GetPanelIdentifier(), child->GetPanelIdentifier());
        auto it = std::remove_if(m_Children.begin(), m_Children.end(), [&child](const std::shared_ptr<IPanel>& p){ return p == child; });
        m_Children.resize(it - m_Children.begin());
        child->NotifyParentChanged(nullptr);
        InvalidateSelfImpl(); // 子のあった場所の描き直しが必要
    }

    bool PanelContainer::CheckRedrawRequired() const NN_NOEXCEPT
    {
        if(this->BaseType::CheckRedrawRequired())
        {
            return true;
        }

        for(auto& c : m_Children)
        {
            if(c->CheckRedrawRequired())
            {
                return true;
            }
        }

        return false;
    }

    void PanelContainer::InvalidateRecursiveImpl() NN_NOEXCEPT
    {
        InvalidateSelfImpl();
        for(auto& c : m_Children)
        {
            c->RequestRedraw();
        }
    }
}
